# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['read_data', 'save_data', 'split_fires', 'to_polygon', 'run_all']

# Cell
from IPython.core.debugger import set_trace
import rasterio
import rasterio.features
from rasterio.crs import CRS
from shapely.geometry import MultiPolygon
import geopandas as gpd
import numpy as np
from tqdm import tqdm
import scipy.ndimage as ndimage
import matplotlib.pyplot as plt
from pathlib import Path
import pandas as pd

# Cell
def read_data(filename):
    data = rasterio.open(filename)
    meta = data.meta
    crs  = data.crs
    transform = data.transform
    data = data.read().squeeze()
    data[data<0] = 0
    return data, meta, crs, transform

def save_data(filename, raster, meta=None, crs=None, transform=None):
    if meta is None:
        assert crs is not None
        assert transform is not None
        shp = raster.shape
        if len(shp) == 2:
            count = 1
            height, width = shp
        elif len(shp) == 3:
            count, height, width = shp
        else: raise NotImplementedError
        meta = {'driver': 'GTiff', 'dtype': 'uint32',
                'nodata': 0, 'width': width, 'height': height,
                'count': count, 'crs': crs,
                'transform': transform}
    else:
        meta['nodata'] = 0
        meta['dtype'] = 'uint32'
    meta['compress'] = 'lzw'
    with rasterio.open(filename, 'w', **meta) as dst:
        if len(raster.shape) == 2:
            raster = raster[None]
        dst.write(raster.astype('uint32'))

def split_fires(data, interval_days=4, interval_pixels=4, min_size_pixels=1,
                min_daily_size=10):
    data = data.astype(np.uint32)
    kernel = np.ones((interval_pixels,interval_pixels))
    data_conv = ndimage.convolve(data>0, kernel, mode='constant', cval=0.0)>0

    # Initial labels with no temporal separation
    labels, nlabels = ndimage.label(data_conv)
    labels[data == 0] = 0
    del data_conv

    # Remove fires with less than min_size_pixels
    classes = np.arange(0, np.max(labels)+1, 1)
    ba_size_pixels = np.bincount(labels.reshape(-1))
    labels[np.isin(labels,classes[ba_size_pixels < min_size_pixels])] = 0

    # Calculate unique (data, label) pairs ignoring no fire pixels
    data_aux = data[labels>0]
    labels_aux = labels[labels>0]
    #unique_pairs = np.unique(np.array((data_aux, labels_aux), dtype=np.uint32).reshape(2,-1), axis=1)
    unique_labels = np.unique(labels_aux)
    unique_dates = []
    for i in unique_labels:
        a = np.argwhere(np.bincount(data[labels==i].reshape(-1)) >= min_daily_size).reshape(-1)
        unique_dates.append(a)
    del data_aux, labels_aux

    # Find shapes that need time separation
    df = pd.DataFrame({'date': unique_dates, 'label': unique_labels})
    try:
        df_overlap = df.groupby('label').agg(lambda x : len(x.values[0]))
    except:
        df_overlap = df.groupby('label').agg(lambda x : len(x[0]))
    overlap_index = df_overlap.loc[df_overlap.date>1].index
    #df = df.groupby('label').agg(lambda x : x.values.tolist())
    df = df.loc[df.label.isin(overlap_index)]
    df['splits'] = [np.diff(df.iloc[i].date) > interval_days for i in range(len(df))]

    for k in tqdm(range(len(df))):
        df_row = df.iloc[k]
        for i, s in enumerate(df_row.splits):
            if s:
                split1 = df_row.date[i]
                I1 = (labels==df_row.label) & (data<=split1)
                I1_conv = ndimage.convolve(I1, kernel, mode='constant', cval=0.0)>0
                l1, _ = ndimage.label(I1_conv)
                l1[I1==0] = 0
                l1[l1>0] += labels.max()
                labels[I1] = l1[I1]
                if i == np.where(df_row.splits)[0][-1]:
                    I1 = (labels==df_row.label) & (data>split1)
                    I1_conv = ndimage.convolve(I1, kernel, mode='constant', cval=0.0)>0
                    l1, _ = ndimage.label(I1_conv)
                    l1[I1==0] = 0
                    l1[l1>0] += labels.max()
                    labels[I1] = l1[I1]

    # Relabel
    _, labels = np.unique(labels,return_inverse=1)
    labels = labels.reshape(data.shape)

    # Remove fires with less than min_size_pixels
    classes = np.arange(0, np.max(labels)+1, 1)
    ba_size_pixels = np.bincount(labels.reshape(-1))
    labels[np.isin(labels,classes[ba_size_pixels < min_size_pixels])] = 0

    # Relabel
    _, labels = np.unique(labels,return_inverse=1)
    labels = labels.reshape(data.shape)
    labels[data==0] = 0

    # Calculate unique (data, label) pairs ignoring no fire pixels
    data_aux = data[labels>0]
    labels_aux = labels[labels>0]
    unique_pairs = np.unique(np.array((data_aux, labels_aux), dtype=np.uint32).reshape(2,-1), axis=1)
    del data_aux, labels_aux

    # Find min and max dates
    df = pd.DataFrame({'date': unique_pairs[0], 'label_id': unique_pairs[1]})
    df = df.groupby('label_id').agg({'date' : [np.min, np.max]})
    df = df.reset_index()
    df = pd.DataFrame({'label_id': df.label_id, 'tstart': df.date.amin,
                       'tend': df.date.amax})
    return labels, df

def to_polygon(data, crs, transform, base_df, area_epsg=None):
    geoms = list({'properties': {'value': v}, 'geometry': s}
        for i, (s, v) in enumerate(rasterio.features.shapes(data.astype(np.uint16), transform=transform))
    )
    df = gpd.GeoDataFrame.from_features(geoms, crs=crs)
    df = df.loc[df.value>0].reset_index(drop=True)
    geoms = []
    for i in range(1, int(df.value.max())+1):
        polys = df.loc[df.value==i].geometry.values
        if len(polys) > 0:
            mpoly = MultiPolygon(polys)
            geoms.append(mpoly)
    df = gpd.GeoDataFrame({'label_id': list(range(1, int(df.value.max())+1)),
                           'geometry': geoms}, crs=crs)
    df = df.merge(base_df, how='left', on='label_id')
    if area_epsg is None:
        df['area_ha'] = df.area*0.0001
    else:
        df['area_ha'] = np.round((df.to_crs(epsg=area_epsg).area.values*0.0001), 2)
    return df

def run_all(input_path, save_path, interval_days=4, interval_pixels=4,
            min_size_pixels=1, save_tif=True, save_shape=False):
    if input_path.is_dir():
        files = list(input_path.iterdir())
        files = [f for f in files if 'tif' in f.suffix]
    else:
        files = [input_path]

    for f in tqdm(files):
        fsave = save_path/f'{f.stem}_labels'
        data, meta, crs, transform = read_data(f)
        meta.update(compress='lzw')
        labels, df = split_fires(data,
                                 interval_days=interval_days,
                                 interval_pixels=interval_pixels,
                                 min_size_pixels=min_size_pixels)
        if save_tif: save_data(str(fsave) + '.tif', labels, meta)
        if save_shape:
            df = to_polygon(labels, crs, transform, df)
            df.to_file(str(fsave) + '.shp')