"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const monitored_certificate_1 = require("../monitored-certificate");
const config_1 = require("./config");
const home_response_function_1 = require("./home-response-function");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const indexFunctionId = `IndexHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_DISABLED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        const indexBehaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_DISABLED,
            functionAssociations: [{
                    function: new home_response_function_1.HomeResponseFunction(this, indexFunctionId, {
                        functionName: indexFunctionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        const websiteOrigin = new origins.S3Origin(this.bucket);
        this.distribution.addBehavior('/index.html', websiteOrigin, indexBehaviorOptions);
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            cacheControl: [
                aws_s3_deployment_1.CacheControl.setPublic(),
                aws_s3_deployment_1.CacheControl.maxAge(core_1.Duration.hours(1)),
                aws_s3_deployment_1.CacheControl.mustRevalidate(),
                aws_s3_deployment_1.CacheControl.sMaxAge(core_1.Duration.minutes(5)),
                aws_s3_deployment_1.CacheControl.proxyRevalidate(),
            ],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
        });
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            cacheControl: [
                aws_s3_deployment_1.CacheControl.setPublic(),
                aws_s3_deployment_1.CacheControl.maxAge(core_1.Duration.hours(1)),
                aws_s3_deployment_1.CacheControl.mustRevalidate(),
                aws_s3_deployment_1.CacheControl.sMaxAge(core_1.Duration.minutes(5)),
                aws_s3_deployment_1.CacheControl.proxyRevalidate(),
            ],
            sources: [s3deploy.Source.asset(config.file.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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