"use strict";
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, privateMap) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to get private field on non-instance");
    }
    return privateMap.get(receiver);
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, privateMap, value) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to set private field on non-instance");
    }
    privateMap.set(receiver, value);
    return value;
};
var _catalog;
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const https = require("https");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const JSONStream = require("JSONStream");
const aws = require("../../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../../backend/shared/env.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * This package canary monitors the availability of the versions of a specified
 * package in the ConstructHub catalog. It publishes metrics that help
 * understand how much time passes between a pakcage appearing in the public
 * registry and it's availability in the ConstructHub instance.
 *
 * From the moment a package has been published, and until it appeared in
 * catalog, the `MetricName.DWELL_TIME` metric is emitted.
 *
 * Once the package has appeared in catalog, and until a new package version is
 * identified in npmjs.com, the `MetricName.TIME_TO_CATALOG` metric is emitted.
 *
 * If a new package version is published before the previous one has appeared
 * in catalog, both versions will be tracked at the same time, and the metrics
 * will receive one sample per tracked version.
 */
async function handler(event) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const packageName = env_lambda_shared_1.requireEnv("PACKAGE_NAME" /* PACKAGE_NAME */);
    const stateBucket = env_lambda_shared_1.requireEnv("PACKAGE_CANARY_BUCKET_NAME" /* PACKAGE_CANARY_BUCKET_NAME */);
    const constructHubEndpoint = env_lambda_shared_1.requireEnv("CONSTRUCT_HUB_BASE_URL" /* CONSTRUCT_HUB_BASE_URL */);
    const stateService = new CanaryStateService(stateBucket);
    const constructHub = new ConstructHub(constructHubEndpoint);
    const latest = await stateService.latest(packageName);
    const state = (_a = await stateService.load(packageName)) !== null && _a !== void 0 ? _a : {
        latest: {
            ...latest,
            // If that latest version is ALREADY in catalog, pretend it was
            // "instantaneously" there, so we avoid possibly reporting an breach of
            // SLA alarm, when we really just observed presence of the package in
            // catalog too late, for example on first deployment of the canary.
            availableAt: await constructHub.isInCatalog(packageName, latest.version)
                ? latest.publishedAt
                : undefined,
        },
        pending: {},
    };
    // If the current "latest" isn't the one from state, it needs updating.
    updateLatestIfNeeded(state, latest);
    try {
        await aws_embedded_metrics_1.metricScope((metrics) => () => {
            // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
            metrics.setDimensions();
            metrics.putMetric("TrackedVersionCount" /* TRACKED_VERSION_COUNT */, Object.keys(state.pending).length + 1, aws_embedded_metrics_1.Unit.Count);
        })();
        for (const versionState of [state.latest, ...Object.values((_b = state.pending) !== null && _b !== void 0 ? _b : {})]) {
            await aws_embedded_metrics_1.metricScope((metrics) => async () => {
                // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
                metrics.setDimensions();
                metrics.setProperty('PackageName', packageName);
                metrics.setProperty('PackageVersion', versionState.version);
                metrics.setProperty('IsLatest', state.latest.version === versionState.version);
                if (!versionState.availableAt) {
                    if (versionState.version === state.latest.version) {
                        if (await constructHub.isInCatalog(packageName, versionState.version)) {
                            versionState.availableAt = new Date();
                        }
                    }
                    else {
                        // Non-current versions will probably never make it to catalog (they're older than the
                        // current version), so instead, we check whether they have TypeScript documentation.
                        if (await constructHub.hasTypeScriptDocumentation(packageName, versionState.version)) {
                            versionState.availableAt = new Date();
                        }
                    }
                }
                if (versionState.availableAt) {
                    // Tells us how long it's taken for the package to make it to catalog after it was published.
                    metrics.putMetric("TimeToCatalog" /* TIME_TO_CATALOG */, (versionState.availableAt.getTime() - versionState.publishedAt.getTime()) / 1000, aws_embedded_metrics_1.Unit.Seconds);
                    // Stop tracking that version, as it's now available.
                    if (state.pending) {
                        delete state.pending[versionState.version];
                    }
                }
                else {
                    // Tells us how long we've been waiting for this version to show up, so far.
                    metrics.putMetric("DwellTime" /* DWELL_TIME */, (Date.now() - versionState.publishedAt.getTime()) / 1000, aws_embedded_metrics_1.Unit.Seconds);
                }
            })();
        }
    }
    finally {
        await stateService.save(packageName, state);
    }
}
exports.handler = handler;
class ConstructHub {
    constructor(baseUrl) {
        this.baseUrl = baseUrl;
        _catalog.set(this, void 0);
    }
    /**
     * Determines whether the specified package version is present in the catalog
     * object or not.
     *
     * @param packageName    the name of the checked package.
     * @param packageVersion the version of the checked package.
     *
     * @returns `true` IIF the exact package version is found in the catalog.
     */
    async isInCatalog(packageName, packageVersion) {
        const catalog = await this.getCatalog();
        const filtered = catalog.packages.filter((p) => p.name === packageName && p.version === packageVersion);
        if (filtered.length > 1) {
            throw new Error(`Found multiple entries for ${packageName}@${packageVersion} in catalog`);
        }
        return filtered.length === 1;
    }
    /**
     * Checks whether TypeScript documentation exists in ConstructHub for the
     * specified package version.
     *
     * @param packageName    the name of the checked package.
     * @param packageVersion the version of the checked package.
     *
     * @returns `true` IIF the `docs-typescript.md` document exists for the
     *          specified package.
     */
    async hasTypeScriptDocumentation(packageName, packageVersion) {
        return new Promise((ok, ko) => {
            const url = `${this.baseUrl}/data/${packageName}/v${packageVersion}/docs-typescript.md`;
            https.request(url, { method: 'HEAD' }, (res) => {
                var _a;
                if (res.statusCode === 200) {
                    // This returns HTTP 200 with text/html if it's a 404, due to how
                    // we configured CloudFront behaviors.
                    return ok(!!((_a = res.headers['content-type']) === null || _a === void 0 ? void 0 : _a.startsWith('text/markdown')));
                }
                const err = new Error(`HEAD ${url} -- HTTP ${res.statusCode} (${res.statusMessage})`);
                Error.captureStackTrace(err);
                ko(err);
            });
        });
    }
    async getCatalog() {
        if (__classPrivateFieldGet(this, _catalog)) {
            return __classPrivateFieldGet(this, _catalog);
        }
        return __classPrivateFieldSet(this, _catalog, await getJSON(`${this.baseUrl}/catalog.json`));
    }
}
_catalog = new WeakMap();
class CanaryStateService {
    constructor(bucketName) {
        this.bucketName = bucketName;
    }
    /**
     * Save the state to the bucket.
     */
    async save(packageName, state) {
        const url = this.url(packageName);
        console.log(`Saving: ${url}`);
        await aws.s3().putObject({
            Bucket: this.bucketName,
            Key: this.key(packageName),
            Body: JSON.stringify(state, null, 2),
            ContentType: 'application/json',
        }).promise();
    }
    /**
     * Load the state file for this package from the bucket.
     */
    async load(packageName) {
        console.log(`Loading state for package '${packageName}'`);
        const objectKey = this.key(packageName);
        const url = this.url(packageName);
        console.log(`Fetching: ${url}`);
        const data = await aws.s3().getObject({ Bucket: this.bucketName, Key: objectKey }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
        if (!(data === null || data === void 0 ? void 0 : data.Body)) {
            console.log(`Not found: ${url}`);
            return undefined;
        }
        console.log(`Loaded: ${url}`);
        return JSON.parse(data.Body.toString('utf-8'), (key, value) => {
            if (key === 'publishedAt' || key === 'availableAt') {
                return new Date(value);
            }
            return value;
        });
    }
    /**
     * Create a state from the latest version of the package.
     */
    async latest(packageName) {
        console.log(`Fetching latest version information from NPM: ${packageName}`);
        const version = await getJSON(`https://registry.npmjs.org/${encodeURIComponent(packageName)}/latest`, ['version']);
        const publishedAt = await getJSON(`https://registry.npmjs.org/${encodeURIComponent(packageName)}`, ['time', version]);
        console.log(`Package: ${packageName} | Version : ${version} | Published At: ${publishedAt}`);
        return { version, publishedAt: new Date(publishedAt) };
    }
    key(packageName) {
        return `${"package-canary/" /* STATE_PREFIX */}${packageName}${".state.json" /* STATE_SUFFIX */}`;
    }
    url(packageName) {
        return `s3://${this.bucketName}/${this.key(packageName)}`;
    }
}
/**
 * Makes a request to the provided URL and returns the response after having
 * parsed it from JSON.
 *
 * @param url the URL to get.
 * @param jsonPath a JSON path to extract only a subset of the object.
 */
function getJSON(url, jsonPath) {
    return new Promise((ok, ko) => {
        https.get(url, { headers: { 'Accept': 'application/json', 'Accept-Encoding': 'identity' } }, (res) => {
            if (res.statusCode !== 200) {
                const error = new Error(`GET ${url} - HTTP ${res.statusCode} (${res.statusMessage})`);
                Error.captureStackTrace(error);
                return ko(error);
            }
            res.once('error', ko);
            const json = JSONStream.parse(jsonPath);
            json.once('data', ok);
            json.once('error', ko);
            const plainPayload = res.headers['content-encoding'] === 'gzip' ? gunzip(res) : res;
            plainPayload.pipe(json, { end: true });
        });
    });
}
/**
 * Updates the `latest` property of `state` ti the provided `latest` value,
 * unless this is already the current latest.
 *
 * If the previous latest version does not have the `availableAt` property, adds
 * that to the `pending` set.
 *
 * @param state  the state to be updated.
 * @param latest the current "latest" version of the tracked package.
 */
function updateLatestIfNeeded(state, latest) {
    if (state.latest.version === latest.version) {
        return;
    }
    // If the current "latest" isn't available yet, add it to the `pending` versions.
    if (state.latest.availableAt == null) {
        // The TypeScript version of jsii doesn't do control flow analysis well enough here to
        // determine that the`if` branch guarantees `availableAt` is undefined here.
        state.pending[state.latest.version] = { ...state.latest, availableAt: undefined };
    }
    state.latest = latest;
}
function gunzip(readable) {
    const gz = zlib_1.createGunzip();
    readable.pipe(gz, { end: true });
    return gz;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibnBtanMtcGFja2FnZS1jYW5hcnkubGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vc3JjL3BhY2thZ2Utc291cmNlcy9ucG1qcy9jYW5hcnkvbnBtanMtcGFja2FnZS1jYW5hcnkubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBQUEsK0JBQStCO0FBRS9CLCtCQUFvQztBQUNwQywrREFBd0U7QUFFeEUseUNBQXlDO0FBRXpDLGlFQUFpRTtBQUNqRSxpRkFBdUU7QUFDdkUsMkNBQW9GO0FBRXBGLG9DQUFhLENBQUMsU0FBUyxHQUFHLDZCQUFpQixDQUFDO0FBRTVDOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNJLEtBQUssVUFBVSxPQUFPLENBQUMsS0FBYzs7SUFFMUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxVQUFVLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFeEQsTUFBTSxXQUFXLEdBQUcsOEJBQVUsbUNBQTBCLENBQUM7SUFDekQsTUFBTSxXQUFXLEdBQUcsOEJBQVUsK0RBQXdDLENBQUM7SUFDdkUsTUFBTSxvQkFBb0IsR0FBRyw4QkFBVSx1REFBb0MsQ0FBQztJQUU1RSxNQUFNLFlBQVksR0FBRyxJQUFJLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3pELE1BQU0sWUFBWSxHQUFHLElBQUksWUFBWSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFFNUQsTUFBTSxNQUFNLEdBQUcsTUFBTSxZQUFZLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQ3RELE1BQU0sS0FBSyxTQUFnQixNQUFNLFlBQVksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUUxRDtRQUNELE1BQU0sRUFBRTtZQUNOLEdBQUcsTUFBTTtZQUNULCtEQUErRDtZQUMvRCx1RUFBdUU7WUFDdkUscUVBQXFFO1lBQ3JFLG1FQUFtRTtZQUNuRSxXQUFXLEVBQUUsTUFBTSxZQUFZLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDO2dCQUN0RSxDQUFDLENBQUMsTUFBTSxDQUFDLFdBQVc7Z0JBQ3BCLENBQUMsQ0FBQyxTQUFTO1NBQ2Q7UUFDRCxPQUFPLEVBQUUsRUFBRTtLQUNaLENBQUM7SUFFSix1RUFBdUU7SUFDdkUsb0JBQW9CLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRXBDLElBQUk7UUFDRixNQUFNLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEdBQUcsRUFBRTtZQUNsQywySEFBMkg7WUFDM0gsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQ3hCLE9BQU8sQ0FBQyxTQUFTLG9EQUFtQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekcsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUVMLEtBQUssTUFBTSxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLEdBQUcsTUFBTSxDQUFDLE1BQU0sT0FBQyxLQUFLLENBQUMsT0FBTyxtQ0FBSSxFQUFFLENBQUMsQ0FBQyxFQUFFO1lBQ2hGLE1BQU0sa0NBQVcsQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsS0FBSyxJQUFJLEVBQUU7Z0JBQ3hDLDJIQUEySDtnQkFDM0gsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO2dCQUN4QixPQUFPLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQztnQkFDaEQsT0FBTyxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsRUFBRSxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQzVELE9BQU8sQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxLQUFLLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFFL0UsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQzdCLElBQUksWUFBWSxDQUFDLE9BQU8sS0FBSyxLQUFLLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRTt3QkFDakQsSUFBSSxNQUFNLFlBQVksQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLFlBQVksQ0FBQyxPQUFPLENBQUMsRUFBRTs0QkFDckUsWUFBWSxDQUFDLFdBQVcsR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDO3lCQUN2QztxQkFDRjt5QkFBTTt3QkFDTCxzRkFBc0Y7d0JBQ3RGLHFGQUFxRjt3QkFDckYsSUFBSSxNQUFNLFlBQVksQ0FBQywwQkFBMEIsQ0FBQyxXQUFXLEVBQUUsWUFBWSxDQUFDLE9BQU8sQ0FBQyxFQUFFOzRCQUNwRixZQUFZLENBQUMsV0FBVyxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7eUJBQ3ZDO3FCQUNGO2lCQUNGO2dCQUVELElBQUksWUFBWSxDQUFDLFdBQVcsRUFBRTtvQkFDNUIsNkZBQTZGO29CQUM3RixPQUFPLENBQUMsU0FBUyx3Q0FFZixDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLEdBQUcsWUFBWSxDQUFDLFdBQVcsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxHQUFHLElBQUssRUFDakYsMkJBQUksQ0FBQyxPQUFPLENBQ2IsQ0FBQztvQkFFRixxREFBcUQ7b0JBQ3JELElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRTt3QkFDakIsT0FBTyxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztxQkFDNUM7aUJBQ0Y7cUJBQU07b0JBQ0wsNEVBQTRFO29CQUM1RSxPQUFPLENBQUMsU0FBUywrQkFFZixDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxZQUFZLENBQUMsV0FBVyxDQUFDLE9BQU8sRUFBRSxDQUFDLEdBQUcsSUFBSyxFQUN6RCwyQkFBSSxDQUFDLE9BQU8sQ0FDYixDQUFDO2lCQUNIO1lBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztTQUNOO0tBQ0Y7WUFBUztRQUNSLE1BQU0sWUFBWSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDN0M7QUFDSCxDQUFDO0FBckZELDBCQXFGQztBQUVELE1BQU0sWUFBWTtJQUdoQixZQUE2QixPQUFlO1FBQWYsWUFBTyxHQUFQLE9BQU8sQ0FBUTtRQUY1QywyQkFBd0I7SUFFdUIsQ0FBQztJQUVoRDs7Ozs7Ozs7T0FRRztJQUNJLEtBQUssQ0FBQyxXQUFXLENBQUMsV0FBbUIsRUFBRSxjQUFzQjtRQUVsRSxNQUFNLE9BQU8sR0FBRyxNQUFNLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQU0sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxXQUFXLElBQUksQ0FBQyxDQUFDLE9BQU8sS0FBSyxjQUFjLENBQUMsQ0FBQztRQUU3RyxJQUFJLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsOEJBQThCLFdBQVcsSUFBSSxjQUFjLGFBQWEsQ0FBQyxDQUFDO1NBQzNGO1FBRUQsT0FBTyxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksS0FBSyxDQUFDLDBCQUEwQixDQUFDLFdBQW1CLEVBQUUsY0FBc0I7UUFDakYsT0FBTyxJQUFJLE9BQU8sQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRTtZQUM1QixNQUFNLEdBQUcsR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLFNBQVMsV0FBVyxLQUFLLGNBQWMscUJBQXFCLENBQUM7WUFDeEYsS0FBSyxDQUFDLE9BQU8sQ0FDWCxHQUFHLEVBQ0gsRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFLEVBQ2xCLENBQUMsR0FBRyxFQUFFLEVBQUU7O2dCQUNOLElBQUksR0FBRyxDQUFDLFVBQVUsS0FBSyxHQUFHLEVBQUU7b0JBQzFCLGlFQUFpRTtvQkFDakUsc0NBQXNDO29CQUN0QyxPQUFPLEVBQUUsQ0FBQyxDQUFDLFFBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsMENBQUUsVUFBVSxDQUFDLGVBQWUsRUFBQyxDQUFDLENBQUM7aUJBQ3ZFO2dCQUNELE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLFFBQVEsR0FBRyxZQUFZLEdBQUcsQ0FBQyxVQUFVLEtBQUssR0FBRyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUM7Z0JBQ3RGLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDN0IsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ1YsQ0FBQyxDQUNGLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxLQUFLLENBQUMsVUFBVTtRQUN0Qiw0Q0FBbUI7WUFDakIsOENBQXFCO1NBQ3RCO1FBQ0QsOEJBQU8sSUFBSSxZQUFZLE1BQU0sT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sZUFBZSxDQUFDLEVBQUM7SUFDdkUsQ0FBQztDQUNGOztBQUVELE1BQU0sa0JBQWtCO0lBRXRCLFlBQTZCLFVBQWtCO1FBQWxCLGVBQVUsR0FBVixVQUFVLENBQVE7SUFBRyxDQUFDO0lBRW5EOztPQUVHO0lBQ0ksS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFtQixFQUFFLEtBQWtCO1FBRXZELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUM7UUFFbEMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxXQUFXLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDOUIsTUFBTSxHQUFHLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxDQUFDO1lBQ3ZCLE1BQU0sRUFBRSxJQUFJLENBQUMsVUFBVTtZQUN2QixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUM7WUFDMUIsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7WUFDcEMsV0FBVyxFQUFFLGtCQUFrQjtTQUNoQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUM7SUFFZixDQUFDO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQW1CO1FBRW5DLE9BQU8sQ0FBQyxHQUFHLENBQUMsOEJBQThCLFdBQVcsR0FBRyxDQUFDLENBQUM7UUFFMUQsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN4QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRWxDLE9BQU8sQ0FBQyxHQUFHLENBQUMsYUFBYSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ2hDLE1BQU0sSUFBSSxHQUFHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE1BQU0sRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLEdBQUcsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLE9BQU8sRUFBRTthQUN6RixLQUFLLENBQUMsQ0FBQyxHQUFhLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEtBQUssV0FBVztZQUNoRCxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUM7WUFDckIsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsRUFBRSxhQUFhLENBQXdCLENBQUMsQ0FBQyxDQUFDO1FBRWhFLElBQUksRUFBQyxJQUFJLGFBQUosSUFBSSx1QkFBSixJQUFJLENBQUUsSUFBSSxDQUFBLEVBQUU7WUFDZixPQUFPLENBQUMsR0FBRyxDQUFDLGNBQWMsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUNqQyxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE9BQU8sQ0FBQyxHQUFHLENBQUMsV0FBVyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsRUFBRTtZQUM1RCxJQUFJLEdBQUcsS0FBSyxhQUFhLElBQUksR0FBRyxLQUFLLGFBQWEsRUFBRTtnQkFDbEQsT0FBTyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUN4QjtZQUNELE9BQU8sS0FBSyxDQUFDO1FBQ2YsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQW1CO1FBRXJDLE9BQU8sQ0FBQyxHQUFHLENBQUMsaURBQWlELFdBQVcsRUFBRSxDQUFDLENBQUM7UUFDNUUsTUFBTSxPQUFPLEdBQUcsTUFBTSxPQUFPLENBQUMsOEJBQThCLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ25ILE1BQU0sV0FBVyxHQUFHLE1BQU0sT0FBTyxDQUFDLDhCQUE4QixrQkFBa0IsQ0FBQyxXQUFXLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFFdEgsT0FBTyxDQUFDLEdBQUcsQ0FBQyxZQUFZLFdBQVcsZ0JBQWdCLE9BQU8sb0JBQW9CLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFN0YsT0FBTyxFQUFFLE9BQU8sRUFBRSxXQUFXLEVBQUUsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQztJQUN6RCxDQUFDO0lBRU8sR0FBRyxDQUFDLFdBQW1CO1FBQzdCLE9BQU8sR0FBRyxvQ0FBc0IsR0FBRyxXQUFXLEdBQUcsZ0NBQXNCLEVBQUUsQ0FBQztJQUM1RSxDQUFDO0lBRU8sR0FBRyxDQUFDLFdBQW1CO1FBQzdCLE9BQU8sUUFBUSxJQUFJLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQztJQUM1RCxDQUFDO0NBR0Y7QUE4Q0Q7Ozs7OztHQU1HO0FBQ0gsU0FBUyxPQUFPLENBQUMsR0FBVyxFQUFFLFFBQW1CO0lBQy9DLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7UUFDNUIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRSxRQUFRLEVBQUUsa0JBQWtCLEVBQUUsaUJBQWlCLEVBQUUsVUFBVSxFQUFFLEVBQUUsRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQ25HLElBQUksR0FBRyxDQUFDLFVBQVUsS0FBSyxHQUFHLEVBQUU7Z0JBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksS0FBSyxDQUFDLE9BQU8sR0FBRyxXQUFXLEdBQUcsQ0FBQyxVQUFVLEtBQUssR0FBRyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUM7Z0JBQ3RGLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDL0IsT0FBTyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDbEI7WUFFRCxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztZQUV0QixNQUFNLElBQUksR0FBRyxVQUFVLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ3hDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBRXZCLE1BQU0sWUFBWSxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLENBQUMsS0FBSyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO1lBQ3BGLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDekMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRDs7Ozs7Ozs7O0dBU0c7QUFDSCxTQUFTLG9CQUFvQixDQUFDLEtBQWtCLEVBQUUsTUFBNkI7SUFDN0UsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLE9BQU8sS0FBSyxNQUFNLENBQUMsT0FBTyxFQUFFO1FBQzNDLE9BQU87S0FDUjtJQUVELGlGQUFpRjtJQUNqRixJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVyxJQUFJLElBQUksRUFBRTtRQUNwQyxzRkFBc0Y7UUFDdEYsNEVBQTRFO1FBQzVFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLENBQUM7S0FDbkY7SUFFRCxLQUFLLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztBQUN4QixDQUFDO0FBRUQsU0FBUyxNQUFNLENBQUMsUUFBa0I7SUFDaEMsTUFBTSxFQUFFLEdBQUcsbUJBQVksRUFBRSxDQUFDO0lBQzFCLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLEVBQUUsR0FBRyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDakMsT0FBTyxFQUFFLENBQUM7QUFDWixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaHR0cHMgZnJvbSAnaHR0cHMnO1xuaW1wb3J0IHsgUmVhZGFibGUgfSBmcm9tICdzdHJlYW0nO1xuaW1wb3J0IHsgY3JlYXRlR3VuemlwIH0gZnJvbSAnemxpYic7XG5pbXBvcnQgeyBtZXRyaWNTY29wZSwgQ29uZmlndXJhdGlvbiwgVW5pdCB9IGZyb20gJ2F3cy1lbWJlZGRlZC1tZXRyaWNzJztcbmltcG9ydCB0eXBlIHsgQVdTRXJyb3IsIFMzIH0gZnJvbSAnYXdzLXNkayc7XG5pbXBvcnQgKiBhcyBKU09OU3RyZWFtIGZyb20gJ0pTT05TdHJlYW0nO1xuaW1wb3J0IHsgQ2F0YWxvZ01vZGVsIH0gZnJvbSAnLi4vLi4vLi4vYmFja2VuZCc7XG5pbXBvcnQgKiBhcyBhd3MgZnJvbSAnLi4vLi4vLi4vYmFja2VuZC9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uLy4uLy4uL2JhY2tlbmQvc2hhcmVkL2Vudi5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IE1FVFJJQ1NfTkFNRVNQQUNFLCBNZXRyaWNOYW1lLCBFbnZpcm9ubWVudCwgT2JqZWN0S2V5IH0gZnJvbSAnLi9jb25zdGFudHMnO1xuXG5Db25maWd1cmF0aW9uLm5hbWVzcGFjZSA9IE1FVFJJQ1NfTkFNRVNQQUNFO1xuXG4vKipcbiAqIFRoaXMgcGFja2FnZSBjYW5hcnkgbW9uaXRvcnMgdGhlIGF2YWlsYWJpbGl0eSBvZiB0aGUgdmVyc2lvbnMgb2YgYSBzcGVjaWZpZWRcbiAqIHBhY2thZ2UgaW4gdGhlIENvbnN0cnVjdEh1YiBjYXRhbG9nLiBJdCBwdWJsaXNoZXMgbWV0cmljcyB0aGF0IGhlbHBcbiAqIHVuZGVyc3RhbmQgaG93IG11Y2ggdGltZSBwYXNzZXMgYmV0d2VlbiBhIHBha2NhZ2UgYXBwZWFyaW5nIGluIHRoZSBwdWJsaWNcbiAqIHJlZ2lzdHJ5IGFuZCBpdCdzIGF2YWlsYWJpbGl0eSBpbiB0aGUgQ29uc3RydWN0SHViIGluc3RhbmNlLlxuICpcbiAqIEZyb20gdGhlIG1vbWVudCBhIHBhY2thZ2UgaGFzIGJlZW4gcHVibGlzaGVkLCBhbmQgdW50aWwgaXQgYXBwZWFyZWQgaW5cbiAqIGNhdGFsb2csIHRoZSBgTWV0cmljTmFtZS5EV0VMTF9USU1FYCBtZXRyaWMgaXMgZW1pdHRlZC5cbiAqXG4gKiBPbmNlIHRoZSBwYWNrYWdlIGhhcyBhcHBlYXJlZCBpbiBjYXRhbG9nLCBhbmQgdW50aWwgYSBuZXcgcGFja2FnZSB2ZXJzaW9uIGlzXG4gKiBpZGVudGlmaWVkIGluIG5wbWpzLmNvbSwgdGhlIGBNZXRyaWNOYW1lLlRJTUVfVE9fQ0FUQUxPR2AgbWV0cmljIGlzIGVtaXR0ZWQuXG4gKlxuICogSWYgYSBuZXcgcGFja2FnZSB2ZXJzaW9uIGlzIHB1Ymxpc2hlZCBiZWZvcmUgdGhlIHByZXZpb3VzIG9uZSBoYXMgYXBwZWFyZWRcbiAqIGluIGNhdGFsb2csIGJvdGggdmVyc2lvbnMgd2lsbCBiZSB0cmFja2VkIGF0IHRoZSBzYW1lIHRpbWUsIGFuZCB0aGUgbWV0cmljc1xuICogd2lsbCByZWNlaXZlIG9uZSBzYW1wbGUgcGVyIHRyYWNrZWQgdmVyc2lvbi5cbiAqL1xuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGhhbmRsZXIoZXZlbnQ6IHVua25vd24pOiBQcm9taXNlPHZvaWQ+IHtcblxuICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gIGNvbnN0IHBhY2thZ2VOYW1lID0gcmVxdWlyZUVudihFbnZpcm9ubWVudC5QQUNLQUdFX05BTUUpO1xuICBjb25zdCBzdGF0ZUJ1Y2tldCA9IHJlcXVpcmVFbnYoRW52aXJvbm1lbnQuUEFDS0FHRV9DQU5BUllfQlVDS0VUX05BTUUpO1xuICBjb25zdCBjb25zdHJ1Y3RIdWJFbmRwb2ludCA9IHJlcXVpcmVFbnYoRW52aXJvbm1lbnQuQ09OU1RSVUNUX0hVQl9CQVNFX1VSTCk7XG5cbiAgY29uc3Qgc3RhdGVTZXJ2aWNlID0gbmV3IENhbmFyeVN0YXRlU2VydmljZShzdGF0ZUJ1Y2tldCk7XG4gIGNvbnN0IGNvbnN0cnVjdEh1YiA9IG5ldyBDb25zdHJ1Y3RIdWIoY29uc3RydWN0SHViRW5kcG9pbnQpO1xuXG4gIGNvbnN0IGxhdGVzdCA9IGF3YWl0IHN0YXRlU2VydmljZS5sYXRlc3QocGFja2FnZU5hbWUpO1xuICBjb25zdCBzdGF0ZTogQ2FuYXJ5U3RhdGUgPSBhd2FpdCBzdGF0ZVNlcnZpY2UubG9hZChwYWNrYWdlTmFtZSlcbiAgICAvLyBJZiB3ZSBkaWQgbm90IGhhdmUgYW55IHN0YXRlLCB3ZSdsbCBib290c3RyYXAgdXNpbmcgdGhlIGN1cnJlbnQgbGF0ZXN0IHZlcnNpb24uXG4gICAgPz8ge1xuICAgICAgbGF0ZXN0OiB7XG4gICAgICAgIC4uLmxhdGVzdCxcbiAgICAgICAgLy8gSWYgdGhhdCBsYXRlc3QgdmVyc2lvbiBpcyBBTFJFQURZIGluIGNhdGFsb2csIHByZXRlbmQgaXQgd2FzXG4gICAgICAgIC8vIFwiaW5zdGFudGFuZW91c2x5XCIgdGhlcmUsIHNvIHdlIGF2b2lkIHBvc3NpYmx5IHJlcG9ydGluZyBhbiBicmVhY2ggb2ZcbiAgICAgICAgLy8gU0xBIGFsYXJtLCB3aGVuIHdlIHJlYWxseSBqdXN0IG9ic2VydmVkIHByZXNlbmNlIG9mIHRoZSBwYWNrYWdlIGluXG4gICAgICAgIC8vIGNhdGFsb2cgdG9vIGxhdGUsIGZvciBleGFtcGxlIG9uIGZpcnN0IGRlcGxveW1lbnQgb2YgdGhlIGNhbmFyeS5cbiAgICAgICAgYXZhaWxhYmxlQXQ6IGF3YWl0IGNvbnN0cnVjdEh1Yi5pc0luQ2F0YWxvZyhwYWNrYWdlTmFtZSwgbGF0ZXN0LnZlcnNpb24pXG4gICAgICAgICAgPyBsYXRlc3QucHVibGlzaGVkQXRcbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgICBwZW5kaW5nOiB7fSxcbiAgICB9O1xuXG4gIC8vIElmIHRoZSBjdXJyZW50IFwibGF0ZXN0XCIgaXNuJ3QgdGhlIG9uZSBmcm9tIHN0YXRlLCBpdCBuZWVkcyB1cGRhdGluZy5cbiAgdXBkYXRlTGF0ZXN0SWZOZWVkZWQoc3RhdGUsIGxhdGVzdCk7XG5cbiAgdHJ5IHtcbiAgICBhd2FpdCBtZXRyaWNTY29wZSgobWV0cmljcykgPT4gKCkgPT4ge1xuICAgICAgLy8gQ2xlYXIgb3V0IGRlZmF1bHQgZGltZW5zaW9ucyBhcyB3ZSBkb24ndCBuZWVkIHRob3NlLiBTZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3c2xhYnMvYXdzLWVtYmVkZGVkLW1ldHJpY3Mtbm9kZS9pc3N1ZXMvNzMuXG4gICAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcbiAgICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuVFJBQ0tFRF9WRVJTSU9OX0NPVU5ULCBPYmplY3Qua2V5cyhzdGF0ZS5wZW5kaW5nKS5sZW5ndGggKyAxLCBVbml0LkNvdW50KTtcbiAgICB9KSgpO1xuXG4gICAgZm9yIChjb25zdCB2ZXJzaW9uU3RhdGUgb2YgW3N0YXRlLmxhdGVzdCwgLi4uT2JqZWN0LnZhbHVlcyhzdGF0ZS5wZW5kaW5nID8/IHt9KV0pIHtcbiAgICAgIGF3YWl0IG1ldHJpY1Njb3BlKChtZXRyaWNzKSA9PiBhc3luYyAoKSA9PiB7XG4gICAgICAgIC8vIENsZWFyIG91dCBkZWZhdWx0IGRpbWVuc2lvbnMgYXMgd2UgZG9uJ3QgbmVlZCB0aG9zZS4gU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3NsYWJzL2F3cy1lbWJlZGRlZC1tZXRyaWNzLW5vZGUvaXNzdWVzLzczLlxuICAgICAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcbiAgICAgICAgbWV0cmljcy5zZXRQcm9wZXJ0eSgnUGFja2FnZU5hbWUnLCBwYWNrYWdlTmFtZSk7XG4gICAgICAgIG1ldHJpY3Muc2V0UHJvcGVydHkoJ1BhY2thZ2VWZXJzaW9uJywgdmVyc2lvblN0YXRlLnZlcnNpb24pO1xuICAgICAgICBtZXRyaWNzLnNldFByb3BlcnR5KCdJc0xhdGVzdCcsIHN0YXRlLmxhdGVzdC52ZXJzaW9uID09PSB2ZXJzaW9uU3RhdGUudmVyc2lvbik7XG5cbiAgICAgICAgaWYgKCF2ZXJzaW9uU3RhdGUuYXZhaWxhYmxlQXQpIHtcbiAgICAgICAgICBpZiAodmVyc2lvblN0YXRlLnZlcnNpb24gPT09IHN0YXRlLmxhdGVzdC52ZXJzaW9uKSB7XG4gICAgICAgICAgICBpZiAoYXdhaXQgY29uc3RydWN0SHViLmlzSW5DYXRhbG9nKHBhY2thZ2VOYW1lLCB2ZXJzaW9uU3RhdGUudmVyc2lvbikpIHtcbiAgICAgICAgICAgICAgdmVyc2lvblN0YXRlLmF2YWlsYWJsZUF0ID0gbmV3IERhdGUoKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgLy8gTm9uLWN1cnJlbnQgdmVyc2lvbnMgd2lsbCBwcm9iYWJseSBuZXZlciBtYWtlIGl0IHRvIGNhdGFsb2cgKHRoZXkncmUgb2xkZXIgdGhhbiB0aGVcbiAgICAgICAgICAgIC8vIGN1cnJlbnQgdmVyc2lvbiksIHNvIGluc3RlYWQsIHdlIGNoZWNrIHdoZXRoZXIgdGhleSBoYXZlIFR5cGVTY3JpcHQgZG9jdW1lbnRhdGlvbi5cbiAgICAgICAgICAgIGlmIChhd2FpdCBjb25zdHJ1Y3RIdWIuaGFzVHlwZVNjcmlwdERvY3VtZW50YXRpb24ocGFja2FnZU5hbWUsIHZlcnNpb25TdGF0ZS52ZXJzaW9uKSkge1xuICAgICAgICAgICAgICB2ZXJzaW9uU3RhdGUuYXZhaWxhYmxlQXQgPSBuZXcgRGF0ZSgpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuXG4gICAgICAgIGlmICh2ZXJzaW9uU3RhdGUuYXZhaWxhYmxlQXQpIHtcbiAgICAgICAgICAvLyBUZWxscyB1cyBob3cgbG9uZyBpdCdzIHRha2VuIGZvciB0aGUgcGFja2FnZSB0byBtYWtlIGl0IHRvIGNhdGFsb2cgYWZ0ZXIgaXQgd2FzIHB1Ymxpc2hlZC5cbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhcbiAgICAgICAgICAgIE1ldHJpY05hbWUuVElNRV9UT19DQVRBTE9HLFxuICAgICAgICAgICAgKHZlcnNpb25TdGF0ZS5hdmFpbGFibGVBdC5nZXRUaW1lKCkgLSB2ZXJzaW9uU3RhdGUucHVibGlzaGVkQXQuZ2V0VGltZSgpKSAvIDFfMDAwLFxuICAgICAgICAgICAgVW5pdC5TZWNvbmRzLFxuICAgICAgICAgICk7XG5cbiAgICAgICAgICAvLyBTdG9wIHRyYWNraW5nIHRoYXQgdmVyc2lvbiwgYXMgaXQncyBub3cgYXZhaWxhYmxlLlxuICAgICAgICAgIGlmIChzdGF0ZS5wZW5kaW5nKSB7XG4gICAgICAgICAgICBkZWxldGUgc3RhdGUucGVuZGluZ1t2ZXJzaW9uU3RhdGUudmVyc2lvbl07XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIFRlbGxzIHVzIGhvdyBsb25nIHdlJ3ZlIGJlZW4gd2FpdGluZyBmb3IgdGhpcyB2ZXJzaW9uIHRvIHNob3cgdXAsIHNvIGZhci5cbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhcbiAgICAgICAgICAgIE1ldHJpY05hbWUuRFdFTExfVElNRSxcbiAgICAgICAgICAgIChEYXRlLm5vdygpIC0gdmVyc2lvblN0YXRlLnB1Ymxpc2hlZEF0LmdldFRpbWUoKSkgLyAxXzAwMCxcbiAgICAgICAgICAgIFVuaXQuU2Vjb25kcyxcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICB9KSgpO1xuICAgIH1cbiAgfSBmaW5hbGx5IHtcbiAgICBhd2FpdCBzdGF0ZVNlcnZpY2Uuc2F2ZShwYWNrYWdlTmFtZSwgc3RhdGUpO1xuICB9XG59XG5cbmNsYXNzIENvbnN0cnVjdEh1YiB7XG4gICNjYXRhbG9nPzogQ2F0YWxvZ01vZGVsO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgYmFzZVVybDogc3RyaW5nKSB7fVxuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIHdoZXRoZXIgdGhlIHNwZWNpZmllZCBwYWNrYWdlIHZlcnNpb24gaXMgcHJlc2VudCBpbiB0aGUgY2F0YWxvZ1xuICAgKiBvYmplY3Qgb3Igbm90LlxuICAgKlxuICAgKiBAcGFyYW0gcGFja2FnZU5hbWUgICAgdGhlIG5hbWUgb2YgdGhlIGNoZWNrZWQgcGFja2FnZS5cbiAgICogQHBhcmFtIHBhY2thZ2VWZXJzaW9uIHRoZSB2ZXJzaW9uIG9mIHRoZSBjaGVja2VkIHBhY2thZ2UuXG4gICAqXG4gICAqIEByZXR1cm5zIGB0cnVlYCBJSUYgdGhlIGV4YWN0IHBhY2thZ2UgdmVyc2lvbiBpcyBmb3VuZCBpbiB0aGUgY2F0YWxvZy5cbiAgICovXG4gIHB1YmxpYyBhc3luYyBpc0luQ2F0YWxvZyhwYWNrYWdlTmFtZTogc3RyaW5nLCBwYWNrYWdlVmVyc2lvbjogc3RyaW5nKTogUHJvbWlzZTxib29sZWFuPiB7XG5cbiAgICBjb25zdCBjYXRhbG9nID0gYXdhaXQgdGhpcy5nZXRDYXRhbG9nKCk7XG4gICAgY29uc3QgZmlsdGVyZWQgPSBjYXRhbG9nLnBhY2thZ2VzLmZpbHRlcigocDogYW55KSA9PiBwLm5hbWUgPT09IHBhY2thZ2VOYW1lICYmIHAudmVyc2lvbiA9PT0gcGFja2FnZVZlcnNpb24pO1xuXG4gICAgaWYgKGZpbHRlcmVkLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRm91bmQgbXVsdGlwbGUgZW50cmllcyBmb3IgJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gaW4gY2F0YWxvZ2ApO1xuICAgIH1cblxuICAgIHJldHVybiBmaWx0ZXJlZC5sZW5ndGggPT09IDE7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2tzIHdoZXRoZXIgVHlwZVNjcmlwdCBkb2N1bWVudGF0aW9uIGV4aXN0cyBpbiBDb25zdHJ1Y3RIdWIgZm9yIHRoZVxuICAgKiBzcGVjaWZpZWQgcGFja2FnZSB2ZXJzaW9uLlxuICAgKlxuICAgKiBAcGFyYW0gcGFja2FnZU5hbWUgICAgdGhlIG5hbWUgb2YgdGhlIGNoZWNrZWQgcGFja2FnZS5cbiAgICogQHBhcmFtIHBhY2thZ2VWZXJzaW9uIHRoZSB2ZXJzaW9uIG9mIHRoZSBjaGVja2VkIHBhY2thZ2UuXG4gICAqXG4gICAqIEByZXR1cm5zIGB0cnVlYCBJSUYgdGhlIGBkb2NzLXR5cGVzY3JpcHQubWRgIGRvY3VtZW50IGV4aXN0cyBmb3IgdGhlXG4gICAqICAgICAgICAgIHNwZWNpZmllZCBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIGFzeW5jIGhhc1R5cGVTY3JpcHREb2N1bWVudGF0aW9uKHBhY2thZ2VOYW1lOiBzdHJpbmcsIHBhY2thZ2VWZXJzaW9uOiBzdHJpbmcpOiBQcm9taXNlPGJvb2xlYW4+IHtcbiAgICByZXR1cm4gbmV3IFByb21pc2UoKG9rLCBrbykgPT4ge1xuICAgICAgY29uc3QgdXJsID0gYCR7dGhpcy5iYXNlVXJsfS9kYXRhLyR7cGFja2FnZU5hbWV9L3Yke3BhY2thZ2VWZXJzaW9ufS9kb2NzLXR5cGVzY3JpcHQubWRgO1xuICAgICAgaHR0cHMucmVxdWVzdChcbiAgICAgICAgdXJsLFxuICAgICAgICB7IG1ldGhvZDogJ0hFQUQnIH0sXG4gICAgICAgIChyZXMpID0+IHtcbiAgICAgICAgICBpZiAocmVzLnN0YXR1c0NvZGUgPT09IDIwMCkge1xuICAgICAgICAgICAgLy8gVGhpcyByZXR1cm5zIEhUVFAgMjAwIHdpdGggdGV4dC9odG1sIGlmIGl0J3MgYSA0MDQsIGR1ZSB0byBob3dcbiAgICAgICAgICAgIC8vIHdlIGNvbmZpZ3VyZWQgQ2xvdWRGcm9udCBiZWhhdmlvcnMuXG4gICAgICAgICAgICByZXR1cm4gb2soISFyZXMuaGVhZGVyc1snY29udGVudC10eXBlJ10/LnN0YXJ0c1dpdGgoJ3RleHQvbWFya2Rvd24nKSk7XG4gICAgICAgICAgfVxuICAgICAgICAgIGNvbnN0IGVyciA9IG5ldyBFcnJvcihgSEVBRCAke3VybH0gLS0gSFRUUCAke3Jlcy5zdGF0dXNDb2RlfSAoJHtyZXMuc3RhdHVzTWVzc2FnZX0pYCk7XG4gICAgICAgICAgRXJyb3IuY2FwdHVyZVN0YWNrVHJhY2UoZXJyKTtcbiAgICAgICAgICBrbyhlcnIpO1xuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgZ2V0Q2F0YWxvZygpOiBQcm9taXNlPENhdGFsb2dNb2RlbD4ge1xuICAgIGlmICh0aGlzLiNjYXRhbG9nKSB7XG4gICAgICByZXR1cm4gdGhpcy4jY2F0YWxvZztcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuI2NhdGFsb2cgPSBhd2FpdCBnZXRKU09OKGAke3RoaXMuYmFzZVVybH0vY2F0YWxvZy5qc29uYCk7XG4gIH1cbn1cblxuY2xhc3MgQ2FuYXJ5U3RhdGVTZXJ2aWNlIHtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IGJ1Y2tldE5hbWU6IHN0cmluZykge31cblxuICAvKipcbiAgICogU2F2ZSB0aGUgc3RhdGUgdG8gdGhlIGJ1Y2tldC5cbiAgICovXG4gIHB1YmxpYyBhc3luYyBzYXZlKHBhY2thZ2VOYW1lOiBzdHJpbmcsIHN0YXRlOiBDYW5hcnlTdGF0ZSkge1xuXG4gICAgY29uc3QgdXJsID0gdGhpcy51cmwocGFja2FnZU5hbWUpO1xuXG4gICAgY29uc29sZS5sb2coYFNhdmluZzogJHt1cmx9YCk7XG4gICAgYXdhaXQgYXdzLnMzKCkucHV0T2JqZWN0KHtcbiAgICAgIEJ1Y2tldDogdGhpcy5idWNrZXROYW1lLFxuICAgICAgS2V5OiB0aGlzLmtleShwYWNrYWdlTmFtZSksXG4gICAgICBCb2R5OiBKU09OLnN0cmluZ2lmeShzdGF0ZSwgbnVsbCwgMiksXG4gICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL2pzb24nLFxuICAgIH0pLnByb21pc2UoKTtcblxuICB9XG5cbiAgLyoqXG4gICAqIExvYWQgdGhlIHN0YXRlIGZpbGUgZm9yIHRoaXMgcGFja2FnZSBmcm9tIHRoZSBidWNrZXQuXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgbG9hZChwYWNrYWdlTmFtZTogc3RyaW5nKTogUHJvbWlzZTxDYW5hcnlTdGF0ZSB8IHVuZGVmaW5lZD4ge1xuXG4gICAgY29uc29sZS5sb2coYExvYWRpbmcgc3RhdGUgZm9yIHBhY2thZ2UgJyR7cGFja2FnZU5hbWV9J2ApO1xuXG4gICAgY29uc3Qgb2JqZWN0S2V5ID0gdGhpcy5rZXkocGFja2FnZU5hbWUpO1xuICAgIGNvbnN0IHVybCA9IHRoaXMudXJsKHBhY2thZ2VOYW1lKTtcblxuICAgIGNvbnNvbGUubG9nKGBGZXRjaGluZzogJHt1cmx9YCk7XG4gICAgY29uc3QgZGF0YSA9IGF3YWl0IGF3cy5zMygpLmdldE9iamVjdCh7IEJ1Y2tldDogdGhpcy5idWNrZXROYW1lLCBLZXk6IG9iamVjdEtleSB9KS5wcm9taXNlKClcbiAgICAgIC5jYXRjaCgoZXJyOiBBV1NFcnJvcikgPT4gZXJyLmNvZGUgIT09ICdOb1N1Y2hLZXknXG4gICAgICAgID8gUHJvbWlzZS5yZWplY3QoZXJyKVxuICAgICAgICA6IFByb21pc2UucmVzb2x2ZSh7IC8qIG5vIGRhdGEgKi8gfSBhcyBTMy5HZXRPYmplY3RPdXRwdXQpKTtcblxuICAgIGlmICghZGF0YT8uQm9keSkge1xuICAgICAgY29uc29sZS5sb2coYE5vdCBmb3VuZDogJHt1cmx9YCk7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGNvbnNvbGUubG9nKGBMb2FkZWQ6ICR7dXJsfWApO1xuICAgIHJldHVybiBKU09OLnBhcnNlKGRhdGEuQm9keS50b1N0cmluZygndXRmLTgnKSwgKGtleSwgdmFsdWUpID0+IHtcbiAgICAgIGlmIChrZXkgPT09ICdwdWJsaXNoZWRBdCcgfHwga2V5ID09PSAnYXZhaWxhYmxlQXQnKSB7XG4gICAgICAgIHJldHVybiBuZXcgRGF0ZSh2YWx1ZSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gdmFsdWU7XG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgc3RhdGUgZnJvbSB0aGUgbGF0ZXN0IHZlcnNpb24gb2YgdGhlIHBhY2thZ2UuXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgbGF0ZXN0KHBhY2thZ2VOYW1lOiBzdHJpbmcpOiBQcm9taXNlPENhbmFyeVN0YXRlWydsYXRlc3QnXT4ge1xuXG4gICAgY29uc29sZS5sb2coYEZldGNoaW5nIGxhdGVzdCB2ZXJzaW9uIGluZm9ybWF0aW9uIGZyb20gTlBNOiAke3BhY2thZ2VOYW1lfWApO1xuICAgIGNvbnN0IHZlcnNpb24gPSBhd2FpdCBnZXRKU09OKGBodHRwczovL3JlZ2lzdHJ5Lm5wbWpzLm9yZy8ke2VuY29kZVVSSUNvbXBvbmVudChwYWNrYWdlTmFtZSl9L2xhdGVzdGAsIFsndmVyc2lvbiddKTtcbiAgICBjb25zdCBwdWJsaXNoZWRBdCA9IGF3YWl0IGdldEpTT04oYGh0dHBzOi8vcmVnaXN0cnkubnBtanMub3JnLyR7ZW5jb2RlVVJJQ29tcG9uZW50KHBhY2thZ2VOYW1lKX1gLCBbJ3RpbWUnLCB2ZXJzaW9uXSk7XG5cbiAgICBjb25zb2xlLmxvZyhgUGFja2FnZTogJHtwYWNrYWdlTmFtZX0gfCBWZXJzaW9uIDogJHt2ZXJzaW9ufSB8IFB1Ymxpc2hlZCBBdDogJHtwdWJsaXNoZWRBdH1gKTtcblxuICAgIHJldHVybiB7IHZlcnNpb24sIHB1Ymxpc2hlZEF0OiBuZXcgRGF0ZShwdWJsaXNoZWRBdCkgfTtcbiAgfVxuXG4gIHByaXZhdGUga2V5KHBhY2thZ2VOYW1lOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiBgJHtPYmplY3RLZXkuU1RBVEVfUFJFRklYfSR7cGFja2FnZU5hbWV9JHtPYmplY3RLZXkuU1RBVEVfU1VGRklYfWA7XG4gIH1cblxuICBwcml2YXRlIHVybChwYWNrYWdlTmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIGBzMzovLyR7dGhpcy5idWNrZXROYW1lfS8ke3RoaXMua2V5KHBhY2thZ2VOYW1lKX1gO1xuICB9XG5cblxufVxuXG5pbnRlcmZhY2UgQ2FuYXJ5U3RhdGUge1xuICAvKipcbiAgICogVGhlIGxhdGVzdCBwYWNrYWdlIHZlcnNpb24sIGFzIG9mIHRoZSBsYXN0IGV4ZWN1dGlvbiBvZiB0aGUgY2FuYXJ5LlxuICAgKi9cbiAgbGF0ZXN0OiB7XG4gICAgLyoqXG4gICAgICogVGhlIHZlcnNpb24gd2UgYXJlIHRyYWNraW5nLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZlcnNpb246IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFRoZSBwdWJsaXNoIGRhdGUgb2YgdGhlIHZlcnNpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgcHVibGlzaGVkQXQ6IERhdGU7XG5cbiAgICAvKipcbiAgICAgKiBUaGUgZGF0ZSBhdCB3aGljaCB0aGUgdmVyc2lvbiBpcyBhdmFpbGFibGUgb24gdGhlIGh1Yi5cbiAgICAgKi9cbiAgICBhdmFpbGFibGVBdD86IERhdGU7XG4gIH07XG5cbiAgLyoqXG4gICAqIEVhY2ggZXhpc3RpbmcsIGJ1dCBub3QteWV0LWZvdW5kIHZlcnNpb25zIHRoYXQgYXJlIHN0aWxsIHRyYWNrZWQuXG4gICAqL1xuICBwZW5kaW5nOiB7XG4gICAgW3ZlcnNpb246IHN0cmluZ106IHtcbiAgICAgIC8qKlxuICAgICAgICogVGhlIHZlcnNpb24gd2UgYXJlIHRyYWNraW5nLlxuICAgICAgICovXG4gICAgICByZWFkb25seSB2ZXJzaW9uOiBzdHJpbmc7XG5cbiAgICAgIC8qKlxuICAgICAgICogVGhlIHB1Ymxpc2ggZGF0ZSBvZiB0aGUgdmVyc2lvbi5cbiAgICAgICAqL1xuICAgICAgcmVhZG9ubHkgcHVibGlzaGVkQXQ6IERhdGU7XG5cbiAgICAgIC8qKlxuICAgICAgICogVGhlc2UgcGVuZGluZyBwYWNrYWdlcyBhcmUgTkVWRVIgYXZhaWxhYmxlIGF0IHRoaXMgcG9pbnQuXG4gICAgICAgKi9cbiAgICAgIGF2YWlsYWJsZUF0OiB1bmRlZmluZWQ7XG4gICAgfTtcbiAgfTtcbn1cblxuLyoqXG4gKiBNYWtlcyBhIHJlcXVlc3QgdG8gdGhlIHByb3ZpZGVkIFVSTCBhbmQgcmV0dXJucyB0aGUgcmVzcG9uc2UgYWZ0ZXIgaGF2aW5nXG4gKiBwYXJzZWQgaXQgZnJvbSBKU09OLlxuICpcbiAqIEBwYXJhbSB1cmwgdGhlIFVSTCB0byBnZXQuXG4gKiBAcGFyYW0ganNvblBhdGggYSBKU09OIHBhdGggdG8gZXh0cmFjdCBvbmx5IGEgc3Vic2V0IG9mIHRoZSBvYmplY3QuXG4gKi9cbmZ1bmN0aW9uIGdldEpTT04odXJsOiBzdHJpbmcsIGpzb25QYXRoPzogc3RyaW5nW10pOiBQcm9taXNlPGFueT4ge1xuICByZXR1cm4gbmV3IFByb21pc2UoKG9rLCBrbykgPT4ge1xuICAgIGh0dHBzLmdldCh1cmwsIHsgaGVhZGVyczogeyAnQWNjZXB0JzogJ2FwcGxpY2F0aW9uL2pzb24nLCAnQWNjZXB0LUVuY29kaW5nJzogJ2lkZW50aXR5JyB9IH0sIChyZXMpID0+IHtcbiAgICAgIGlmIChyZXMuc3RhdHVzQ29kZSAhPT0gMjAwKSB7XG4gICAgICAgIGNvbnN0IGVycm9yID0gbmV3IEVycm9yKGBHRVQgJHt1cmx9IC0gSFRUUCAke3Jlcy5zdGF0dXNDb2RlfSAoJHtyZXMuc3RhdHVzTWVzc2FnZX0pYCk7XG4gICAgICAgIEVycm9yLmNhcHR1cmVTdGFja1RyYWNlKGVycm9yKTtcbiAgICAgICAgcmV0dXJuIGtvKGVycm9yKTtcbiAgICAgIH1cblxuICAgICAgcmVzLm9uY2UoJ2Vycm9yJywga28pO1xuXG4gICAgICBjb25zdCBqc29uID0gSlNPTlN0cmVhbS5wYXJzZShqc29uUGF0aCk7XG4gICAgICBqc29uLm9uY2UoJ2RhdGEnLCBvayk7XG4gICAgICBqc29uLm9uY2UoJ2Vycm9yJywga28pO1xuXG4gICAgICBjb25zdCBwbGFpblBheWxvYWQgPSByZXMuaGVhZGVyc1snY29udGVudC1lbmNvZGluZyddID09PSAnZ3ppcCcgPyBndW56aXAocmVzKSA6IHJlcztcbiAgICAgIHBsYWluUGF5bG9hZC5waXBlKGpzb24sIHsgZW5kOiB0cnVlIH0pO1xuICAgIH0pO1xuICB9KTtcbn1cblxuLyoqXG4gKiBVcGRhdGVzIHRoZSBgbGF0ZXN0YCBwcm9wZXJ0eSBvZiBgc3RhdGVgIHRpIHRoZSBwcm92aWRlZCBgbGF0ZXN0YCB2YWx1ZSxcbiAqIHVubGVzcyB0aGlzIGlzIGFscmVhZHkgdGhlIGN1cnJlbnQgbGF0ZXN0LlxuICpcbiAqIElmIHRoZSBwcmV2aW91cyBsYXRlc3QgdmVyc2lvbiBkb2VzIG5vdCBoYXZlIHRoZSBgYXZhaWxhYmxlQXRgIHByb3BlcnR5LCBhZGRzXG4gKiB0aGF0IHRvIHRoZSBgcGVuZGluZ2Agc2V0LlxuICpcbiAqIEBwYXJhbSBzdGF0ZSAgdGhlIHN0YXRlIHRvIGJlIHVwZGF0ZWQuXG4gKiBAcGFyYW0gbGF0ZXN0IHRoZSBjdXJyZW50IFwibGF0ZXN0XCIgdmVyc2lvbiBvZiB0aGUgdHJhY2tlZCBwYWNrYWdlLlxuICovXG5mdW5jdGlvbiB1cGRhdGVMYXRlc3RJZk5lZWRlZChzdGF0ZTogQ2FuYXJ5U3RhdGUsIGxhdGVzdDogQ2FuYXJ5U3RhdGVbJ2xhdGVzdCddKTogdm9pZCB7XG4gIGlmIChzdGF0ZS5sYXRlc3QudmVyc2lvbiA9PT0gbGF0ZXN0LnZlcnNpb24pIHtcbiAgICByZXR1cm47XG4gIH1cblxuICAvLyBJZiB0aGUgY3VycmVudCBcImxhdGVzdFwiIGlzbid0IGF2YWlsYWJsZSB5ZXQsIGFkZCBpdCB0byB0aGUgYHBlbmRpbmdgIHZlcnNpb25zLlxuICBpZiAoc3RhdGUubGF0ZXN0LmF2YWlsYWJsZUF0ID09IG51bGwpIHtcbiAgICAvLyBUaGUgVHlwZVNjcmlwdCB2ZXJzaW9uIG9mIGpzaWkgZG9lc24ndCBkbyBjb250cm9sIGZsb3cgYW5hbHlzaXMgd2VsbCBlbm91Z2ggaGVyZSB0b1xuICAgIC8vIGRldGVybWluZSB0aGF0IHRoZWBpZmAgYnJhbmNoIGd1YXJhbnRlZXMgYGF2YWlsYWJsZUF0YCBpcyB1bmRlZmluZWQgaGVyZS5cbiAgICBzdGF0ZS5wZW5kaW5nW3N0YXRlLmxhdGVzdC52ZXJzaW9uXSA9IHsgLi4uc3RhdGUubGF0ZXN0LCBhdmFpbGFibGVBdDogdW5kZWZpbmVkIH07XG4gIH1cblxuICBzdGF0ZS5sYXRlc3QgPSBsYXRlc3Q7XG59XG5cbmZ1bmN0aW9uIGd1bnppcChyZWFkYWJsZTogUmVhZGFibGUpOiBSZWFkYWJsZSB7XG4gIGNvbnN0IGd6ID0gY3JlYXRlR3VuemlwKCk7XG4gIHJlYWRhYmxlLnBpcGUoZ3osIHsgZW5kOiB0cnVlIH0pO1xuICByZXR1cm4gZ3o7XG59XG4iXX0=