"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../../backend/shared/tarball.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const DETAIL_TYPE = 'CodeArtifact Package Version State Change';
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    metrics.setNamespace(constants_lambda_shared_1.METRICS_NAMESPACE);
    metrics.setDimensions({
        [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: event.detail.domainOwner,
        [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: event.detail.domainName,
        [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: event.detail.repositoryName,
    });
    const packageName = event.detail.packageNamespace
        ? `@${event.detail.packageNamespace}/${event.detail.packageName}`
        : event.detail.packageName;
    if (event.detail.operationType === 'Deleted') {
        console.log(`Operation type is "Deleted" for ${packageName}@${event.detail.packageVersion}`);
        metrics.putMetric("DeletedCount" /* DELETED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const denied = denyList.lookup(packageName, event.detail.packageVersion);
    if (denied) {
        console.log(`Package ${packageName}@${event.detail.packageVersion} denied: ${JSON.stringify(denied, null, 2)}`);
        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const { asset, packageVersionRevision } = await aws.codeArtifact().getPackageVersionAsset({
        asset: 'package.tgz',
        domainOwner: event.detail.domainOwner,
        domain: event.detail.domainName,
        repository: event.detail.repositoryName,
        format: event.detail.packageFormat,
        namespace: event.detail.packageNamespace,
        package: event.detail.packageName,
        packageVersion: event.detail.packageVersion,
    }).promise();
    const tarball = Buffer.from(asset);
    const { assemblyJson, packageJson } = await tarball_lambda_shared_1.extractObjects(tarball, {
        assemblyJson: { path: 'package/.jsii' },
        packageJson: { path: 'package/package.json', required: true },
    });
    metrics.putMetric("NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */, assemblyJson ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (assemblyJson == null) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not contain a .jsii assembly`);
        return;
    }
    const metadata = JSON.parse(packageJson.toString('utf-8'));
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const eligibleLicense = licenseList.lookup((_a = metadata.license) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    metrics.putMetric("IneligibleLicenseCount" /* INELIGIBLE_LICENSE */, eligibleLicense ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (!eligibleLicense) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not use allow-listed license: ${(_b = metadata.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
        return;
    }
    const stagingKey = `${packageName}/${event.detail.packageVersion}/${packageVersionRevision}/package.tgz`;
    await aws.s3().putObject({
        Bucket: stagingBucket,
        Key: stagingKey,
        Body: asset,
        ContentType: 'application/octet-stream',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
    const message = integrity_lambda_shared_1.integrity({
        tarballUri: `s3://${stagingBucket}/${stagingKey}`,
        metadata: { resources: event.resources.join(', ') },
        time: event.time,
    }, tarball);
    return aws.sqs().sendMessage({
        MessageAttributes: {
            AWS_REQUEST_ID: { DataType: 'String', StringValue: context.awsRequestId },
            LOG_GROUP_NAME: { DataType: 'String', StringValue: context.logGroupName },
            LOG_STREAM_NAME: { DataType: 'String', StringValue: context.logStreamName },
        },
        MessageBody: JSON.stringify(message),
        MessageDeduplicationId: event.detail.eventDeduplicationId,
        QueueUrl: queueUrl,
    }).promise();
});
//# sourceMappingURL=data:application/json;base64,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