import os
import subprocess
from distutils import sysconfig
import platform
from pathlib import Path
import rust_circuit
from interp.circuit.testing.topdown_circuit import st_Circuit, CircuitProbs as CP
from interp.circuit.circuit import Circuit
from interp.circuit.interop_rust import py_to_rust
from interp.circuit.print_circuit import PrintCircuit
import hypothesis
import hypothesis.strategies as st
import hypothesis.extra.numpy as st_np

"""
This builds a Cargo project and a bunch of tests to check print::rust_expression_notation_circuit is correct

This function serializes a Rust circuit to the Rust source code that makes a circuit of that structure,
so to test it we need to invoke the Rust compiler
"""


def generate_rust_test(path_to_rust_circuit: Path, file_contents: str):
    cargo_toml = f"""
    [package]
    name = "test_rust_circuit_lambda"
    version = "0.1.0"
    edition = "2021"

    [dependencies]
    rust_circuit = {{ path = "{path_to_rust_circuit}", default-features = false }}
    """

    toolchain_file = """
    [toolchain]
    channel = "nightly-2022-08-04"
    """

    # we write to a gitignore'd sibling folder so that builds are cached
    dirpath = path_to_rust_circuit.parent / ".generated_test_project"
    dirpath.mkdir(exist_ok=True)
    with (dirpath / "Cargo.toml").open("w") as f:
        f.write(cargo_toml)

    with (dirpath / "rust-toolchain.toml").open("w") as f:
        f.write(toolchain_file)

    (dirpath / "src").mkdir(exist_ok=True)
    with (dirpath / "src" / "lib.rs").open("w") as f:
        f.write(file_contents)

    # get libpython3.XX.dylib location (breaks on Mac otherwise)
    # and put in the right environment variable (https://doc.rust-lang.org/cargo/reference/environment-variables.html#dynamic-library-paths)
    varname = {
        "Darwin": "DYLD_FALLBACK_LIBRARY_PATH",
        "Windows": "PATH",
        "Linux": "LD_LIBRARY_PATH",
    }[platform.uname().system]
    # folder containing libpython (eg /path/to/anaconda3/lib for conda)
    dylib_dir = sysconfig.get_config_var("LIBDIR")
    env = os.environ.copy()
    env[varname] = env[varname] + os.pathsep + dylib_dir if varname in env else dylib_dir
    # `cargo test`
    p = subprocess.run(["cargo", "test", "--", "--test-threads", "1"], cwd=dirpath, env=env)
    assert (
        p.returncode == 0
    ), "Scroll up to see stdout/stderr. For better output, try running `python test_rust_printing.py` locally"


def sample_circuits(n: int) -> list[Circuit]:
    ret = []

    @hypothesis.settings(max_examples=n)
    @hypothesis.given(data=st.data())
    def helper(data) -> list[Circuit]:
        source = st_Circuit(
            st_np.array_shapes(min_dims=0),
            probs_default=CP.kw(all=1, Cumulant=0),
            must_be_explicitly_computable=True,
            must_be_constant=True,
        )
        nonlocal ret
        ret.append(data.draw(source))

    helper()
    return ret


def escape_str_rs(s: str) -> str:
    return s.replace('"', '\\"').replace("{", "{{").replace("}", "}}")


def build_test(circuit: Circuit, name: str):
    PrintCircuit()(circuit)
    rcircuit = py_to_rust(circuit)
    # get the code gen from the input circuit before re-randomizing
    rs_code_gen = rust_circuit.rust_expression_notation_circuit(rcircuit)
    # re-randomize tensors with the same seed as we'll get in Rust
    rcircuit = rust_circuit.replace_all_randn_seeded(rcircuit)
    # use the hash of the randomized tensor
    py_hash = rcircuit.hash  # list of ints
    return f"""
    #[test]
    fn {name}() {{
        pyo3::prepare_freethreaded_python();
        let result: CircuitRc = {rs_code_gen};

        let py_hash: HashBytes = {py_hash};
        let rs_hash: HashBytes = result.info().hash;

        assert!(py_hash == rs_hash, "Circuit hashes did not match.\nRust print output: {escape_str_rs(rs_code_gen)}\nPrinting compiled printed output: {{}}", print::rust_expression_notation_circuit(result));
    }}
    """


def test_rust_printing():
    path_to_rust_circuit = Path.cwd()
    circuits_to_test = sample_circuits(1000)
    test_methods = [build_test(circuit, f"test_{i}") for i, circuit in enumerate(circuits_to_test)]
    file_contents = """
    ///! THIS FILE IS AUTOGENERATED - DO NOT EDIT
    ///! To change tests, see `test_rust_printing.py`

    #[allow(unused_imports)]

    // Import anything a test might need
    use rust_circuit::all_imports::*;

    #[test]
    fn test_tests_work() {
        let a = ScalarConstant::new(34., Shape::new(), None);
        println!("{:?}", a);
    }

    """ + "\n".join(
        test_methods
    )

    generate_rust_test(path_to_rust_circuit, file_contents)


if __name__ == "__main__":
    test_rust_printing()
