import os
from typing import Tuple, Union, Optional

from pydantic import ValidationError
from yaml import safe_load

from tktl.core.exceptions.exceptions import (
    NoContentsFoundException,
    UserRepoValidationException,
)
from tktl.core.schemas.project import (
    ProjectFileWithContent,
    TktlYamlConfigSchema,
    ProjectDirectory,
    ProjectFile,
    ProjectContentMultiItemT,
    ProjectContentT,
)
from tktl.core.t import (
    RequiredUserProjectPathsT,
    UserProjectFileT,
    UserRepoConfigFileNameT,
)


def get_user_repo_contents(path: str) -> Optional[ProjectContentT]:
    contents = []
    files = [os.path.join(path, f) for f in os.listdir(path)]
    for item in files:
        if os.path.isfile(item):
            contents.append(ProjectFile(name=os.path.basename(item), path=item))
        elif os.path.isdir(item):
            contents.append(ProjectDirectory(name=os.path.basename(item), path=item))
    return contents


def validate_project_contents(
    path: str,
) -> Tuple[TktlYamlConfigSchema, ProjectContentMultiItemT]:
    contents = get_user_repo_contents(path=path)
    config_found = False
    config_file = None
    required_files = RequiredUserProjectPathsT.strictly_required_files()
    required_dirs = RequiredUserProjectPathsT.strictly_required_dirs()

    if not contents:
        raise NoContentsFoundException(
            missing_config=True,
            missing_directories=list(required_dirs),
            missing_files=list(required_files),
        )

    for file in contents:
        if file.type == UserProjectFileT.FILE.value:
            if file.name in UserRepoConfigFileNameT.set():
                config_found = True
                config_file = file
            elif file.name in required_files:
                required_files.remove(file.name)
        elif file.type == UserProjectFileT.DIRECTORY.value:
            if file.name in required_dirs:
                required_dirs.remove(file.name)

    if not config_found or required_dirs or required_files:
        raise UserRepoValidationException(
            missing_files=list(required_files),
            missing_directories=list(required_dirs),
            missing_config=not config_found,
        )
    config_file_with_content = decode_file_content(config_file)
    return config_file_with_content, contents


def decode_file_content(
    file_with_content: ProjectFileWithContent,
) -> Union[TktlYamlConfigSchema, ValidationError]:
    with open(file_with_content.path, "r") as yml:
        as_dict = safe_load(yml)
    return TktlYamlConfigSchema(**as_dict)
