"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Http302Backend = void 0;
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const apigw = require("@aws-cdk/aws-apigatewayv2");
const integrations = require("@aws-cdk/aws-apigatewayv2-integrations");
const cf = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
class Http302Backend extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const api = new apigw.HttpApi(this, 'Api', {
            defaultIntegration: new integrations.LambdaProxyIntegration({
                handler: new lambda.Function(this, 'ApiHandler', {
                    runtime: lambda.Runtime.PYTHON_3_7,
                    handler: 'index.handler',
                    code: new lambda.InlineCode(`
def handler(event, context):
      return {
        'statusCode': 302,
        'headers': {
          'location': 'https://www.google.com',
        },
      }`),
                }),
            }),
        });
        this.endpoint = api.apiEndpoint;
        const stack = cdk.Stack.of(this);
        this.domainName = `${api.apiId}.execute-api.${stack.region}.${stack.urlSuffix}`;
    }
}
exports.Http302Backend = Http302Backend;
test('basic minimal setting', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'http302-integ');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const ext = new extensions.HTTP302FromOrigin(stack, 'http302');
    const backend = new Http302Backend(stack, 'Http302Backend');
    // create the cloudfront distribution with extension(s)
    new cf.Distribution(stack, 'dist', {
        defaultBehavior: {
            origin: new origins.HttpOrigin(backend.domainName),
            cachePolicy: cf.CachePolicy.CACHING_DISABLED,
            edgeLambdas: [ext],
        },
    });
    // THEN
    // match snapshot
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    // match custom lambda function version
    expect(stack).toHaveResourceLike('AWS::Lambda::Version', {
        FunctionName: {
            Ref: 'HTTP302FromOriginFuncFCF29B3D',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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