"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const rewriteUriDemo = new extensions.DefaultDirIndex(stack, 'DefaultDirIndexDemo');
    // create Demo S3 Bucket.
    const bucket = new s3.Bucket(rewriteUriDemo, 'demoBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
        websiteIndexDocument: 'index.html',
        websiteErrorDocument: 'index.html',
    });
    // Put demo Object to Bucket.
    new aws_s3_deployment_1.BucketDeployment(rewriteUriDemo, 'Deployment', {
        sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, '.'))],
        destinationBucket: bucket,
        retainOnDelete: false,
    });
    // CloudFront OriginAccessIdentity for Bucket
    const originAccessIdentity = new cf.OriginAccessIdentity(rewriteUriDemo, 'OriginAccessIdentity', {
        comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
    });
    // CloudfrontWebDistribution
    const cloudfrontWebDistribution = new cf.CloudFrontWebDistribution(stack, 'CloudFrontWebDistribution', {
        enableIpV6: false,
        originConfigs: [
            {
                s3OriginSource: {
                    originAccessIdentity,
                    s3BucketSource: bucket,
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        lambdaFunctionAssociations: [rewriteUriDemo],
                    }],
            },
        ],
    });
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: cloudfrontWebDistribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                ],
                CachedMethods: [
                    'GET',
                    'HEAD',
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: 'none',
                    },
                    QueryString: false,
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-request',
                        LambdaFunctionARN: {
                            Ref: 'DefaultDirIndexFuncCurrentVersion61C934367319e2ee8e68fee60f04764644690803',
                        },
                    },
                ],
                TargetOriginId: 'origin1',
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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