"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const customErrorPage = new extensions.CustomErrorPage(stack, 'CustomErrorPage');
    // create s3 bucket
    const bucket = new s3.Bucket(customErrorPage, 'demoBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
        websiteIndexDocument: 'index.html',
        websiteErrorDocument: 'error.html',
    });
    // Put demo Object to Bucket.
    new aws_s3_deployment_1.BucketDeployment(customErrorPage, 'Deployment', {
        sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
        destinationBucket: bucket,
        retainOnDelete: false,
    });
    // cloudFront OriginAccessIdentity for bucket
    const originAccessIdentity = new cf.OriginAccessIdentity(customErrorPage, 'OriginAccessIdentity', {
        comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
    });
    // cloudfront distribution
    const distribution = new cf.CloudFrontWebDistribution(stack, 'distribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    originAccessIdentity,
                    s3BucketSource: bucket,
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        defaultTtl: cdk.Duration.seconds(10),
                        lambdaFunctionAssociations: [customErrorPage],
                    }],
            },
        ],
    });
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: distribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                ],
                CachedMethods: [
                    'GET',
                    'HEAD',
                ],
                Compress: true,
                DefaultTTL: 10,
                ForwardedValues: {
                    Cookies: {
                        Forward: 'none',
                    },
                    QueryString: false,
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-response',
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: 'CustomErrorPageFuncCurrentVersion76C88CF00d354f88ac9cbca9ef78e282fb833740',
                        },
                    },
                ],
                TargetOriginId: 'origin1',
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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