"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            // alarms that come from watchful are all considered normal severity
            alarmActionArns: ((_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.normalSeverity) ? [this.alarmActions.normalSeverity] : [],
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'HighSeverityDashboard');
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        var _a;
        const highSeverityAction = (_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.highSeverity;
        if (highSeverityAction) {
            alarm.addAlarmAction({
                bind: () => ({ alarmActionArn: highSeverityAction }),
            });
        }
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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