"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
const EFS_MOUNT_PATH = '/mnt/efs';
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            TARGET_LANGUAGE: props.language.toString(),
            // Override $TMPDIR to be on the EFS volume (so we are not limited to 512MB)
            TMPDIR: EFS_MOUNT_PATH,
            // Configure a fixed directory in the EFS volume where we share npm caches
            NPM_CACHE: `${EFS_MOUNT_PATH}${Transliterator.SHARED_NPM_CACHE_PATH}`,
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Resource', {
            description: `Creates ${props.language} documentation from jsii-enabled npm packages`,
            environment,
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        this.function = lambda;
        // TODO: The @aws-cdk/aws-lambda library does not support EFS mounts yet T_T
        lambda.node.defaultChild.addPropertyOverride('FileSystemConfigs', [{
                Arn: props.efsAccessPoint.accessPointArn,
                LocalMountPath: EFS_MOUNT_PATH,
            }]);
        props.efsAccessPoint.fileSystem.connections.allowFrom(lambda, aws_ec2_1.Port.allTraffic());
        if (props.vpcEndpoints) {
            props.vpcEndpoints.elasticFileSystem.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['elasticfilesystem:ClientMount', 'elasticfilesystem:ClientWrite'],
                conditions: {
                    Bool: { 'aws:SecureTransport': 'true' },
                    ArnEquals: { 'elasticfilesystem:AccessPointArn': props.efsAccessPoint.accessPointArn },
                },
                principals: [lambda.grantPrincipal],
                resources: [props.efsAccessPoint.fileSystem.fileSystemArn],
            }));
        }
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.function);
        // The handler reads & writes to this bucket.
        bucket.grantRead(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}${constants.NOT_SUPPORTED_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
    }
}
exports.Transliterator = Transliterator;
/**
 * The path under which the npm cache will be located, within the EFS mount.
 */
Transliterator.SHARED_NPM_CACHE_PATH = '/npm-cache';
//# sourceMappingURL=data:application/json;base64,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