"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const clean_up_efs_1 = require("./clean-up-efs");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Failed to add to catalog.json', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue));
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        const efsAccessPoint = this.newEfsAccessPoint(props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
            lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, efsAccessPoint, language }).function,
            outputPath: '$.result',
            resultSelector: {
                result: {
                    'language': language,
                    'success.$': '$.Payload',
                },
            },
        }).addRetry({ interval: core_1.Duration.seconds(30) })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Failed ${language}`, {
            parameters: {
                'error.$': 'States.StringToJson($.Cause)',
                language,
            },
        }))))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // Ensure the State Machine does not get to run before the VPC can be used.
        this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
    newEfsAccessPoint(props) {
        const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
            encrypted: true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            throughputMode: aws_efs_1.ThroughputMode.BURSTING,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const efsAccessPoint = fs.addAccessPoint('AccessPoint', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '0777',
            },
            path: '/lambda-shared',
            posixUser: {
                uid: '1000',
                gid: '1000',
            },
        });
        efsAccessPoint.node.addDependency(fs.mountTargetsAvailable);
        const efsMountPath = '/mnt/efs';
        const cleanUp = new clean_up_efs_1.CleanUpEfs(this, 'EFSCleanUp', {
            description: '[ConstructHub/CleanUpEFS] Cleans up leftover files from an EFS file system',
            environment: {
                EFS_MOUNT_PATH: efsMountPath,
                IGNORE_DIRS: `${efsMountPath}${transliterator_1.Transliterator.SHARED_NPM_CACHE_PATH}`,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        // TODO: The @aws-cdk/aws-lambda library does not support EFS mounts yet T_T
        cleanUp.node.defaultChild.addPropertyOverride('FileSystemConfigs', [{
                Arn: efsAccessPoint.accessPointArn,
                LocalMountPath: efsMountPath,
            }]);
        fs.connections.allowFrom(cleanUp, aws_ec2_1.Port.allTraffic());
        const rule = new aws_events_1.Rule(this, 'CleanUpEFSTrigger', { description: `Runs ${cleanUp.functionName} every hour`, schedule: aws_events_1.Schedule.rate(core_1.Duration.hours(1)) });
        rule.addTarget(new aws_events_targets_1.LambdaFunction(cleanUp));
        if (props.vpcEndpoints) {
            props.vpcEndpoints.elasticFileSystem.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['elasticfilesystem:ClientMount', 'elasticfilesystem:ClientWrite'],
                conditions: {
                    Bool: { 'aws:SecureTransport': 'true' },
                    ArnEquals: { 'elasticfilesystem:AccessPointArn': efsAccessPoint.accessPointArn },
                },
                principals: [cleanUp.grantPrincipal],
                resources: [fs.fileSystemArn],
            }));
        }
        return efsAccessPoint;
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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