"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
/**
 * Deletes items older than 15 minutes from the directory denoted by
 * EFS_MOUNT_PATH, keeping any item included in subfolders of IGNORE_DIRS (a `:`
 * separated list of directory names).
 */
async function handler(event, _context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const mountPath = path_1.resolve(env_lambda_shared_1.requireEnv('EFS_MOUNT_PATH'));
    const ignoreDirs = new Set(env_lambda_shared_1.requireEnv('IGNORE_DIRS').split(' ').map((item) => path_1.resolve(item)));
    for await (const [directory, date] of leftOverDirectories(mountPath, ignoreDirs)) {
        console.log(`Removing leftover directory (${date.toISOString()}): ${directory}`);
        // Using fs-extra, so this is recursive, and itempotently succeeds if a file us concurrently deleted
        await fs_extra_1.remove(directory);
    }
}
exports.handler = handler;
const FIFTEEN_MINUTES_AS_MILLIS = 900000 /* ms */;
/**
 * Yields a list of all first-level subdirectories of the provided `root` that
 * are empty or recursively contain only files older than the specified
 * timestamp, expressed as milliseconds since the epoch.
 *
 * @param root the root directory to scan.
 * @param ignoreDirs a set of directories to ignore when scanning.
 * @param olderThan the boundary timestamp, as milliseconds since the epoch.
 *                  Defaults to 15 minutes ago (which is the maximum timeout of
 *                  a Lambda function).
 */
async function* leftOverDirectories(root, ignoreDirs, olderThan = Date.now() - FIFTEEN_MINUTES_AS_MILLIS) {
    for (const file of await fs_extra_1.readdir(root)) {
        const fullPath = path_1.resolve(root, file);
        // Using lstat so we can expire symlinks correctly.
        const stat = await fs_extra_1.lstat(fullPath);
        if (stat.isDirectory()) {
            // If the directory is in the ignore list, never yield for deletion
            if (ignoreDirs.has(fullPath)) {
                console.log(`Path is in ignore list: ${fullPath}`);
                continue;
            }
            const newest = await newestFileIn(fullPath);
            if (newest < olderThan) {
                // All files are older than the threshold, so we yield it for deletion
                yield [fullPath, new Date(newest)];
            }
        }
        else if (stat.isFile() || stat.isSymbolicLink()) {
            console.log(`Ignoring non-directory entity: ${fullPath}`);
        }
    }
}
async function newestFileIn(dir) {
    let result;
    for (const file of await fs_extra_1.readdir(dir)) {
        const fullPath = path_1.resolve(dir, file);
        const stat = await fs_extra_1.lstat(fullPath);
        if (stat.isFile() || stat.isSymbolicLink()) {
            if (result == null || result < stat.mtimeMs) {
                result = stat.mtimeMs;
            }
        }
        else if (stat.isDirectory()) {
            const children = await newestFileIn(fullPath);
            if (result == null || result < children) {
                result = children;
            }
        }
        else {
            // Weird file type, we'll leave it there...
            console.log(`Unexpected file type: ${fullPath}...`);
            result = Number.MAX_SAFE_INTEGER;
        }
    }
    // If result is undefined, it means the directory is empty, so we return
    // the EPOCH timestamp (i.e: 0 milliseconds after... EPOCH).
    return result !== null && result !== void 0 ? result : 0;
}
//# sourceMappingURL=data:application/json;base64,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