"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(15);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 2 minutes ahead of the timeout.
        const handler = new discovery_1.Discovery(this, 'Default', {
            description: '[ConstructHub/Discovery] Periodically query npm.js index for new construct libraries',
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        this.bucket.grantReadWrite(this.function);
        props.queue.grantSendMessages(this.function);
        props.denyList.grantRead(handler);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(this.function)],
        });
        this.alarmErrors = this.function.metricErrors({ period: core_1.Duration.minutes(15) }).createAlarm(this, 'ErrorsAlarm', {
            alarmName: `${this.node.path}/Errors`,
            alarmDescription: [
                'The discovery function (on npmjs.com) failed to run',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = this.function.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The discovery function (on npmjs.com) is not running as scheduled',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        props.monitoring.addHighSeverityAlarm('Discovery Failures', this.alarmErrors);
        props.monitoring.addHighSeverityAlarm('Discovery not Running', this.alarmNoInvocations);
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of staging failures.
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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