"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants_1 = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos, metrics);
                    metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }, metrics) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            metrics.putMetric("StagingFailureCount" /* STAGING_FAILURE_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
            await putObject(`${"failed/" /* FAILED_KEY_PREFIX */}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'application/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList) {
    var _a;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = usesEligibleLicenses(infos);
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (usesEligibleLicenses(infos)) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED'}`);
                    }
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
    function usesEligibleLicenses({ license }) {
        var _a;
        return constants_1.ELIGIBLE_LICENSES.has((_a = license === null || license === void 0 ? void 0 : license.toUpperCase()) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    }
}
//# sourceMappingURL=data:application/json;base64,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