"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const monitoring_1 = require("../monitoring");
const http_get_function_lambda_1 = require("../monitoring/http-get-function.lambda");
require("@aws-cdk/assert/jest");
const actions = {
    highSeverity: 'arn:aws:sns:us-east-1:123456789012:high',
    normalSeverity: 'arn:aws:sns:us-east-1:123456789012:normal',
};
test('minimal', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
    });
    // a dashboard is automatically created
    expect(stack).toHaveResource('AWS::CloudWatch::Dashboard');
});
test('watchful can be used for setting up automatic monitoring', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'Function', {
        runtime: lambda.Runtime.NODEJS_12_X,
        code: lambda.Code.fromInline('foo'),
        handler: 'index.handler',
    });
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
    });
    // WHEN
    monitoring.watchful.watchLambdaFunction('My Function', fn);
    // an alarm is automatically created for this function
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Lambda',
        MetricName: 'Errors',
        //TODO: uncomment when we can use cdk-watchful 0.145.0 or above
        // AlarmActions: ['arn:aws:sns:us-east-1:123456789012:normal'],
        Dimensions: [{ Name: 'FunctionName', Value: { Ref: 'Function76856677' } }],
    });
});
test('high severity alarms trigger the correct action', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
    });
    const alarm = topic.metricNumberOfNotificationsFailed().createAlarm(stack, 'Alarm', { threshold: 1, evaluationPeriods: 1 });
    // WHEN
    monitoring.addHighSeverityAlarm('My Alarm', alarm);
    // a dashboard is automatically created
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        AlarmActions: ['arn:aws:sns:us-east-1:123456789012:high'],
        Dimensions: [{ Name: 'TopicName', Value: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } }],
    });
    expect(stack).toHaveResource('AWS::CloudWatch::Dashboard', {
        DashboardBody: {
            'Fn::Join': [
                '',
                [
                    '{"widgets":[{"type":"metric","width":24,"height":6,"x":0,"y":0,"properties":{"view":"timeSeries","title":"My Alarm","region":"',
                    { Ref: 'AWS::Region' },
                    '","annotations":{"alarms":["',
                    { 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] },
                    '"]},"yAxis":{}}}]}',
                ],
            ],
        },
    });
});
test('web canaries can ping URLs and raise high severity alarms', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
    });
    // WHEN
    monitoring.addWebCanary('Ping1', 'https://ping1');
    // THEN
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanOrEqualToThreshold',
        EvaluationPeriods: 1,
        AlarmActions: [
            'arn:aws:sns:us-east-1:123456789012:high',
        ],
        AlarmDescription: '80% error rate for https://ping1 (Ping1)',
        Metrics: [
            {
                Id: 'm1',
                Label: 'https://ping1 Errors',
                MetricStat: {
                    Metric: {
                        Dimensions: [
                            {
                                Name: 'FunctionName',
                                Value: {
                                    Ref: 'MonitoringWebCanaryPing1HttpGetFunction2403444A',
                                },
                            },
                        ],
                        MetricName: 'Errors',
                        Namespace: 'AWS/Lambda',
                    },
                    Period: 300,
                    Stat: 'Sum',
                },
                ReturnData: true,
            },
        ],
        Threshold: 4,
        TreatMissingData: 'breaching',
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                URL: 'https://ping1',
            },
        },
    });
});
describe('web canary handler', () => {
    test('web ping is successful', async () => {
        process.env.URL = 'https://amazon.com';
        await http_get_function_lambda_1.handler({});
    });
    test('web ping throws for a non-200 response', async () => {
        process.env.URL = 'https://amazon.com/not-found-please12345';
        await expect(http_get_function_lambda_1.handler({})).rejects.toThrow(/Response code 404 \(Not Found\)/);
    });
});
//# sourceMappingURL=data:application/json;base64,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