"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const language_1 = require("../../../backend/shared/language");
const transliterator_1 = require("../../../backend/transliterator");
const repository_1 = require("../../../codeartifact/repository");
const monitoring_1 = require("../../../monitoring");
test('basic use', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        efsAccessPoint: efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
    });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc, efsFileSystem, efsAccessPoint),
    });
});
test('CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    const codeArtifact = new repository_1.Repository(stack, 'CodeArtifact');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact,
        efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: codeArtifact.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: codeArtifact.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: codeArtifact.repositoryNpmEndpoint,
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, codeArtifact, vpc, efsFileSystem, efsAccessPoint),
    });
});
test('VPC Endpoints', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const efsAccessPoint = new aws_efs_1.FileSystem(stack, 'EFS', { vpc, vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED } })
        .addAccessPoint('EFS-AP');
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const elasticFileSystem = vpc.addInterfaceEndpoint('EFS', {
        service: aws_ec2_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        efsAccessPoint: efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, elasticFileSystem, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc),
    });
});
test('VPC Endpoints and CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const repository = new repository_1.Repository(stack, 'CodeArtifact');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc, vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED } });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const elasticFileSystem = vpc.addInterfaceEndpoint('EFS', {
        service: aws_ec2_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact: repository,
        efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, elasticFileSystem, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: repository.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: repository.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: repository.repositoryNpmEndpoint,
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, repository, monitoring, vpc, efsFileSystem, efsAccessPoint),
    });
});
/**
 * Creates a property matcher that expects "anything" for all CfnResource
 * logicalIds found within the provided constructs.
 */
function ignoreResources(stack, ...constructs) {
    const result = {};
    for (const resource of resourcesIn(...constructs)) {
        result[stack.resolve(resource.logicalId)] = expect.anything();
    }
    return result;
    function* resourcesIn(...scopes) {
        for (const scope of scopes) {
            if (core_1.CfnResource.isCfnResource(scope)) {
                yield scope;
            }
            yield* resourcesIn(...scope.node.children);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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