"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const pip_1 = require("./pip");
const poetry_1 = require("./poetry");
const projenrc_1 = require("./projenrc");
const pytest_1 = require("./pytest");
const pytest_sample_1 = require("./pytest-sample");
const python_sample_1 = require("./python-sample");
const setuptools_1 = require("./setuptools");
const venv_1 = require("./venv");
const github_1 = require("../github");
const projenrc_2 = require("../javascript/projenrc");
const project_1 = require("../project");
const util_1 = require("../util");
/** Allowed characters in python project names */
const PYTHON_PROJECT_NAME_REGEX = /^[A-Za-z0-9-_\.]+$/;
/**
 * Python project.
 *
 * @pjid python
 */
class PythonProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        if (!PYTHON_PROJECT_NAME_REGEX.test(options.name)) {
            throw new Error("Python projects must only consist of alphanumeric characters, hyphens, and underscores.");
        }
        this.moduleName = options.moduleName;
        this.version = options.version;
        const rcFileTypeOptions = [
            options.projenrcPython,
            options.projenrcJs,
            options.projenrcJson,
        ];
        if (util_1.multipleSelected(rcFileTypeOptions)) {
            throw new Error("Only one of projenrcPython, projenrcJs, and projenrcJson can be selected.");
        }
        const poetry = options.poetry ?? false;
        const pip = options.pip ?? !poetry;
        const venv = options.venv ?? !poetry;
        const setuptools = options.setuptools ?? (!poetry && this.projectType === project_1.ProjectType.LIB);
        if (poetry && (pip || venv || setuptools)) {
            throw new Error("poetry is true - pip, venv, and setuptools must be undefined or false");
        }
        // default to projenrc.py if no other projenrc type was elected
        if (options.projenrcPython ?? !util_1.anySelected(rcFileTypeOptions)) {
            new projenrc_1.Projenrc(this, options.projenrcPythonOptions);
        }
        if (options.projenrcJs ?? false) {
            new projenrc_2.Projenrc(this, options.projenrcJsOptions);
        }
        if (venv) {
            this.envManager = new venv_1.Venv(this, options.venvOptions);
        }
        if (pip) {
            this.depsManager = new pip_1.Pip(this);
        }
        if (setuptools) {
            this.packagingManager = new setuptools_1.Setuptools(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                setupConfig: options.setupConfig,
            });
        }
        // if (options.conda ?? false) {
        //   this.depsManager = new Conda(this, options);
        //   this.envManager = this.depsManager;
        // }
        // if (options.pipenv ?? false) {
        //   this.depsManager = new Pipenv(this, options);
        //   this.envManager = this.depsManager;
        // }
        if (poetry) {
            const poetryProject = new poetry_1.Poetry(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                poetryOptions: {
                    readme: options.readme?.filename ?? "README.md",
                    ...options.poetryOptions,
                },
            });
            this.depsManager = poetryProject;
            this.envManager = poetryProject;
            this.packagingManager = poetryProject;
        }
        if (!this.envManager) {
            throw new Error("At least one tool must be chosen for managing the environment (venv, conda, pipenv, or poetry).");
        }
        if (!this.depsManager) {
            throw new Error("At least one tool must be chosen for managing dependencies (pip, conda, pipenv, or poetry).");
        }
        if (!this.packagingManager && this.projectType === project_1.ProjectType.LIB) {
            throw new Error("At least one tool must be chosen for managing packaging (setuptools or poetry).");
        }
        if (Number(venv) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing the environment (venv, conda, pipenv, or poetry)");
        }
        if (Number(pip) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing dependencies (pip, conda, pipenv, or poetry)");
        }
        if (Number(setuptools) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing packaging (setuptools or poetry)");
        }
        if (options.pytest ?? true) {
            this.pytest = new pytest_1.Pytest(this, options.pytestOptions);
            new pytest_sample_1.PytestSample(this, {
                moduleName: this.moduleName,
                testdir: this.pytest.testdir,
            });
        }
        if (options.sample ?? true) {
            new python_sample_1.PythonSample(this, {
                dir: this.moduleName,
            });
        }
        for (const dep of options.deps ?? []) {
            this.addDependency(dep);
        }
        for (const dep of options.devDeps ?? []) {
            this.addDevDependency(dep);
        }
        this.addDefaultGitIgnore();
    }
    /**
     * Adds default gitignore options for a Python project based on
     * https://github.com/github/gitignore/blob/master/Python.gitignore
     */
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Byte-compiled / optimized / DLL files", "__pycache__/", "*.py[cod]", "*$py.class", "", "# C extensions", "*.so", "", "# Distribution / packaging", ".Python", "build/", "develop-eggs/", "dist/", "downloads/", "eggs/", ".eggs/", "lib/", "lib64/", "parts/", "sdist/", "var/", "wheels/", "share/python-wheels/", "*.egg-info/", ".installed.cfg", "*.egg", "MANIFEST", "", "# PyInstaller", "#  Usually these files are written by a python script from a template", "#  before PyInstaller builds the exe, so as to inject date/other infos into it.", "*.manifest", "*.spec", "", "# Installer logs", "pip-log.txt", "pip-delete-this-directory.txt", "", "# Unit test / coverage reports", "htmlcov/", ".tox/", ".nox/", ".coverage", ".coverage.*", ".cache", "nosetests.xml", "coverage.xml", "*.cover", "*.py,cover", ".hypothesis/", ".pytest_cache/", "cover/", "", "# Translations", "*.mo", "*.pot", "", "# Django stuff:", "*.log", "local_settings.py", "db.sqlite3", "db.sqlite3-journal", "", "# Flask stuff:", "instance/", ".webassets-cache", "", "# Scrapy stuff:", ".scrapy", "", "# Sphinx documentation", "docs/_build/", "", "# PyBuilder", ".pybuilder/", "target/", "", "# Jupyter Notebook", ".ipynb_checkpoints", "", "# IPython", "profile_default/", "ipython_config.py", "", "# PEP 582; used by e.g. github.com/David-OConnor/pyflow", "__pypackages__/", "", "# Celery stuff", "celerybeat-schedule", "celerybeat.pid", "", "# SageMath parsed files", "*.sage.py", "", "# Environments", ".env", ".venv", "env/", "venv/", "ENV/", "env.bak/", "venv.bak/", "", "# Spyder project settings", ".spyderproject", ".spyproject", "", "# Rope project settings", ".ropeproject", "", "# mkdocs documentation", "/site", "", "# mypy", ".mypy_cache/", ".dmypy.json", "dmypy.json", "", "# Pyre type checker", ".pyre/", "", "# pytype static type analyzer", ".pytype/", "", "# Cython debug symbols", "cython_debug/");
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        return this.depsManager.addDependency(spec);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        return this.depsManager.addDevDependency(spec);
    }
    postSynthesize() {
        super.postSynthesize();
        this.envManager.setupEnvironment();
        this.depsManager.installDependencies();
    }
}
exports.PythonProject = PythonProject;
_a = JSII_RTTI_SYMBOL_1;
PythonProject[_a] = { fqn: "projen.python.PythonProject", version: "0.67.21" };
//# sourceMappingURL=data:application/json;base64,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