"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const release_1 = require("../release");
function context(value) {
    return `\${{ ${value} }}`;
}
const REPO = context("github.repository");
const RUN_ID = context("github.run_id");
const RUN_URL = `https://github.com/${REPO}/actions/runs/${RUN_ID}`;
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        project.addDevDeps("npm-check-updates@^16");
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        const exclude = this.options.exclude ?? [];
        // exclude depedencies that has already version pinned (fully or with patch version) by Projen with ncu (but not package manager upgrade)
        // Getting only unique values through set
        const ncuExcludes = [
            ...new Set(this.project.deps.all
                .filter((dep) => dep.version &&
                dep.version[0] !== "^" &&
                dep.type !== dependencies_1.DependencyType.OVERRIDE)
                .map((dep) => dep.name)
                .concat(exclude)),
        ];
        const ncuIncludes = this.options.include?.filter((item) => !ncuExcludes.includes(item));
        const includeLength = this.options.include?.length ?? 0;
        const ncuIncludesLength = ncuIncludes?.length ?? 0;
        // If all explicit includes already have version pinned, don't add task.
        // Note that without explicit includes task gets added
        if (includeLength > 0 && ncuIncludesLength === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        const steps = new Array();
        // update npm-check-updates before everything else, in case there is a bug
        // in it or one of its dependencies. This will make upgrade workflows
        // slightly more stable and resilient to upstream changes.
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand([], ["npm-check-updates"]),
        });
        for (const dep of ["dev", "optional", "peer", "prod", "bundle"]) {
            const ncuCommand = [
                "npm-check-updates",
                "--dep",
                dep,
                "--upgrade",
                "--target=minor",
            ];
            // Don't add includes and excludes same time
            if (ncuIncludes) {
                ncuCommand.push(`--filter='${ncuIncludes.join(",")}'`);
            }
            else if (ncuExcludes.length > 0) {
                ncuCommand.push(`--reject='${ncuExcludes.join(",")}'`);
            }
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this._project.package.renderUpgradePackagesCommand(exclude, this.options.include),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, branch) {
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        // thats all we should need at this stage since all we do is clone.
        // note that this also prevents new code that is introduced in the upgrade
        // to have write access to anything, in case its somehow executed. (for example during build)
        const permissions = {
            contents: github_1.workflows.JobPermission.READ,
        };
        const steps = [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: branch ? { ref: branch } : undefined,
            },
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.createUploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: permissions,
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        // default to API access method used by all GitHub workflows, unless a
        // custom one is specified
        const apiAccess = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const token = apiAccess.tokenRef;
        const runsOn = this.options.workflowOptions?.runsOn ?? ["ubuntu-latest"];
        const workflowName = workflow.name;
        const branchName = `github-actions/${workflowName}`;
        const prStepId = "create-pr";
        const title = `chore(deps): ${this.pullRequestTitle}`;
        const description = [
            "Upgrades project dependencies. See details in [workflow run].",
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflow.name}" workflow*`,
        ].join("\n");
        const committer = `${this.gitIdentity.name} <${this.gitIdentity.email}>`;
        const steps = [
            ...apiAccess.setupSteps,
            ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                ref: upgrade.ref,
            }),
            ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
            {
                name: "Create Pull Request",
                id: prStepId,
                uses: "peter-evans/create-pull-request@v4",
                with: {
                    // the pr can modify workflow files, so we need to use the custom
                    // secret if one is configured.
                    token: token,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    title: title,
                    labels: this.options.workflowOptions?.labels?.join(",") || undefined,
                    assignees: this.options.workflowOptions?.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: this.options.signoff ?? true,
                },
            },
        ];
        return {
            job: {
                name: "Create Pull Request",
                if: `\${{ needs.${upgrade.jobId}.outputs.${PATCH_CREATED_OUTPUT} }}`,
                needs: [upgrade.jobId],
                permissions: {
                    contents: github_1.workflows.JobPermission.READ,
                },
                runsOn: runsOn ?? ["ubuntu-latest"],
                steps: steps,
            },
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.67.21" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    constructor(cron) {
        this.cron = cron;
    }
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.67.21" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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