"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const common_1 = require("../common");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/util");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const release_1 = require("../release");
const util_2 = require("../util");
const version_1 = require("../version");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease = exports.AutoRelease || (exports.AutoRelease = {}));
/**
 * Node.js project
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.artifactsJavascriptDirectory = path_1.join(this.artifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", '$(npx -c "node -e \\"console.log(process.env.PATH)\\"")');
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore");
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_RC}`);
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        this.gitignore.include(`/${common_1.PROJEN_RC}`);
        const projen = options.projenDevDependency ?? true;
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        if (buildEnabled && this.github) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                preBuildSteps: this.renderWorkflowSetup({
                    mutable: options.mutableBuild ?? true,
                }),
                postBuildSteps: options.postBuildSteps,
                runsOn: options.workflowRunsOn,
                workflowTriggers: options.buildWorkflowTriggers,
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.addDevDeps(version_1.Version.STANDARD_VERSION);
            this.release = new release_1.Release(this, {
                versionFile: "package.json",
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                artifactsDirectory: this.artifactsDirectory,
                ...options,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({ mutable: false }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
            });
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = release_1.isAwsCodeArtifactRegistry(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot(util_2.deepMerge([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, util_2.deepMerge([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (projenrcJs) {
            new projenrc_1.Projenrc(this, options.projenrcJsOptions);
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        if (options.package ?? true) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            // always use npm here - yarn doesn't add much value
            // sadly we cannot use --pack-destination because it is not supported by older npm
            this.packageTask.exec(`mv $(npm pack) ${this.artifactsJavascriptDirectory}/`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version required by this pacakge.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v3",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
        };
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v1",
                    with: {
                        "aws-access-key-id": util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.nodeVersion) {
            install.push({
                name: "Setup Node.js",
                uses: "actions/setup-node@v3",
                with: { "node-version": this.nodeVersion },
            });
        }
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.2",
                with: { version: "7" },
            });
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.67.21" };
//# sourceMappingURL=data:application/json;base64,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