"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.defaultNpmToken = exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = exports.CodeArtifactAuthProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const url_1 = require("url");
const fs_extra_1 = require("fs-extra");
const util_1 = require("./util");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const util_2 = require("../util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider = exports.CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = {}));
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = options.packageName ?? project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = options.allowLibraryDependencies ?? true;
        this.packageManager = options.packageManager ?? NodePackageManager.YARN;
        this.entrypoint = options.entrypoint ?? "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, scopedPackagesOptions, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.scopedPackagesOptions = scopedPackagesOptions;
        this.processDeps(options);
        this.addCodeArtifactLoginScript();
        const prev = this.readPackageJson() ?? {};
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => this.license ?? UNLICENSED,
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(prev?.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // override any scripts from options (if specified)
        for (const [cmdname, shell] of Object.entries(options.scripts ?? {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this.project, "package.json", {
            obj: this.manifest,
            readonly: false,
            newline: true,
            committed: true,
        });
        this.addKeywords(...(options.keywords ?? []));
        this.addBin(options.bin ?? {});
        // automatically add all executable files under "bin"
        if (options.autoDetectBin ?? true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.addNodeEngine();
        // license
        if (options.licensed ?? true) {
            this.license = options.license ?? "Apache-2.0";
        }
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Override the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.file.addOverride(`scripts.${name}`, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.file.addDeletionOverride(`scripts.${name}`);
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`.
     */
    addPackageResolutions(...resolutions) {
        const fieldName = util_1.packageResolutionsFieldName(this.packageManager);
        for (const resolution of resolutions) {
            this.project.deps.addDependency(resolution, dependencies_1.DependencyType.OVERRIDE);
            const { name, version = "*" } = dependencies_1.Dependencies.parseDependency(resolution);
            this.file.addOverride(`${fieldName}.${name}`, version);
        }
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(exclude, include) {
        const project = this.project;
        function upgradePackages(command) {
            return () => {
                if (exclude.length === 0 && !include) {
                    // request to upgrade all packages
                    // separated for asthetic reasons.
                    return command;
                }
                // filter by exclude and include.
                return `${command} ${project.deps.all
                    .filter((d) => d.type !== dependencies_1.DependencyType.OVERRIDE)
                    .map((d) => d.name)
                    .filter((d) => (include ? include.includes(d) : true))
                    .filter((d) => !exclude.includes(d))
                    .join(" ")}`;
            };
        }
        let lazy = undefined;
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                lazy = upgradePackages("yarn upgrade");
                break;
            case NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    // ---------------------------------------------------------------------------------------
    preSynthesize() {
        super.preSynthesize();
        this._renderedDeps = this.renderDependencies();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !fs_extra_1.existsSync(path_1.join(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return util_2.isTruthy(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion ?? "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return util_2.isTruthy(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = url_1.parse(npmRegistryUrl ?? DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = options.npmAccess ?? defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const isAwsCodeArtifact = release_1.isAwsCodeArtifactRegistry(npmRegistryUrl);
        const hasScopedPackage = options.scopedPackagesOptions &&
            options.scopedPackagesOptions.length !== 0;
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
            else if (options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC) {
                if (options.codeArtifactOptions.accessKeyIdSecret ||
                    options.codeArtifactOptions.secretAccessKeySecret) {
                    throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
                }
                else if (!options.codeArtifactOptions.roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
            }
        }
        else {
            if ((options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret ||
                options.codeArtifactOptions?.roleToAssume) &&
                !hasScopedPackage) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact or used in scoped packages.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact || hasScopedPackage) {
            const authProvider = options.codeArtifactOptions?.authProvider ??
                CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            const isAccessSecretKeyPairAuth = authProvider === CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            codeArtifactOptions = {
                authProvider,
                accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_ACCESS_KEY_ID" : undefined),
                secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_SECRET_ACCESS_KEY" : undefined),
                roleToAssume: options.codeArtifactOptions?.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
            scopedPackagesOptions: this.parseScopedPackagesOptions(options.scopedPackagesOptions),
        };
    }
    parseScopedPackagesOptions(scopedPackagesOptions) {
        if (!scopedPackagesOptions) {
            return undefined;
        }
        return scopedPackagesOptions.map((option) => {
            if (!isScoped(option.scope)) {
                throw new Error(`Scope must start with "@" in options, found ${option.scope}`);
            }
            if (!release_1.isAwsCodeArtifactRegistry(option.registryUrl)) {
                throw new Error(`Only AWS Code artifact scoped registry is supported for now, found ${option.registryUrl}`);
            }
            const result = {
                registryUrl: option.registryUrl,
                scope: option.scope,
            };
            return result;
        });
    }
    addCodeArtifactLoginScript() {
        if (!this.scopedPackagesOptions ||
            this.scopedPackagesOptions.length === 0) {
            return;
        }
        this.project.addTask("ca:login", {
            requiredEnv: ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
            steps: [
                { exec: "which aws" },
                ...this.scopedPackagesOptions.map((scopedPackagesOption) => {
                    const { registryUrl, scope } = scopedPackagesOption;
                    const { domain, region, accountId, registry } = util_1.extractCodeArtifactDetails(registryUrl);
                    // reference: https://docs.aws.amazon.com/codeartifact/latest/ug/npm-auth.html
                    const commands = [
                        `npm config set ${scope}:registry ${registryUrl}`,
                        `CODEARTIFACT_AUTH_TOKEN=$(aws codeartifact get-authorization-token --domain ${domain} --region ${region} --domain-owner ${accountId} --query authorizationToken --output text)`,
                        `npm config set //${registry}:_authToken=$CODEARTIFACT_AUTH_TOKEN`,
                        `npm config set //${registry}:always-auth=true`,
                    ];
                    return {
                        exec: commands.join("; "),
                    };
                }),
            ],
        });
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
                return [
                    "yarn install",
                    "--check-files",
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return frozen
                    ? "pnpm i --frozen-lockfile"
                    : "pnpm i --no-frozen-lockfile";
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        this.addDeps(...(options.deps ?? []));
        this.addDevDeps(...(options.devDeps ?? []));
        this.addPeerDeps(...(options.peerDeps ?? []));
        this.addBundledDeps(...(options.bundledDeps ?? []));
    }
    renderDependencies() {
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // skip if we already have a runtime dependency on this peer
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME)) {
                    continue;
                }
                if (dep.version) {
                    const ver = util_1.minVersion(dep.version);
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            let version = dep.version ?? "*";
            let name = dep.name;
            if (name.startsWith("file:")) {
                const localDependencyPath = name.substring(5);
                const depPackageJson = path_1.resolve(this.project.outdir, localDependencyPath, "package.json");
                const pkgFile = fs_1.readFileSync(depPackageJson, "utf8");
                const pkg = JSON.parse(pkgFile);
                version = localDependencyPath;
                name = pkg.name;
            }
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(name);
                    if (this.project.deps.all.find((d) => d.name === name && d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${name}". it cannot appear as a peer dependency`);
                    }
                    // also add as a runtime dependency
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => util_2.sorted(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = util_2.sorted(bundledDependencies);
        // nothing further to do if package.json file does not exist
        const pkg = this.readPackageJson();
        if (!pkg) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current ?? {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, pkg.devDependencies);
        readDeps(dependencies, pkg.dependencies);
        readDeps(peerDependencies, pkg.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        const outdir = this.project.outdir;
        const rootPackageJson = path_1.join(outdir, "package.json");
        const original = fs_1.readFileSync(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current ?? {};
            user = user ?? {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    try {
                        const modulePath = require.resolve(`${name}/package.json`, {
                            paths: [outdir],
                        });
                        const module = fs_extra_1.readJsonSync(modulePath);
                        desiredVersion = `^${module.version}`;
                    }
                    catch (e) { }
                    if (!desiredVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = util_1.minVersion(version);
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = util_2.sorted(deps);
        pkg.devDependencies = util_2.sorted(devDeps);
        pkg.peerDependencies = util_2.sorted(peerDeps);
        const updated = JSON.stringify(pkg, undefined, 2) + "\n";
        if (original === updated) {
            return false;
        }
        util_2.writeFile(rootPackageJson, updated);
        return true;
    }
    renderPublishConfig() {
        // omit values if they are the same as the npm defaults
        return _resolve_1.resolve({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: this.npmAccess !== defaultNpmAccess(this.packageName)
                ? this.npmAccess
                : undefined,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return util_2.sorted(kwds.sort());
    }
    renderEngines() {
        return util_2.sorted(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = path_1.join(this.project.outdir, binrel);
        if (fs_extra_1.existsSync(bindir)) {
            for (const file of fs_extra_1.readdirSync(bindir)) {
                try {
                    fs_extra_1.accessSync(path_1.join(bindir, file), fs_extra_1.constants.X_OK);
                    this.bin[file] = path_1.join(binrel, file).replace(/\\/g, "/");
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: options.authorOrganization ?? false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return util_2.sorted(this.bin);
    }
    renderScripts() {
        const result = {};
        for (const task of this.project.tasks.all.sort((x, y) => x.name.localeCompare(y.name))) {
            result[task.name] = this.npmScriptForTask(task);
        }
        return result;
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = path_1.join(this.project.outdir, "package.json");
        if (!fs_extra_1.existsSync(file)) {
            return undefined;
        }
        return fs_extra_1.readJsonSync(file);
    }
    installDependencies() {
        util_2.exec(this.renderInstallCommand(this.isAutomatedBuild), {
            cwd: this.project.outdir,
        });
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.67.21" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
})(NodePackageManager = exports.NodePackageManager || (exports.NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess = exports.NpmAccess || (exports.NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if (release_1.isAwsCodeArtifactRegistry(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken ??
        (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
exports.defaultNpmToken = defaultNpmToken;
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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