"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const PULL_REQUEST_REF = "${{ github.event.pull_request.head.ref }}";
const PULL_REQUEST_REPOSITORY = "${{ github.event.pull_request.head.repo.full_name }}";
const BUILD_JOBID = "build";
const SELF_MUTATION_STEP = "self_mutation";
const SELF_MUTATION_HAPPENED_OUTPUT = "self_mutation_happened";
const IS_FORK = "github.event.pull_request.head.repo.full_name != github.repository";
const NOT_FORK = `!(${IS_FORK})`;
const SELF_MUTATION_CONDITION = `needs.${BUILD_JOBID}.outputs.${SELF_MUTATION_HAPPENED_OUTPUT}`;
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.defaultRunners = ["ubuntu-latest"];
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = options.preBuildSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        const mutableBuilds = options.mutableBuild ?? true;
        this.workflow = new github_1.GithubWorkflow(github, "build");
        this.workflow.on(options.workflowTriggers ?? {
            pullRequest: {},
            workflowDispatch: {},
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
        if (project instanceof javascript_1.NodeProject) {
            project.addPackageIgnore(constants_1.PERMISSION_BACKUP_FILE);
        }
    }
    addBuildJob(options) {
        this.workflow.addJob(BUILD_JOBID, {
            runsOn: options.runsOn ?? this.defaultRunners,
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            steps: (() => this.renderBuildSteps()),
            outputs: {
                [SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        });
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            steps.push({
                name: "Download build artifacts",
                uses: "actions/download-artifact@v3",
                with: {
                    name: constants_1.BUILD_ARTIFACT_NAME,
                    path: this.artifactsDirectory,
                },
            }, {
                name: "Restore build artifact permissions",
                continueOnError: true,
                run: [
                    `cd ${this.artifactsDirectory} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                ].join("\n"),
            });
        }
        steps.push(...(job.steps ?? []));
        this.workflow.addJob(id, {
            needs: [BUILD_JOBID],
            // only run if build did not self-mutate
            if: `! ${SELF_MUTATION_CONDITION}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options = {}) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            runsOn: options.runsOn,
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        const steps = [];
        if (options?.checkoutRepo) {
            steps.push({
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            });
        }
        if (options?.checkoutRepo &&
            options?.installDeps &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options?.tools,
            runsOn: options?.runsOn ?? this.defaultRunners,
            steps,
        });
    }
    addSelfMutationJob(options) {
        this.workflow.addJob("self-mutation", {
            runsOn: options.runsOn ?? this.defaultRunners,
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [BUILD_JOBID],
            if: `always() && ${SELF_MUTATION_CONDITION} && ${NOT_FORK}`,
            steps: [
                ...this.workflow.projenCredentials.setupSteps,
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: this.workflow.projenCredentials.tokenRef,
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                }),
                ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
                {
                    name: "Push changes",
                    run: [
                        "  git add .",
                        '  git commit -s -m "chore: self mutation"',
                        `  git push origin HEAD:${PULL_REQUEST_REF}`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps() {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            },
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.createUploadGitPatch({
                stepId: SELF_MUTATION_STEP,
                outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Backup artifact permissions",
                        continueOnError: true,
                        run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
                    },
                    {
                        name: "Upload artifact",
                        uses: "actions/upload-artifact@v3",
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.artifactsDirectory,
                        },
                    },
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.67.21" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVpbGQtd29ya2Zsb3cuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvYnVpbGQvYnVpbGQtd29ya2Zsb3cudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSw0Q0FBeUM7QUFDekMsc0NBQWdFO0FBQ2hFLG1EQUk2QjtBQUM3QixpRUFBNkQ7QUFDN0QsK0RBTW1DO0FBQ25DLDhDQUE0QztBQUc1QyxNQUFNLGdCQUFnQixHQUFHLDJDQUEyQyxDQUFDO0FBQ3JFLE1BQU0sdUJBQXVCLEdBQzNCLHNEQUFzRCxDQUFDO0FBQ3pELE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQztBQUM1QixNQUFNLGtCQUFrQixHQUFHLGVBQWUsQ0FBQztBQUMzQyxNQUFNLDZCQUE2QixHQUFHLHdCQUF3QixDQUFDO0FBQy9ELE1BQU0sT0FBTyxHQUNYLG9FQUFvRSxDQUFDO0FBQ3ZFLE1BQU0sUUFBUSxHQUFHLEtBQUssT0FBTyxHQUFHLENBQUM7QUFDakMsTUFBTSx1QkFBdUIsR0FBRyxTQUFTLFdBQVcsWUFBWSw2QkFBNkIsRUFBRSxDQUFDO0FBdUVoRyxNQUFhLGFBQWMsU0FBUSxxQkFBUztJQVkxQyxZQUFZLE9BQWdCLEVBQUUsT0FBNkI7UUFDekQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBTEEsbUJBQWMsR0FBYSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBRTdDLG1CQUFjLEdBQWEsRUFBRSxDQUFDO1FBSzdDLE1BQU0sTUFBTSxHQUFHLGVBQU0sQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStELENBQ2hFLENBQUM7U0FDSDtRQUVELElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxhQUFhLEdBQUcsT0FBTyxDQUFDLGFBQWEsSUFBSSxFQUFFLENBQUM7UUFDakQsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLENBQUMsY0FBYyxJQUFJLEVBQUUsQ0FBQztRQUNuRCxJQUFJLENBQUMsV0FBVyxHQUFHLE9BQU8sQ0FBQyxXQUFXLElBQUksdUNBQTJCLENBQUM7UUFDdEUsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUMsU0FBUyxDQUFDO1FBQ25DLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxPQUFPLENBQUMsa0JBQWtCLENBQUM7UUFDckQsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUM7UUFFbkQsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHVCQUFjLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUNkLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSTtZQUMxQixXQUFXLEVBQUUsRUFBRTtZQUNmLGdCQUFnQixFQUFFLEVBQUU7U0FDckIsQ0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUUxQixJQUFJLGFBQWEsRUFBRTtZQUNqQixJQUFJLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDbEM7UUFFRCxJQUFJLE9BQU8sWUFBWSx3QkFBVyxFQUFFO1lBQ2xDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxrQ0FBc0IsQ0FBQyxDQUFDO1NBQ2xEO0lBQ0gsQ0FBQztJQUVPLFdBQVcsQ0FBQyxPQUE2QjtRQUMvQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUU7WUFDaEMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLGNBQWM7WUFDN0MsU0FBUyxFQUFFLE9BQU8sQ0FBQyxjQUFjO2dCQUMvQixDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsT0FBTyxDQUFDLGNBQWMsRUFBRTtnQkFDbkMsQ0FBQyxDQUFDLFNBQVM7WUFDYixHQUFHLEVBQUU7Z0JBQ0gsRUFBRSxFQUFFLE1BQU07Z0JBQ1YsR0FBRyxPQUFPLENBQUMsR0FBRzthQUNmO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLFFBQVEsRUFBRSwrQkFBYSxDQUFDLEtBQUs7YUFDOUI7WUFDRCxLQUFLLEVBQUUsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBUTtZQUM3QyxPQUFPLEVBQUU7Z0JBQ1AsQ0FBQyw2QkFBNkIsQ0FBQyxFQUFFO29CQUMvQixNQUFNLEVBQUUsa0JBQWtCO29CQUMxQixVQUFVLEVBQUUsNkJBQTZCO2lCQUMxQzthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sQ0FBQyxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGlCQUFpQixDQUFDLEdBQUcsS0FBZ0I7UUFDMUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWUsQ0FBQyxFQUFVLEVBQUUsR0FBUTtRQUN6QyxNQUFNLEtBQUssR0FBRyxFQUFFLENBQUM7UUFFakIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDM0IsS0FBSyxDQUFDLElBQUksQ0FDUjtnQkFDRSxJQUFJLEVBQUUsMEJBQTBCO2dCQUNoQyxJQUFJLEVBQUUsOEJBQThCO2dCQUNwQyxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLCtCQUFtQjtvQkFDekIsSUFBSSxFQUFFLElBQUksQ0FBQyxrQkFBa0I7aUJBQzlCO2FBQ0YsRUFDRDtnQkFDRSxJQUFJLEVBQUUsb0NBQW9DO2dCQUMxQyxlQUFlLEVBQUUsSUFBSTtnQkFDckIsR0FBRyxFQUFFO29CQUNILE1BQU0sSUFBSSxDQUFDLGtCQUFrQix5QkFBeUIsa0NBQXNCLEVBQUU7aUJBQy9FLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQzthQUNiLENBQ0YsQ0FBQztTQUNIO1FBRUQsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRWpDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRTtZQUN2QixLQUFLLEVBQUUsQ0FBQyxXQUFXLENBQUM7WUFDcEIsd0NBQXdDO1lBQ3hDLEVBQUUsRUFBRSxLQUFLLHVCQUF1QixFQUFFO1lBQ2xDLEdBQUcsR0FBRztZQUNOLEtBQUssRUFBRSxLQUFLO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsbUNBQW1DO1FBQ25DLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxtQkFBbUIsQ0FDeEIsSUFBVSxFQUNWLFVBQXNDLEVBQUU7UUFFeEMsSUFBSSxDQUFDLHVCQUF1QixDQUMxQixjQUFjLElBQUksQ0FBQyxJQUFJLEVBQUUsRUFDekIsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUM5QztZQUNFLFlBQVksRUFBRSxJQUFJO1lBQ2xCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLEtBQUssRUFBRSxPQUFPLENBQUMsS0FBSztZQUNwQixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU07U0FDdkIsQ0FDRixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVCQUF1QixDQUM1QixFQUFVLEVBQ1YsUUFBa0IsRUFDbEIsT0FBd0M7UUFFeEMsTUFBTSxLQUFLLEdBQUcsRUFBRSxDQUFDO1FBRWpCLElBQUksT0FBTyxFQUFFLFlBQVksRUFBRTtZQUN6QixLQUFLLENBQUMsSUFBSSxDQUFDO2dCQUNULElBQUksRUFBRSxVQUFVO2dCQUNoQixJQUFJLEVBQUUscUJBQXFCO2dCQUMzQixJQUFJLEVBQUU7b0JBQ0osR0FBRyxFQUFFLGdCQUFnQjtvQkFDckIsVUFBVSxFQUFFLHVCQUF1QjtpQkFDcEM7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELElBQ0UsT0FBTyxFQUFFLFlBQVk7WUFDckIsT0FBTyxFQUFFLFdBQVc7WUFDcEIsSUFBSSxDQUFDLE9BQU8sWUFBWSx3QkFBVyxFQUNuQztZQUNBLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ1QsSUFBSSxFQUFFLHNCQUFzQjtnQkFDNUIsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFO2FBQzlDLENBQUMsQ0FBQztTQUNKO1FBRUQsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsRUFBRSxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUV6QyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRTtZQUN2QixXQUFXLEVBQUU7Z0JBQ1gsUUFBUSxFQUFFLCtCQUFhLENBQUMsSUFBSTthQUM3QjtZQUNELEtBQUssRUFBRSxPQUFPLEVBQUUsS0FBSztZQUNyQixNQUFNLEVBQUUsT0FBTyxFQUFFLE1BQU0sSUFBSSxJQUFJLENBQUMsY0FBYztZQUM5QyxLQUFLO1NBQ04sQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGtCQUFrQixDQUFDLE9BQTZCO1FBQ3RELElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLGVBQWUsRUFBRTtZQUNwQyxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsY0FBYztZQUM3QyxXQUFXLEVBQUU7Z0JBQ1gsUUFBUSxFQUFFLCtCQUFhLENBQUMsS0FBSzthQUM5QjtZQUNELEtBQUssRUFBRSxDQUFDLFdBQVcsQ0FBQztZQUNwQixFQUFFLEVBQUUsZUFBZSx1QkFBdUIsT0FBTyxRQUFRLEVBQUU7WUFDM0QsS0FBSyxFQUFFO2dCQUNMLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVO2dCQUM3QyxHQUFHLGtDQUFlLENBQUMsaUJBQWlCLENBQUM7b0JBQ25DLHdFQUF3RTtvQkFDeEUsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsUUFBUTtvQkFDL0MsR0FBRyxFQUFFLGdCQUFnQjtvQkFDckIsVUFBVSxFQUFFLHVCQUF1QjtpQkFDcEMsQ0FBQztnQkFDRixHQUFHLGtDQUFlLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7Z0JBQ25EO29CQUNFLElBQUksRUFBRSxjQUFjO29CQUNwQixHQUFHLEVBQUU7d0JBQ0gsYUFBYTt3QkFDYiwyQ0FBMkM7d0JBQzNDLDBCQUEwQixnQkFBZ0IsRUFBRTtxQkFDN0MsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO2lCQUNiO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxnQkFBZ0I7UUFDdEIsT0FBTztZQUNMO2dCQUNFLElBQUksRUFBRSxVQUFVO2dCQUNoQixJQUFJLEVBQUUscUJBQXFCO2dCQUMzQixJQUFJLEVBQUU7b0JBQ0osR0FBRyxFQUFFLGdCQUFnQjtvQkFDckIsVUFBVSxFQUFFLHVCQUF1QjtpQkFDcEM7YUFDRjtZQUVELEdBQUcsSUFBSSxDQUFDLGFBQWE7WUFFckI7Z0JBQ0UsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSTtnQkFDekIsR0FBRyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDO2FBQ3hEO1lBRUQsR0FBRyxJQUFJLENBQUMsY0FBYztZQUV0QixpRUFBaUU7WUFDakUsR0FBRyxrQ0FBZSxDQUFDLG9CQUFvQixDQUFDO2dCQUN0QyxNQUFNLEVBQUUsa0JBQWtCO2dCQUMxQixVQUFVLEVBQUUsNkJBQTZCO2dCQUN6QyxhQUFhLEVBQ1gsMEhBQTBIO2FBQzdILENBQUM7WUFFRiwwRkFBMEY7WUFDMUYsR0FBRyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxJQUFJLENBQUM7Z0JBQ2pDLENBQUMsQ0FBQyxFQUFFO2dCQUNKLENBQUMsQ0FBQztvQkFDRTt3QkFDRSxJQUFJLEVBQUUsNkJBQTZCO3dCQUNuQyxlQUFlLEVBQUUsSUFBSTt3QkFDckIsR0FBRyxFQUFFLE1BQU0sSUFBSSxDQUFDLGtCQUFrQixzQkFBc0Isa0NBQXNCLEVBQUU7cUJBQ2pGO29CQUNEO3dCQUNFLElBQUksRUFBRSxpQkFBaUI7d0JBQ3ZCLElBQUksRUFBRSw0QkFBNEI7d0JBQ2xDLElBQUksRUFBRTs0QkFDSixJQUFJLEVBQUUsK0JBQW1COzRCQUN6QixJQUFJLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjt5QkFDOUI7cUJBQ0Y7aUJBQ0YsQ0FBQztTQUNQLENBQUM7SUFDSixDQUFDOztBQXRTSCxzQ0F1U0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBUYXNrIH0gZnJvbSBcIi4uXCI7XG5pbXBvcnQgeyBDb21wb25lbnQgfSBmcm9tIFwiLi4vY29tcG9uZW50XCI7XG5pbXBvcnQgeyBHaXRIdWIsIEdpdGh1YldvcmtmbG93LCBHaXRJZGVudGl0eSB9IGZyb20gXCIuLi9naXRodWJcIjtcbmltcG9ydCB7XG4gIEJVSUxEX0FSVElGQUNUX05BTUUsXG4gIERFRkFVTFRfR0lUSFVCX0FDVElPTlNfVVNFUixcbiAgUEVSTUlTU0lPTl9CQUNLVVBfRklMRSxcbn0gZnJvbSBcIi4uL2dpdGh1Yi9jb25zdGFudHNcIjtcbmltcG9ydCB7IFdvcmtmbG93QWN0aW9ucyB9IGZyb20gXCIuLi9naXRodWIvd29ya2Zsb3ctYWN0aW9uc1wiO1xuaW1wb3J0IHtcbiAgSm9iLFxuICBKb2JQZXJtaXNzaW9uLFxuICBKb2JTdGVwLFxuICBUb29scyxcbiAgVHJpZ2dlcnMsXG59IGZyb20gXCIuLi9naXRodWIvd29ya2Zsb3dzLW1vZGVsXCI7XG5pbXBvcnQgeyBOb2RlUHJvamVjdCB9IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQgeyBQcm9qZWN0IH0gZnJvbSBcIi4uL3Byb2plY3RcIjtcblxuY29uc3QgUFVMTF9SRVFVRVNUX1JFRiA9IFwiJHt7IGdpdGh1Yi5ldmVudC5wdWxsX3JlcXVlc3QuaGVhZC5yZWYgfX1cIjtcbmNvbnN0IFBVTExfUkVRVUVTVF9SRVBPU0lUT1JZID1cbiAgXCIke3sgZ2l0aHViLmV2ZW50LnB1bGxfcmVxdWVzdC5oZWFkLnJlcG8uZnVsbF9uYW1lIH19XCI7XG5jb25zdCBCVUlMRF9KT0JJRCA9IFwiYnVpbGRcIjtcbmNvbnN0IFNFTEZfTVVUQVRJT05fU1RFUCA9IFwic2VsZl9tdXRhdGlvblwiO1xuY29uc3QgU0VMRl9NVVRBVElPTl9IQVBQRU5FRF9PVVRQVVQgPSBcInNlbGZfbXV0YXRpb25faGFwcGVuZWRcIjtcbmNvbnN0IElTX0ZPUksgPVxuICBcImdpdGh1Yi5ldmVudC5wdWxsX3JlcXVlc3QuaGVhZC5yZXBvLmZ1bGxfbmFtZSAhPSBnaXRodWIucmVwb3NpdG9yeVwiO1xuY29uc3QgTk9UX0ZPUksgPSBgISgke0lTX0ZPUkt9KWA7XG5jb25zdCBTRUxGX01VVEFUSU9OX0NPTkRJVElPTiA9IGBuZWVkcy4ke0JVSUxEX0pPQklEfS5vdXRwdXRzLiR7U0VMRl9NVVRBVElPTl9IQVBQRU5FRF9PVVRQVVR9YDtcblxuZXhwb3J0IGludGVyZmFjZSBCdWlsZFdvcmtmbG93T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgdGFzayB0byBleGVjdXRlIGluIG9yZGVyIHRvIGJ1aWxkIHRoZSBwcm9qZWN0LlxuICAgKi9cbiAgcmVhZG9ubHkgYnVpbGRUYXNrOiBUYXNrO1xuXG4gIC8qKlxuICAgKiBBIG5hbWUgb2YgYSBkaXJlY3RvcnkgdGhhdCBpbmNsdWRlcyBidWlsZCBhcnRpZmFjdHMuXG4gICAqL1xuICByZWFkb25seSBhcnRpZmFjdHNEaXJlY3Rvcnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvbnRhaW5lciBpbWFnZSB0byB1c2UgZm9yIGJ1aWxkcy5cbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCB3b3JrZmxvdyBjb250YWluZXJcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRhaW5lckltYWdlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IHVwZGF0ZSBmaWxlcyBtb2RpZmllZCBkdXJpbmcgYnVpbGRzIHRvIHB1bGwtcmVxdWVzdCBicmFuY2hlcy5cbiAgICogVGhpcyBtZWFucyB0aGF0IGFueSBmaWxlcyBzeW50aGVzaXplZCBieSBwcm9qZW4gb3IgZS5nLiB0ZXN0IHNuYXBzaG90cyB3aWxsXG4gICAqIGFsd2F5cyBiZSB1cC10by1kYXRlIGJlZm9yZSBhIFBSIGlzIG1lcmdlZC5cbiAgICpcbiAgICogSW1wbGllcyB0aGF0IFBSIGJ1aWxkcyBkbyBub3QgaGF2ZSBhbnRpLXRhbXBlciBjaGVja3MuXG4gICAqXG4gICAqIFRoaXMgaXMgZW5hYmxlZCBieSBkZWZhdWx0IG9ubHkgaWYgYGdpdGh1YlRva2VuU2VjcmV0YCBpcyBzZXQuIE90aGVyd2lzZSBpdFxuICAgKiBpcyBkaXNhYmxlZCwgd2hpY2ggaW1wbGllcyB0aGF0IGZpbGUgY2hhbmdlcyB0aGF0IGhhcHBlbiBkdXJpbmcgYnVpbGQgd2lsbFxuICAgKiBub3QgYmUgcHVzaGVkIGJhY2sgdG8gdGhlIGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZUJ1aWxkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogU3RlcHMgdG8gZXhlY3V0ZSBiZWZvcmUgdGhlIGJ1aWxkLlxuICAgKiBAZGVmYXVsdCBbXVxuICAgKi9cbiAgcmVhZG9ubHkgcHJlQnVpbGRTdGVwcz86IEpvYlN0ZXBbXTtcblxuICAvKipcbiAgICogU3RlcHMgdG8gZXhlY3V0ZSBhZnRlciBidWlsZC5cbiAgICogQGRlZmF1bHQgW11cbiAgICovXG4gIHJlYWRvbmx5IHBvc3RCdWlsZFN0ZXBzPzogSm9iU3RlcFtdO1xuXG4gIC8qKlxuICAgKiBHaXQgaWRlbnRpdHkgdG8gdXNlIGZvciB0aGUgd29ya2Zsb3cuXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBpZGVudGl0eVxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0SWRlbnRpdHk/OiBHaXRJZGVudGl0eTtcblxuICAvKipcbiAgICogQnVpbGQgZW52aXJvbm1lbnQgdmFyaWFibGVzLlxuICAgKiBAZGVmYXVsdCB7fVxuICAgKi9cbiAgcmVhZG9ubHkgZW52PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICAvKipcbiAgICogR2l0aHViIFJ1bm5lciBzZWxlY3Rpb24gbGFiZWxzXG4gICAqIEBkZWZhdWx0IFtcInVidW50dS1sYXRlc3RcIl1cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnNPbj86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBCdWlsZCB3b3JrZmxvdyB0cmlnZ2Vyc1xuICAgKiBAZGVmYXVsdCBcInsgcHVsbFJlcXVlc3Q6IHt9LCB3b3JrZmxvd0Rpc3BhdGNoOiB7fSB9XCJcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93VHJpZ2dlcnM/OiBUcmlnZ2Vycztcbn1cblxuZXhwb3J0IGNsYXNzIEJ1aWxkV29ya2Zsb3cgZXh0ZW5kcyBDb21wb25lbnQge1xuICBwcml2YXRlIHJlYWRvbmx5IHBvc3RCdWlsZFN0ZXBzOiBKb2JTdGVwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlQnVpbGRTdGVwczogSm9iU3RlcFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGdpdElkZW50aXR5OiBHaXRJZGVudGl0eTtcbiAgcHJpdmF0ZSByZWFkb25seSBidWlsZFRhc2s6IFRhc2s7XG4gIHByaXZhdGUgcmVhZG9ubHkgZ2l0aHViOiBHaXRIdWI7XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3c6IEdpdGh1YldvcmtmbG93O1xuICBwcml2YXRlIHJlYWRvbmx5IGFydGlmYWN0c0RpcmVjdG9yeTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGRlZmF1bHRSdW5uZXJzOiBzdHJpbmdbXSA9IFtcInVidW50dS1sYXRlc3RcIl07XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfcG9zdEJ1aWxkSm9iczogc3RyaW5nW10gPSBbXTtcblxuICBjb25zdHJ1Y3Rvcihwcm9qZWN0OiBQcm9qZWN0LCBvcHRpb25zOiBCdWlsZFdvcmtmbG93T3B0aW9ucykge1xuICAgIHN1cGVyKHByb2plY3QpO1xuXG4gICAgY29uc3QgZ2l0aHViID0gR2l0SHViLm9mKHByb2plY3QpO1xuICAgIGlmICghZ2l0aHViKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiQnVpbGRXb3JrZmxvdyBpcyBjdXJyZW50bHkgb25seSBzdXBwb3J0ZWQgZm9yIEdpdEh1YiBwcm9qZWN0c1wiXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuZ2l0aHViID0gZ2l0aHViO1xuICAgIHRoaXMucHJlQnVpbGRTdGVwcyA9IG9wdGlvbnMucHJlQnVpbGRTdGVwcyA/PyBbXTtcbiAgICB0aGlzLnBvc3RCdWlsZFN0ZXBzID0gb3B0aW9ucy5wb3N0QnVpbGRTdGVwcyA/PyBbXTtcbiAgICB0aGlzLmdpdElkZW50aXR5ID0gb3B0aW9ucy5naXRJZGVudGl0eSA/PyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVI7XG4gICAgdGhpcy5idWlsZFRhc2sgPSBvcHRpb25zLmJ1aWxkVGFzaztcbiAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSA9IG9wdGlvbnMuYXJ0aWZhY3RzRGlyZWN0b3J5O1xuICAgIGNvbnN0IG11dGFibGVCdWlsZHMgPSBvcHRpb25zLm11dGFibGVCdWlsZCA/PyB0cnVlO1xuXG4gICAgdGhpcy53b3JrZmxvdyA9IG5ldyBHaXRodWJXb3JrZmxvdyhnaXRodWIsIFwiYnVpbGRcIik7XG4gICAgdGhpcy53b3JrZmxvdy5vbihcbiAgICAgIG9wdGlvbnMud29ya2Zsb3dUcmlnZ2VycyA/PyB7XG4gICAgICAgIHB1bGxSZXF1ZXN0OiB7fSxcbiAgICAgICAgd29ya2Zsb3dEaXNwYXRjaDoge30sIC8vIGFsbG93IG1hbnVhbCB0cmlnZ2VyaW5nXG4gICAgICB9XG4gICAgKTtcblxuICAgIHRoaXMuYWRkQnVpbGRKb2Iob3B0aW9ucyk7XG5cbiAgICBpZiAobXV0YWJsZUJ1aWxkcykge1xuICAgICAgdGhpcy5hZGRTZWxmTXV0YXRpb25Kb2Iob3B0aW9ucyk7XG4gICAgfVxuXG4gICAgaWYgKHByb2plY3QgaW5zdGFuY2VvZiBOb2RlUHJvamVjdCkge1xuICAgICAgcHJvamVjdC5hZGRQYWNrYWdlSWdub3JlKFBFUk1JU1NJT05fQkFDS1VQX0ZJTEUpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYWRkQnVpbGRKb2Iob3B0aW9uczogQnVpbGRXb3JrZmxvd09wdGlvbnMpIHtcbiAgICB0aGlzLndvcmtmbG93LmFkZEpvYihCVUlMRF9KT0JJRCwge1xuICAgICAgcnVuc09uOiBvcHRpb25zLnJ1bnNPbiA/PyB0aGlzLmRlZmF1bHRSdW5uZXJzLFxuICAgICAgY29udGFpbmVyOiBvcHRpb25zLmNvbnRhaW5lckltYWdlXG4gICAgICAgID8geyBpbWFnZTogb3B0aW9ucy5jb250YWluZXJJbWFnZSB9XG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgZW52OiB7XG4gICAgICAgIENJOiBcInRydWVcIixcbiAgICAgICAgLi4ub3B0aW9ucy5lbnYsXG4gICAgICB9LFxuICAgICAgcGVybWlzc2lvbnM6IHtcbiAgICAgICAgY29udGVudHM6IEpvYlBlcm1pc3Npb24uV1JJVEUsXG4gICAgICB9LFxuICAgICAgc3RlcHM6ICgoKSA9PiB0aGlzLnJlbmRlckJ1aWxkU3RlcHMoKSkgYXMgYW55LFxuICAgICAgb3V0cHV0czoge1xuICAgICAgICBbU0VMRl9NVVRBVElPTl9IQVBQRU5FRF9PVVRQVVRdOiB7XG4gICAgICAgICAgc3RlcElkOiBTRUxGX01VVEFUSU9OX1NURVAsXG4gICAgICAgICAgb3V0cHV0TmFtZTogU0VMRl9NVVRBVElPTl9IQVBQRU5FRF9PVVRQVVQsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIGpvYiBJRHMgdGhhdCBhcmUgcGFydCBvZiB0aGUgYnVpbGQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGJ1aWxkSm9iSWRzKCk6IHN0cmluZ1tdIHtcbiAgICByZXR1cm4gW0JVSUxEX0pPQklELCAuLi50aGlzLl9wb3N0QnVpbGRKb2JzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIHN0ZXBzIHRoYXQgYXJlIGV4ZWN1dGVkIGFmdGVyIHRoZSBidWlsZC5cbiAgICogQHBhcmFtIHN0ZXBzIFRoZSBqb2Igc3RlcHNcbiAgICovXG4gIHB1YmxpYyBhZGRQb3N0QnVpbGRTdGVwcyguLi5zdGVwczogSm9iU3RlcFtdKTogdm9pZCB7XG4gICAgdGhpcy5wb3N0QnVpbGRTdGVwcy5wdXNoKC4uLnN0ZXBzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFub3RoZXIgam9iIHRvIHRoZSBidWlsZCB3b3JrZmxvdyB3aGljaCBpcyBleGVjdXRlZCBhZnRlciB0aGUgYnVpbGRcbiAgICogam9iIHN1Y2NlZWRlZC5cbiAgICpcbiAgICogSm9icyBhcmUgZXhlY3V0ZWQgX29ubHlfIGlmIHRoZSBidWlsZCBkaWQgTk9UIHNlbGYgbXV0YXRlLiBJZiB0aGUgYnVpbGRcbiAgICogc2VsZi1tdXRhdGUsIHRoZSBicmFuY2ggd2lsbCBlaXRoZXIgYmUgdXBkYXRlZCBvciB0aGUgYnVpbGQgd2lsbCBmYWlsIChpblxuICAgKiBmb3JrcyksIHNvIHRoZXJlIGlzIG5vIHBvaW50IGluIGV4ZWN1dGluZyB0aGUgcG9zdC1idWlsZCBqb2IuXG4gICAqXG4gICAqIEBwYXJhbSBpZCBUaGUgaWQgb2YgdGhlIG5ldyBqb2JcbiAgICogQHBhcmFtIGpvYiBUaGUgam9iIHNwZWNpZmljYXRpb25cbiAgICovXG4gIHB1YmxpYyBhZGRQb3N0QnVpbGRKb2IoaWQ6IHN0cmluZywgam9iOiBKb2IpIHtcbiAgICBjb25zdCBzdGVwcyA9IFtdO1xuXG4gICAgaWYgKHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5KSB7XG4gICAgICBzdGVwcy5wdXNoKFxuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogXCJEb3dubG9hZCBidWlsZCBhcnRpZmFjdHNcIixcbiAgICAgICAgICB1c2VzOiBcImFjdGlvbnMvZG93bmxvYWQtYXJ0aWZhY3RAdjNcIixcbiAgICAgICAgICB3aXRoOiB7XG4gICAgICAgICAgICBuYW1lOiBCVUlMRF9BUlRJRkFDVF9OQU1FLFxuICAgICAgICAgICAgcGF0aDogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiUmVzdG9yZSBidWlsZCBhcnRpZmFjdCBwZXJtaXNzaW9uc1wiLFxuICAgICAgICAgIGNvbnRpbnVlT25FcnJvcjogdHJ1ZSxcbiAgICAgICAgICBydW46IFtcbiAgICAgICAgICAgIGBjZCAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fSAmJiBzZXRmYWNsIC0tcmVzdG9yZT0ke1BFUk1JU1NJT05fQkFDS1VQX0ZJTEV9YCxcbiAgICAgICAgICBdLmpvaW4oXCJcXG5cIiksXG4gICAgICAgIH1cbiAgICAgICk7XG4gICAgfVxuXG4gICAgc3RlcHMucHVzaCguLi4oam9iLnN0ZXBzID8/IFtdKSk7XG5cbiAgICB0aGlzLndvcmtmbG93LmFkZEpvYihpZCwge1xuICAgICAgbmVlZHM6IFtCVUlMRF9KT0JJRF0sXG4gICAgICAvLyBvbmx5IHJ1biBpZiBidWlsZCBkaWQgbm90IHNlbGYtbXV0YXRlXG4gICAgICBpZjogYCEgJHtTRUxGX01VVEFUSU9OX0NPTkRJVElPTn1gLFxuICAgICAgLi4uam9iLFxuICAgICAgc3RlcHM6IHN0ZXBzLFxuICAgIH0pO1xuXG4gICAgLy8gYWRkIHRvIHRoZSBsaXN0IG9mIGJ1aWxkIGpvYiBJRHNcbiAgICB0aGlzLl9wb3N0QnVpbGRKb2JzLnB1c2goaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJ1biBhIHRhc2sgYXMgYSBqb2Igd2l0aGluIHRoZSBidWlsZCB3b3JrZmxvdyB3aGljaCBpcyBleGVjdXRlZCBhZnRlciB0aGVcbiAgICogYnVpbGQgam9iIHN1Y2NlZWRlZC5cbiAgICpcbiAgICogVGhlIGpvYiB3aWxsIGhhdmUgYWNjZXNzIHRvIGJ1aWxkIGFydGlmYWN0cyBhbmQgd2lsbCBpbnN0YWxsIHByb2plY3RcbiAgICogZGVwZW5kZW5jaWVzIGluIG9yZGVyIHRvIGJlIGFibGUgdG8gcnVuIGFueSBjb21tYW5kcyB1c2VkIGluIHRoZSB0YXNrcy5cbiAgICpcbiAgICogSm9icyBhcmUgZXhlY3V0ZWQgX29ubHlfIGlmIHRoZSBidWlsZCBkaWQgTk9UIHNlbGYgbXV0YXRlLiBJZiB0aGUgYnVpbGRcbiAgICogc2VsZi1tdXRhdGUsIHRoZSBicmFuY2ggd2lsbCBlaXRoZXIgYmUgdXBkYXRlZCBvciB0aGUgYnVpbGQgd2lsbCBmYWlsIChpblxuICAgKiBmb3JrcyksIHNvIHRoZXJlIGlzIG5vIHBvaW50IGluIGV4ZWN1dGluZyB0aGUgcG9zdC1idWlsZCBqb2IuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIFNwZWNpZnkgdG9vbHMgYW5kIG90aGVyIG9wdGlvbnNcbiAgICovXG4gIHB1YmxpYyBhZGRQb3N0QnVpbGRKb2JUYXNrKFxuICAgIHRhc2s6IFRhc2ssXG4gICAgb3B0aW9uczogQWRkUG9zdEJ1aWxkSm9iVGFza09wdGlvbnMgPSB7fVxuICApIHtcbiAgICB0aGlzLmFkZFBvc3RCdWlsZEpvYkNvbW1hbmRzKFxuICAgICAgYHBvc3QtYnVpbGQtJHt0YXNrLm5hbWV9YCxcbiAgICAgIFtgJHt0aGlzLnByb2plY3QucHJvamVuQ29tbWFuZH0gJHt0YXNrLm5hbWV9YF0sXG4gICAgICB7XG4gICAgICAgIGNoZWNrb3V0UmVwbzogdHJ1ZSxcbiAgICAgICAgaW5zdGFsbERlcHM6IHRydWUsXG4gICAgICAgIHRvb2xzOiBvcHRpb25zLnRvb2xzLFxuICAgICAgICBydW5zT246IG9wdGlvbnMucnVuc09uLFxuICAgICAgfVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogUnVuIGEgc2VxdWVuY2Ugb2YgY29tbWFuZHMgYXMgYSBqb2Igd2l0aGluIHRoZSBidWlsZCB3b3JrZmxvdyB3aGljaCBpc1xuICAgKiBleGVjdXRlZCBhZnRlciB0aGUgYnVpbGQgam9iIHN1Y2NlZWRlZC5cbiAgICpcbiAgICogSm9icyBhcmUgZXhlY3V0ZWQgX29ubHlfIGlmIHRoZSBidWlsZCBkaWQgTk9UIHNlbGYgbXV0YXRlLiBJZiB0aGUgYnVpbGRcbiAgICogc2VsZi1tdXRhdGUsIHRoZSBicmFuY2ggd2lsbCBlaXRoZXIgYmUgdXBkYXRlZCBvciB0aGUgYnVpbGQgd2lsbCBmYWlsIChpblxuICAgKiBmb3JrcyksIHNvIHRoZXJlIGlzIG5vIHBvaW50IGluIGV4ZWN1dGluZyB0aGUgcG9zdC1idWlsZCBqb2IuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIFNwZWNpZnkgdG9vbHMgYW5kIG90aGVyIG9wdGlvbnNcbiAgICovXG4gIHB1YmxpYyBhZGRQb3N0QnVpbGRKb2JDb21tYW5kcyhcbiAgICBpZDogc3RyaW5nLFxuICAgIGNvbW1hbmRzOiBzdHJpbmdbXSxcbiAgICBvcHRpb25zPzogQWRkUG9zdEJ1aWxkSm9iQ29tbWFuZHNPcHRpb25zXG4gICkge1xuICAgIGNvbnN0IHN0ZXBzID0gW107XG5cbiAgICBpZiAob3B0aW9ucz8uY2hlY2tvdXRSZXBvKSB7XG4gICAgICBzdGVwcy5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJDaGVja291dFwiLFxuICAgICAgICB1c2VzOiBcImFjdGlvbnMvY2hlY2tvdXRAdjNcIixcbiAgICAgICAgd2l0aDoge1xuICAgICAgICAgIHJlZjogUFVMTF9SRVFVRVNUX1JFRixcbiAgICAgICAgICByZXBvc2l0b3J5OiBQVUxMX1JFUVVFU1RfUkVQT1NJVE9SWSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGlmIChcbiAgICAgIG9wdGlvbnM/LmNoZWNrb3V0UmVwbyAmJlxuICAgICAgb3B0aW9ucz8uaW5zdGFsbERlcHMgJiZcbiAgICAgIHRoaXMucHJvamVjdCBpbnN0YW5jZW9mIE5vZGVQcm9qZWN0XG4gICAgKSB7XG4gICAgICBzdGVwcy5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJJbnN0YWxsIGRlcGVuZGVuY2llc1wiLFxuICAgICAgICBydW46IGAke3RoaXMucHJvamVjdC5wYWNrYWdlLmluc3RhbGxDb21tYW5kfWAsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBzdGVwcy5wdXNoKHsgcnVuOiBjb21tYW5kcy5qb2luKFwiXFxuXCIpIH0pO1xuXG4gICAgdGhpcy5hZGRQb3N0QnVpbGRKb2IoaWQsIHtcbiAgICAgIHBlcm1pc3Npb25zOiB7XG4gICAgICAgIGNvbnRlbnRzOiBKb2JQZXJtaXNzaW9uLlJFQUQsXG4gICAgICB9LFxuICAgICAgdG9vbHM6IG9wdGlvbnM/LnRvb2xzLFxuICAgICAgcnVuc09uOiBvcHRpb25zPy5ydW5zT24gPz8gdGhpcy5kZWZhdWx0UnVubmVycyxcbiAgICAgIHN0ZXBzLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRTZWxmTXV0YXRpb25Kb2Iob3B0aW9uczogQnVpbGRXb3JrZmxvd09wdGlvbnMpIHtcbiAgICB0aGlzLndvcmtmbG93LmFkZEpvYihcInNlbGYtbXV0YXRpb25cIiwge1xuICAgICAgcnVuc09uOiBvcHRpb25zLnJ1bnNPbiA/PyB0aGlzLmRlZmF1bHRSdW5uZXJzLFxuICAgICAgcGVybWlzc2lvbnM6IHtcbiAgICAgICAgY29udGVudHM6IEpvYlBlcm1pc3Npb24uV1JJVEUsXG4gICAgICB9LFxuICAgICAgbmVlZHM6IFtCVUlMRF9KT0JJRF0sXG4gICAgICBpZjogYGFsd2F5cygpICYmICR7U0VMRl9NVVRBVElPTl9DT05ESVRJT059ICYmICR7Tk9UX0ZPUkt9YCxcbiAgICAgIHN0ZXBzOiBbXG4gICAgICAgIC4uLnRoaXMud29ya2Zsb3cucHJvamVuQ3JlZGVudGlhbHMuc2V0dXBTdGVwcyxcbiAgICAgICAgLi4uV29ya2Zsb3dBY3Rpb25zLmNoZWNrb3V0V2l0aFBhdGNoKHtcbiAgICAgICAgICAvLyB3ZSBuZWVkIHRvIHVzZSBhIFBBVCBzbyB0aGF0IG91ciBwdXNoIHdpbGwgdHJpZ2dlciB0aGUgYnVpbGQgd29ya2Zsb3dcbiAgICAgICAgICB0b2tlbjogdGhpcy53b3JrZmxvdy5wcm9qZW5DcmVkZW50aWFscy50b2tlblJlZixcbiAgICAgICAgICByZWY6IFBVTExfUkVRVUVTVF9SRUYsXG4gICAgICAgICAgcmVwb3NpdG9yeTogUFVMTF9SRVFVRVNUX1JFUE9TSVRPUlksXG4gICAgICAgIH0pLFxuICAgICAgICAuLi5Xb3JrZmxvd0FjdGlvbnMuc2V0R2l0SWRlbnRpdHkodGhpcy5naXRJZGVudGl0eSksXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBcIlB1c2ggY2hhbmdlc1wiLFxuICAgICAgICAgIHJ1bjogW1xuICAgICAgICAgICAgXCIgIGdpdCBhZGQgLlwiLFxuICAgICAgICAgICAgJyAgZ2l0IGNvbW1pdCAtcyAtbSBcImNob3JlOiBzZWxmIG11dGF0aW9uXCInLFxuICAgICAgICAgICAgYCAgZ2l0IHB1c2ggb3JpZ2luIEhFQUQ6JHtQVUxMX1JFUVVFU1RfUkVGfWAsXG4gICAgICAgICAgXS5qb2luKFwiXFxuXCIpLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgKGxhemlseSkgZHVyaW5nIHN5bnRoIHRvIHJlbmRlciB0aGUgYnVpbGQgam9iIHN0ZXBzLlxuICAgKi9cbiAgcHJpdmF0ZSByZW5kZXJCdWlsZFN0ZXBzKCk6IEpvYlN0ZXBbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJDaGVja291dFwiLFxuICAgICAgICB1c2VzOiBcImFjdGlvbnMvY2hlY2tvdXRAdjNcIixcbiAgICAgICAgd2l0aDoge1xuICAgICAgICAgIHJlZjogUFVMTF9SRVFVRVNUX1JFRixcbiAgICAgICAgICByZXBvc2l0b3J5OiBQVUxMX1JFUVVFU1RfUkVQT1NJVE9SWSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG5cbiAgICAgIC4uLnRoaXMucHJlQnVpbGRTdGVwcyxcblxuICAgICAge1xuICAgICAgICBuYW1lOiB0aGlzLmJ1aWxkVGFzay5uYW1lLFxuICAgICAgICBydW46IHRoaXMuZ2l0aHViLnByb2plY3QucnVuVGFza0NvbW1hbmQodGhpcy5idWlsZFRhc2spLFxuICAgICAgfSxcblxuICAgICAgLi4udGhpcy5wb3N0QnVpbGRTdGVwcyxcblxuICAgICAgLy8gY2hlY2sgZm9yIG11dGF0aW9ucyBhbmQgdXBsb2FkIGEgZ2l0IHBhdGNoIGZpbGUgYXMgYW4gYXJ0aWZhY3RcbiAgICAgIC4uLldvcmtmbG93QWN0aW9ucy5jcmVhdGVVcGxvYWRHaXRQYXRjaCh7XG4gICAgICAgIHN0ZXBJZDogU0VMRl9NVVRBVElPTl9TVEVQLFxuICAgICAgICBvdXRwdXROYW1lOiBTRUxGX01VVEFUSU9OX0hBUFBFTkVEX09VVFBVVCxcbiAgICAgICAgbXV0YXRpb25FcnJvcjpcbiAgICAgICAgICBcIkZpbGVzIHdlcmUgY2hhbmdlZCBkdXJpbmcgYnVpbGQgKHNlZSBidWlsZCBsb2cpLiBJZiB0aGlzIHdhcyB0cmlnZ2VyZWQgZnJvbSBhIGZvcmssIHlvdSB3aWxsIG5lZWQgdG8gdXBkYXRlIHlvdXIgYnJhbmNoLlwiLFxuICAgICAgfSksXG5cbiAgICAgIC8vIHVwbG9hZCB0aGUgYnVpbGQgYXJ0aWZhY3Qgb25seSBpZiB3ZSBoYXZlIHBvc3QtYnVpbGQgam9icyAob3RoZXJ3aXNlLCB0aGVyZSdzIG5vIHBvaW50KVxuICAgICAgLi4uKHRoaXMuX3Bvc3RCdWlsZEpvYnMubGVuZ3RoID09IDBcbiAgICAgICAgPyBbXVxuICAgICAgICA6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgbmFtZTogXCJCYWNrdXAgYXJ0aWZhY3QgcGVybWlzc2lvbnNcIixcbiAgICAgICAgICAgICAgY29udGludWVPbkVycm9yOiB0cnVlLFxuICAgICAgICAgICAgICBydW46IGBjZCAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fSAmJiBnZXRmYWNsIC1SIC4gPiAke1BFUk1JU1NJT05fQkFDS1VQX0ZJTEV9YCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIG5hbWU6IFwiVXBsb2FkIGFydGlmYWN0XCIsXG4gICAgICAgICAgICAgIHVzZXM6IFwiYWN0aW9ucy91cGxvYWQtYXJ0aWZhY3RAdjNcIixcbiAgICAgICAgICAgICAgd2l0aDoge1xuICAgICAgICAgICAgICAgIG5hbWU6IEJVSUxEX0FSVElGQUNUX05BTUUsXG4gICAgICAgICAgICAgICAgcGF0aDogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0pLFxuICAgIF07XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgQnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRKb2JUYXNrYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZFBvc3RCdWlsZEpvYlRhc2tPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRvb2xzIHRoYXQgc2hvdWxkIGJlIGluc3RhbGxlZCBiZWZvcmUgdGhlIHRhc2sgaXMgcnVuLlxuICAgKi9cbiAgcmVhZG9ubHkgdG9vbHM/OiBUb29scztcblxuICAvKipcbiAgICogR2l0aHViIFJ1bm5lciBzZWxlY3Rpb24gbGFiZWxzXG4gICAqIEBkZWZhdWx0IFtcInVidW50dS1sYXRlc3RcIl1cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnNPbj86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBCdWlsZFdvcmtmbG93LmFkZFBvc3RCdWlsZEpvYkNvbW1hbmRzYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFkZFBvc3RCdWlsZEpvYkNvbW1hbmRzT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUb29scyB0aGF0IHNob3VsZCBiZSBpbnN0YWxsZWQgYmVmb3JlIHRoZSBjb21tYW5kcyBhcmUgcnVuLlxuICAgKi9cbiAgcmVhZG9ubHkgdG9vbHM/OiBUb29scztcblxuICAvKipcbiAgICogQ2hlY2sgb3V0IHRoZSByZXBvc2l0b3J5IGF0IHRoZSBwdWxsIHJlcXVlc3QgYnJhbmNoIGJlZm9yZSBjb21tYW5kcyBhcmVcbiAgICogcnVuLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY2hlY2tvdXRSZXBvPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSW5zdGFsbCBwcm9qZWN0IGRlcGVuZGVuY2llcyBiZWZvcmUgcnVubmluZyBjb21tYW5kcy4gYGNoZWNrb3V0UmVwb2AgbXVzdFxuICAgKiBhbHNvIGJlIHNldCB0byB0cnVlLlxuICAgKlxuICAgKiBDdXJyZW50bHkgb25seSBzdXBwb3J0ZWQgZm9yIGBOb2RlUHJvamVjdGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpbnN0YWxsRGVwcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEdpdGh1YiBSdW5uZXIgc2VsZWN0aW9uIGxhYmVsc1xuICAgKiBAZGVmYXVsdCBbXCJ1YnVudHUtbGF0ZXN0XCJdXG4gICAqL1xuICByZWFkb25seSBydW5zT24/OiBzdHJpbmdbXTtcbn1cbiJdfQ==