# GitHub

By default, many projects are initialized with `GitHub` component to enabled GitHub as the default provider for CI/CD workflows. See https://docs.github.com/en/actions for more information.

The use of GitHub (and generating corresponding files in `.github`) can be disabled by specifying `github: false` in your project options.

## GitHub API access

Several workflows generated by projen use APIs that are not available through
the permissions of [GITHUB_TOKEN]. To use these workflows, you must provide
either a Personal Access Token (PAT) or a GitHub App to provide API access.

[GITHUB_TOKEN]: https://docs.github.com/en/actions/security-guides/automatic-token-authentication

### Personal Access Token (classic)

Follow the [GitHub docs
instructions](https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token#creating-a-personal-access-token-classic)
for creating a personal access token (classic).
When creating the classic PAT, grant the token `repo`, `workflow` and `write:packages` permissions.

Add the token as a secret to your repo under the name `PROJEN_GITHUB_TOKEN`.

### Fine-grained Personal Access Token (beta)

Follow the [GitHub docs
instructions](https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token#creating-a-fine-grained-personal-access-token)
for creating a fine-grained personal access token (beta).

Select the repositories you want to use this token for.
You can reuse the same token for multiple repositories if you select them here.
However you should choose the minimal repository access that meets your needs.

Under Permissions, select the following Repository Permissions:

- `Contents` - Read and write
- `Metadata` - Read-only (automatically added)
- `Pull requests` - Read and write
- `Workflows` - Read and write

Add the token as a secret to your repo under the name `PROJEN_GITHUB_TOKEN`.

### GitHub App

Follow the [GitHub docs instructions](https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app) for creating a GitHub App. Enable read & write permission for "Contents" and "Pull Request" scopes.

Add the App ID as a secret to your repo under the name `PROJEN_APP_ID` and the private key you generate as a secret to your repo under the name `PROJEN_APP_PRIVATE_KEY`.

Then, configure your projenrc file to use the GitHub app for API access:

```ts
const { github, javascript } = require('projen');

const project = new javascript.NodeProject({
  // ...other options
  githubOptions: {
    projenCredentials: github.GithubCredentials.fromApp({ ... }),
  },
});
```

If the GitHub app you are using has additional permissions assigned, you can limit the permissions used on the token within the jobs in the workflow to only allow access to write repository contents and create the pull request:

```ts
const { github, javascript } = require('projen');

const project = new javascript.NodeProject({
  // ...other options
  githubOptions: {
    projenCredentials: github.GithubCredentials.fromApp({
      // ...other options
      permissions: {
        pullRequests: github.workflows.AppPermission.WRITE,
        contents: github.workflows.AppPermission.WRITE,
      }
    }),
  },
});
```

## Workflows

TODO

See the `GitHub`, `GithubWorkflow`, and `Job` types in the [API
reference](./api/API.md) for currently available APIs.

Example code of creating a GitHub workflow:
<https://github.com/projen/projen/blob/65b4194c163f47ba4842981b0c92dbe516be787b/src/github/auto-approve.ts#L67-L105>

### Stale workflow

A "stale" workflow can be added which will automatically close issues or pull
requests on your GitHub repository after time has passed without the issue
seeing any comments or updates. You can enable as shown below:

```ts
// or PythonProject, etc.
new typescript.TypeScriptProject({
  stale: true,
  staleOptions: {
    issues: {
      closeMessage: "closing pull request",
      staleLabel: "I-AM-STALE",
      daysBeforeStale: 180,
    },
  }
})
```

Check the API reference for a list of all available options.

When enabled, by default issues with no activity with will be marked as stale
after 60 days and closed within 7 days, and pull requests with no activity will
be marked as stale after 14 days and closed within 2 days.
