"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            dashboardName: props.dashboardName,
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'Dashboard', {
            dashboardName: `${props.dashboardName}-high-severity`,
        });
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        alarm.addAlarmAction({
            bind: () => ({ alarmActionArn: this.alarmActions.highSeverity }),
        });
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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