# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['schemathesis',
 'schemathesis.cli',
 'schemathesis.cli.output',
 'schemathesis.extra',
 'schemathesis.runner',
 'schemathesis.runner.impl']

package_data = \
{'': ['*']}

install_requires = \
['attrs>=19.1,<20.0',
 'click>=7.0,<8.0',
 'hypothesis>=4.32,<6.0',
 'hypothesis_jsonschema>=0.9.13,<1.0',
 'jsonschema>=3.0.0,<4.0.0',
 'pytest-subtests>=0.2.1,<0.3.0',
 'pytest>4.6.4',
 'pyyaml>=5.1,<6.0',
 'requests>=2.22,<3.0',
 'werkzeug>0.16.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=1.1,<2.0']}

entry_points = \
{'console_scripts': ['schemathesis = schemathesis.cli:schemathesis'],
 'pytest11': ['schemathesis = schemathesis.extra.pytest_plugin']}

setup_kwargs = {
    'name': 'schemathesis',
    'version': '1.0.5',
    'description': 'Hypothesis strategies for Open API / Swagger schemas',
    'long_description': 'Schemathesis\n============\n\n|Build| |Coverage| |Version| |Python versions| |Docs| |License|\n\nSchemathesis is a tool for testing your web applications built with Open API / Swagger specifications.\n\nIt reads the application schema and generates test cases which will ensure that your application is compliant with its schema.\n\nThe application under test could be written in any language, the only thing you need is a valid API schema in a supported format.\n\n**Supported specification versions**:\n\n- Swagger 2.0\n- Open API 3.0.x\n\nMore API specifications will be added in the future.\n\nBuilt with:\n\n- `hypothesis`_\n\n- `hypothesis_jsonschema`_\n\n- `pytest`_\n\nInspired by wonderful `swagger-conformance <https://github.com/olipratt/swagger-conformance>`_ project.\n\nIf you are looking for more information, then there is an article about Schemathesis: https://code.kiwi.com/schemathesis-property-based-testing-for-api-schemas-52811fd2b0a4\n\nInstallation\n------------\n\nTo install Schemathesis via ``pip`` run the following command:\n\n.. code:: bash\n\n    pip install schemathesis\n\nGitter: https://gitter.im/kiwicom/schemathesis\n\nUsage\n-----\n\nThere are two basic ways to use Schemathesis:\n\n- `Command Line Interface <https://github.com/kiwicom/schemathesis#command-line-interface>`_\n- `Writing tests in Python <https://github.com/kiwicom/schemathesis#in-code>`_\n\nCLI is pretty simple to use and requires no coding, in-code approach gives more flexibility.\n\nCommand Line Interface\n~~~~~~~~~~~~~~~~~~~~~~\n\nThe ``schemathesis`` command can be used to perform Schemathesis test cases:\n\n.. code:: bash\n\n    schemathesis run https://example.com/api/swagger.json\n\n.. image:: https://github.com/kiwicom/schemathesis/blob/master/img/schemathesis.gif\n\nIf your application requires authorization then you can use ``--auth`` option for Basic Auth and ``--header`` to specify\ncustom headers to be sent with each request.\n\nTo filter your tests by endpoint name, HTTP method or Open API tags you could use ``-E``, ``-M``, ``-T`` options respectively.\n\nCLI supports passing options to ``hypothesis.settings``. All of them are prefixed with ``--hypothesis-``:\n\n.. code:: bash\n\n    schemathesis run --hypothesis-max-examples=1000 https://example.com/api/swagger.json\n\nTo speed up the testing process Schemathesis provides ``-w/--workers`` option for concurrent test execution:\n\n.. code:: bash\n\n    schemathesis run -w 8 https://example.com/api/swagger.json\n\nIn the example above all tests will be distributed among 8 worker threads.\n\nIf you\'d like to test your web app (Flask or AioHTTP for example) then there is ``--app`` option for you:\n\n.. code:: bash\n\n    schemathesis run --app=importable.path:app /swagger.json\n\nYou need to specify an importable path to the module where your app instance resides and a variable name after ``:`` that points\nto your app. **Note**, app factories are not supported. The schema location could be:\n\n- A full URL;\n- An existing filesystem path;\n- In-app endpoint with schema.\n\nThis method is significantly faster for WSGI apps, since it doesn\'t involve network.\n\nFor the full list of options, run:\n\n.. code:: bash\n\n    schemathesis --help\n    # Or\n    schemathesis run --help\n\nDocker\n~~~~~~\n\nSchemathesis CLI also available as a docker image\n\n.. code:: bash\n\n    docker run kiwicom/schemathesis:stable run http://example.com/schema.json\n\nTo run it against localhost server add ``--network=host`` parameter:\n\n.. code:: bash\n\n    docker run --network="host" kiwicom/schemathesis:stable run http://127.0.0.1/schema.json\n\nPre-run CLI hook\n################\n\nSometimes you need to execute custom code before the CLI run, for example setup an environment,\nregister custom string format strategies or modify Schemathesis behavior in runtime you can use ``--pre-run`` hook:\n\n.. code:: bash\n\n    schemathesis --pre-run importable.path.to.module run https://example.com/api/swagger.json\n\n**NOTE**. This option should be passed before the ``run`` part.\n\nThe passed value will be processed as an importable Python path, where you can execute your code.\nAn example - https://github.com/kiwicom/schemathesis#custom-string-strategies\n\nRegistering custom checks for CLI\n#################################\n\nTo add a new check for the Schemathesis CLI there is a special function\n\n.. code:: python\n\n    import schemathesis\n\n    @schemathesis.register_check\n    def new_check(response, case):\n        # some awesome assertions!\n        pass\n\nThe registered check should accept a ``response`` with ``requests.Response`` / ``schemathesis.utils.WSGIResponse`` type and\n``case`` with ``schemathesis.models.Case`` type.\n\nAfter registration, your checks will be available in Schemathesis CLI and you can use them via ``-c`` command line option.\n\n.. code:: bash\n\n    schemathesis --pre-run module.with.checks run -c new_check https://example.com/api/swagger.json\n\nIn-code\n~~~~~~~\n\nTo examine your application with Schemathesis you need to:\n\n- Setup & run your application, so it is accessible via the network;\n- Write a couple of tests in Python;\n- Run the tests via ``pytest``.\n\nSuppose you have your application running on ``http://0.0.0.0:8080`` and its\nschema is available at ``http://0.0.0.0:8080/swagger.json``.\n\nA basic test, that will verify that any data, that fit into the schema will not cause any internal server error could\nlook like this:\n\n.. code:: python\n\n    # test_api.py\n    import requests\n    import schemathesis\n\n    schema = schemathesis.from_uri("http://0.0.0.0:8080/swagger.json")\n\n    @schema.parametrize()\n    def test_no_server_errors(case):\n        # `requests` will make an appropriate call under the hood\n        response = case.call()  # use `call_wsgi` if you used `schemathesis.from_wsgi`\n        # You could use built-in checks\n        case.validate_response(response)\n        # Or assert the response manually\n        assert response.status_code < 500\n\n\nIt consists of four main parts:\n\n1. Schema preparation; ``schemathesis`` package provides multiple ways to initialize the schema - ``from_path``, ``from_dict``, ``from_uri``, ``from_file`` and ``from_wsgi``.\n\n2. Test parametrization; ``@schema.parametrize()`` generates separate tests for all endpoint/method combination available in the schema.\n\n3. A network call to the running application; ``case.call`` does it.\n\n4. Verifying a property you\'d like to test; In the example, we verify that any app response will not indicate a server-side error (HTTP codes 5xx).\n\n**NOTE**. Look for ``from_wsgi`` usage `below <https://github.com/kiwicom/schemathesis#wsgi>`_\n\nRun the tests:\n\n.. code:: bash\n\n    pytest test_api.py\n\n**Other properties that could be tested**:\n\n- Any call will be processed in <50 ms - you can verify the app performance;\n- Any unauthorized access will end with 401 HTTP response code;\n\nEach test function should have the ``case`` fixture, that represents a single test case.\n\nImportant ``Case`` attributes:\n\n- ``method`` - HTTP method\n- ``formatted_path`` - full endpoint path\n- ``headers`` - HTTP headers\n- ``query`` - query parameters\n- ``body`` - request body\n\nYou can use them manually in network calls or can convert to a dictionary acceptable by ``requests.request``:\n\n.. code:: python\n\n    import requests\n\n    schema = schemathesis.from_uri("http://0.0.0.0:8080/swagger.json")\n\n    @schema.parametrize()\n    def test_no_server_errors(case):\n        kwargs = case.as_requests_kwargs()\n        response = requests.request(**kwargs)\n\n\nFor each test, Schemathesis will generate a bunch of random inputs acceptable by the schema.\nThis data could be used to verify that your application works in the way as described in the schema or that schema describes expected behavior.\n\nBy default, there will be 100 test cases per endpoint/method combination.\nTo limit the number of examples you could use ``hypothesis.settings`` decorator on your test functions:\n\n.. code:: python\n\n    from hypothesis import settings\n\n    @schema.parametrize()\n    @settings(max_examples=5)\n    def test_something(client, case):\n        ...\n\nTo narrow down the scope of the schemathesis tests it is possible to filter by method or endpoint:\n\n.. code:: python\n\n    @schema.parametrize(method="GET", endpoint="/pet")\n    def test_no_server_errors(case):\n        ...\n\nThe acceptable values are regexps or list of regexps (matched with ``re.search``).\n\nWSGI applications support\n~~~~~~~~~~~~~~~~~~~~~~~~~\n\nSchemathesis supports making calls to WSGI-compliant applications instead of real network calls, in this case\nthe test execution will go much faster.\n\n.. code:: python\n\n    app = Flask("test_app")\n\n    @app.route("/schema.json")\n    def schema():\n        return {...}\n\n    @app.route("/v1/users", methods=["GET"])\n    def users():\n        return jsonify([{"name": "Robin"}])\n\n    schema = schemathesis.from_wsgi("/schema.json", app)\n\n    @schema.parametrize()\n    def test_no_server_errors(case):\n        response = case.call_wsgi()\n        assert response.status_code < 500\n\nExplicit examples\n~~~~~~~~~~~~~~~~~\n\nIf the schema contains parameters examples, then they will be additionally included in the generated cases.\n\n.. code:: yaml\n\n    paths:\n      get:\n        parameters:\n        - in: body\n          name: body\n          required: true\n          schema: \'#/definitions/Pet\'\n\n    definitions:\n      Pet:\n        additionalProperties: false\n        example:\n          name: Doggo\n        properties:\n          name:\n            type: string\n        required:\n        - name\n        type: object\n\n\nWith this Swagger schema example, there will be a case with body ``{"name": "Doggo"}``.  Examples handled with\n``example`` decorator from Hypothesis, more info about its behavior is `here`_.\n\nIf you\'d like to test only examples provided in the schema, you could utilize ``--hypothesis-phases=explicit`` CLI option:\n\n.. code:: bash\n\n    $ schemathesis run --hypothesis-phases=explicit https://example.com/api/swagger.json\n\nOr add this decorator to your test if you use Schemathesis in your Python tests:\n\n.. code:: python\n\n    from hypothesis import settings, Phase\n\n    ...\n    @schema.parametrize()\n    @settings(phases=[Phase.explicit])\n    def test_api(case):\n        ...\n\n**NOTE**. Schemathesis does not support examples in individual properties that are specified inside Schema Object.\nBut examples in `Parameter Object <https://github.com/OAI/OpenAPI-Specification/blob/master/versions/3.0.3.md#parameter-object>`_, `Media Type Object <https://github.com/OAI/OpenAPI-Specification/blob/master/versions/3.0.3.md#mediaTypeObject>`_ and `Schema Object <https://github.com/OAI/OpenAPI-Specification/blob/master/versions/3.0.3.md#schema-object>`_ are supported.\nSee below:\n\n.. code:: yaml\n\n    ...\n    paths:\n      /users:\n        parameters:\n          - in: query\n            name: foo\n            example: bar  # SUPPORTED!\n            schema:\n              type: string\n              example: spam  # SUPPORTED!\n        post:\n          requestBody:\n            content:\n              application/json:\n                schema:\n                  type: object\n                  properties:\n                    foo:\n                      type: string\n                      example: bar  # NOT SUPPORTED!\n\nDirect strategies access\n~~~~~~~~~~~~~~~~~~~~~~~~\n\nFor convenience you can explore the schemas and strategies manually:\n\n.. code:: python\n\n    >>> import schemathesis\n    >>> schema = schemathesis.from_uri("http://0.0.0.0:8080/petstore.json")\n    >>> endpoint = schema["/v2/pet"]["POST"]\n    >>> strategy = endpoint.as_strategy()\n    >>> strategy.example()\n    Case(\n        path=\'/v2/pet\',\n        method=\'POST\',\n        path_parameters={},\n        headers={},\n        cookies={},\n        query={},\n        body={\n            \'name\': \'\\x15.\\x13\\U0008f42a\',\n            \'photoUrls\': [\'\\x08\\U0009f29a\', \'\\U000abfd6\\U000427c4\', \'\']\n        },\n        form_data={}\n    )\n\nSchema instances implement ``Mapping`` protocol.\n\nIf you want to customize how data is generated, then you can use hooks of two types:\n\n- Global, which are applied to all schemas;\n- Schema-local, which are applied only for specific schema instance;\n\nEach hook accepts a Hypothesis strategy and should return a Hypothesis strategy:\n\n.. code:: python\n\n    import schemathesis\n\n    def global_hook(strategy):\n        return strategy.filter(lambda x: x["id"].isdigit())\n\n    schemathesis.hooks.register("query", hook)\n\n    schema = schemathesis.from_uri("http://0.0.0.0:8080/swagger.json")\n\n    def schema_hook(strategy):\n        return strategy.filter(lambda x: int(x["id"]) % 2 == 0)\n\n    schema.register_hook("query", schema_hook)\n\nThere are 6 places, where hooks can be applied and you need to pass it as the first argument to ``schemathesis.hooks.register`` or ``schema.register_hook``:\n\n- path_parameters\n- headers\n- cookies\n- query\n- body\n- form_data\n\nIt might be useful if you want to exclude certain cases that you don\'t want to test, or modify the generated data, so it\nwill be more meaningful for the application - add existing IDs from the database, custom auth header, etc.\n\n**NOTE**. Global hooks are applied first.\n\nLazy loading\n~~~~~~~~~~~~\n\nIf you have a schema that is not available when the tests are collected, for example it is build with tools\nlike ``apispec`` and requires an application instance available, then you can parametrize the tests from a pytest fixture.\n\n.. code:: python\n\n    # test_api.py\n    import schemathesis\n\n    schema = schemathesis.from_pytest_fixture("fixture_name")\n\n    @schema.parametrize()\n    def test_api(case):\n        ...\n\nIn this case the test body will be used as a sub-test via ``pytest-subtests`` library.\n\n**NOTE**: the used fixture should return a valid schema that could be created via ``schemathesis.from_dict`` or other\n``schemathesis.from_`` variations.\n\nExtending schemathesis\n~~~~~~~~~~~~~~~~~~~~~~\n\nIf you\'re looking for a way to extend ``schemathesis`` or reuse it in your own application, then ``runner`` module might be helpful for you.\nIt can run tests against the given schema URI and will do some simple checks for you.\n\n.. code:: python\n\n    from schemathesis import runner\n\n    events = runner.prepare("http://127.0.0.1:8080/swagger.json")\n    for event in events:\n        # do something with event\n\n``runner.prepare`` creates a generator that yields events of different kinds - ``BeforeExecution``, ``AfterExecution``, etc.\nThey provide a lot of useful information about what happens during tests, but handling of these events is your responsibility.\nYou can take some inspiration from Schemathesis `CLI implementation <https://github.com/kiwicom/schemathesis/blob/master/src/schemathesis/cli/__init__.py#L230>`_.\nSee full description of events in the `source code <https://github.com/kiwicom/schemathesis/blob/master/src/schemathesis/runner/events.py>`_.\n\nIf you want to use Schemathesis CLI with your custom checks, look at `this section <https://github.com/kiwicom/schemathesis/tree/dd/deprecate-execute#registering-custom-checks-for-cli>`_\n\nThe built-in checks list includes the following:\n\n- Not a server error. Asserts that response\'s status code is less than 500;\n- Status code conformance. Asserts that response\'s status code is listed in the schema;\n- Content type conformance. Asserts that response\'s content type is listed in the schema;\n- Response schema conformance. Asserts that response\'s content conforms to the declared schema;\n\nYou can provide your custom checks to the execute function, the check is a callable that accepts one argument of ``requests.Response`` type.\n\n.. code:: python\n\n    from datetime import timedelta\n    from schemathesis import runner, models\n\n    def not_too_long(response, case: models.Case):\n        assert response.elapsed < timedelta(milliseconds=300)\n\n    events = runner.prepare("http://127.0.0.1:8080/swagger.json", checks=[not_too_long])\n    for event in events:\n        # do something with event\n\nCustom string strategies\n########################\n\nSome string fields could use custom format and validators,\ne.g. ``card_number`` and Luhn algorithm validator.\n\nFor such cases it is possible to register custom strategies:\n\n1. Create ``hypothesis.strategies.SearchStrategy`` object\n2. Optionally provide predicate function to filter values\n3. Register it via ``schemathesis.register_string_format``\n\n.. code-block:: python\n\n    strategy = strategies.from_regex(r"\\A4[0-9]{15}\\Z").filter(luhn_validator)\n    schemathesis.register_string_format("visa_cards", strategy)\n\nUnittest support\n################\n\nSchemathesis supports Python\'s built-in ``unittest`` framework out of the box,\nyou only need to specify strategies for ``hypothesis.given``:\n\n.. code-block:: python\n\n    from unittest import TestCase\n    from hypothesis import given\n    import schemathesis\n\n    schema = schemathesis.from_uri("http://0.0.0.0:8080/petstore.json")\n    new_pet_strategy = schema["/v2/pet"]["POST"].as_strategy()\n\n    class TestSchema(TestCase):\n\n        @given(case=new_pet_strategy)\n        def test_pets(self, case):\n            response = case.call()\n            assert response.status_code < 500\n\nSchema validation\n#################\n\nTo avoid obscure and hard to debug errors during test runs Schemathesis validates input schemas for conformance with the relevant spec.\nIf you\'d like to disable this behavior use ``--validate-schema=false`` in CLI and ``validate_schema=False`` argument in loaders.\n\nDocumentation\n-------------\n\nFor the full documentation, please see https://schemathesis.readthedocs.io/en/latest/ (WIP)\n\nOr you can look at the ``docs/`` directory in the repository.\n\nLocal development\n-----------------\n\nFirst, you need to prepare a virtual environment with `poetry`_.\nInstall ``poetry`` (check out the `installation guide`_) and run this command inside the project root:\n\n.. code:: bash\n\n    poetry install\n\nFor simpler local development Schemathesis includes a ``aiohttp``-based server with the following endpoints in Swagger 2.0 schema:\n\n- ``/api/success`` - always returns ``{"success": true}``\n- ``/api/failure`` - always returns 500\n- ``/api/slow`` - always returns ``{"slow": true}`` after 250 ms delay\n- ``/api/unsatisfiable`` - parameters for this endpoint are impossible to generate\n- ``/api/invalid`` - invalid parameter definition. Uses ``int`` instead of ``integer``\n- ``/api/flaky`` - returns 1/1 ratio of 200/500 responses\n- ``/api/multipart`` - accepts multipart data\n- ``/api/teapot`` - returns 418 status code, that is not listed in the schema\n- ``/api/text`` - returns ``plain/text`` responses, which are not declared in the schema\n- ``/api/malformed_json`` - returns malformed JSON with ``application/json`` content type header\n\n\nTo start the server:\n\n.. code:: bash\n\n    ./test_server.sh 8081\n\nIt is possible to configure available endpoints via ``--endpoints`` option.\nThe value is expected to be a comma separated string with endpoint names (``success``, ``failure``, ``slow``, etc):\n\n.. code:: bash\n\n    ./test_server.sh 8081 --endpoints=success,slow\n\nThen you could use CLI against this server:\n\n.. code:: bash\n\n    schemathesis run http://127.0.0.1:8081/swagger.yaml\n    ================================== Schemathesis test session starts =================================\n    platform Linux -- Python 3.7.4, schemathesis-0.12.2, hypothesis-4.39.0, hypothesis_jsonschema-0.9.8\n    rootdir: /\n    hypothesis profile \'default\' -> database=DirectoryBasedExampleDatabase(\'/.hypothesis/examples\')\n    Schema location: http://127.0.0.1:8081/swagger.yaml\n    Base URL: http://127.0.0.1:8081\n    Specification version: Swagger 2.0\n    collected endpoints: 2\n\n    GET /api/slow .                                                                               [ 50%]\n    GET /api/success .                                                                            [100%]\n\n    ============================================== SUMMARY ==============================================\n\n    not_a_server_error            2 / 2 passed          PASSED\n\n    ========================================= 2 passed in 0.29s =========================================\n\n\nRunning tests\n~~~~~~~~~~~~~\n\nYou could run tests via ``tox``:\n\n.. code:: bash\n\n    tox -p all -o\n\nor ``pytest`` in your current environment:\n\n.. code:: bash\n\n    pytest test/ -n auto\n\nContributing\n------------\n\nAny contribution in development, testing or any other area is highly appreciated and useful to the project.\n\nPlease, see the `CONTRIBUTING.rst`_ file for more details.\n\nPython support\n--------------\n\nSchemathesis supports Python 3.6, 3.7 and 3.8.\n\nLicense\n-------\n\nThe code in this project is licensed under `MIT license`_.\nBy contributing to ``schemathesis``, you agree that your contributions\nwill be licensed under its MIT license.\n\n.. |Build| image:: https://github.com/kiwicom/schemathesis/workflows/build/badge.svg\n   :target: https://github.com/kiwicom/schemathesis/actions\n.. |Coverage| image:: https://codecov.io/gh/kiwicom/schemathesis/branch/master/graph/badge.svg\n   :target: https://codecov.io/gh/kiwicom/schemathesis/branch/master\n   :alt: codecov.io status for master branch\n.. |Version| image:: https://img.shields.io/pypi/v/schemathesis.svg\n   :target: https://pypi.org/project/schemathesis/\n.. |Python versions| image:: https://img.shields.io/pypi/pyversions/schemathesis.svg\n   :target: https://pypi.org/project/schemathesis/\n.. |License| image:: https://img.shields.io/pypi/l/schemathesis.svg\n   :target: https://opensource.org/licenses/MIT\n.. |Docs| image:: https://readthedocs.org/projects/schemathesis/badge/?version=stable\n   :target: https://schemathesis.readthedocs.io/en/stable/?badge=stable\n   :alt: Documentation Status\n\n.. _hypothesis: https://hypothesis.works/\n.. _hypothesis_jsonschema: https://github.com/Zac-HD/hypothesis-jsonschema\n.. _pytest: http://pytest.org/en/latest/\n.. _poetry: https://github.com/sdispater/poetry\n.. _installation guide: https://github.com/sdispater/poetry#installation\n.. _here: https://hypothesis.readthedocs.io/en/latest/reproducing.html#providing-explicit-examples\n.. _CONTRIBUTING.rst: https://github.com/kiwicom/schemathesis/blob/master/CONTRIBUTING.rst\n.. _MIT license: https://opensource.org/licenses/MIT\n',
    'author': 'Dmitry Dygalo',
    'author_email': 'dmitry.dygalo@kiwi.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/kiwicom/schemathesis',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
