# -*- coding: utf-8 -*-
from requests import Response
from slack_time import SlackAPI


class Search(SlackAPI):
    def all(
        self,
        query: str,
        count: int = None,
        highlight: bool = None,
        page: int = None,
        sort: str = None,
        sort_dir: str = None,
        **kwargs
    ) -> Response:
        """
        Searches for messages and files matching a query.
        https://api.slack.com/methods/search.all

        :param token: Authentication token bearing required scopes.
        :type str: e.g. xxxx-xxxxxxxxx-xxxx

        :param query: Search query. May contains booleans, etc.
        :type str: e.g. pickleface

        :param count: Number of items to return per page.
        :type int: e.g. 20

        :param highlight: Pass a value of true to enable query highlight markers (see below).
        :type bool: e.g. true

        :param page: Page number of results to return.
        :type int: e.g. 2

        :param sort: Return matches sorted by either score or timestamp.
        :type str: e.g. timestamp

        :param sort_dir: Change sort direction to ascending (asc) or descending (desc).
        :type str: e.g. asc

        :returns response:
        :type requests.Response: e.g. <Response [200]>

        example:
        >>> client = SlackTime(token='insert-your-token-here')
        >>> response = client.search.all(**your_params)
        <Response [200]>
        >>> response.json()
        {
            "files": {
                "matches": [
                    {
                        "channels": [],
                        "comments_count": 1,
                        "created": 1508804330,
                        "display_as_bot": false,
                        "editable": false,
                        "external_type": "",
                        "filetype": "png",
                        "groups": [],
                        "id": "F7PKF1NR7",
                        "image_exif_rotation": 1,
                        "ims": [],
                        "initial_comment": {
                            "comment": "Sure! Here's the workflow diagram!",
                            "created": 1508804330,
                            "id": "Fc7NLL52E7",
                            "is_intro": true,
                            "timestamp": 1508804330,
                            "user": "U2U85N1RZ"
                        },
                        "is_external": false,
                        "is_public": true,
                        "mimetype": "image/png",
                        "mode": "hosted",
                        "name": "slack workflow diagram.png",
                        "original_h": 117,
                        "original_w": 128,
                        "permalink": "https://example.slack.com/files/U2U85N1RZ/F7PKF1NR7/slack_workflow_diagram.png",
                        "permalink_public": "https://slack-files.com/T2U81E2FZ-F7PKF1NR7-bea9143f18",
                        "pretty_type": "PNG",
                        "preview": null,
                        "public_url_shared": false,
                        "score": "0.99982661240974",
                        "size": 35705,
                        "thumb_160": "https://files.slack.com/files-tmb/T2U81E2FZ-F7PKF1NR7-19f33fc256/slack_workflow_diagram_160.png",
                        "thumb_360": "https://files.slack.com/files-tmb/T2U81E2FZ-F7PKF1NR7-19f33fc256/slack_workflow_diagram_360.png",
                        "thumb_360_h": 117,
                        "thumb_360_w": 128,
                        "thumb_64": "https://files.slack.com/files-tmb/T2U81E2FZ-F7PKF1NR7-19f33fc256/slack_workflow_diagram_64.png",
                        "thumb_80": "https://files.slack.com/files-tmb/T2U81E2FZ-F7PKF1NR7-19f33fc256/slack_workflow_diagram_80.png",
                        "timestamp": 1508804330,
                        "title": "slack workflow diagram",
                        "top_file": false,
                        "url_private": "https://files.slack.com/files-pri/T2U81E2FZ-F7PKF1NR7/slack_workflow_diagram.png",
                        "url_private_download": "https://files.slack.com/files-pri/T2U81E2FZ-F7PKF1NR7/download/slack_workflow_diagram.png",
                        "user": "U2U85N1RZ",
                        "username": "amy"
                    }
                ],
                "pagination": {
                    "first": 1,
                    "last": 1,
                    "page": 1,
                    "page_count": 1,
                    "per_page": 20,
                    "total_count": 1
                },
                "paging": {
                    "count": 20,
                    "page": 1,
                    "pages": 1,
                    "total": 1
                },
                "total": 1
            },
            "messages": {
                "matches": [
                    {
                        "channel": {
                            "id": "C2U86NC6M",
                            "is_ext_shared": false,
                            "is_mpim": false,
                            "is_org_shared": false,
                            "is_pending_ext_shared": false,
                            "is_private": false,
                            "is_shared": false,
                            "name": "general",
                            "pending_shared": []
                        },
                        "iid": "35692677-e60e-43d9-ac45-1987cea88975",
                        "next": {
                            "iid": "6f510ea1-e1d3-4f3f-bdb9-f9c6f6e9d609",
                            "text": "Thanks!",
                            "ts": "1508804378.000219",
                            "type": "message",
                            "user": "U2U85HJ7R",
                            "username": "john"
                        },
                        "permalink": "https://example.slack.com/archives/C2U86NC6M/p1508804330000296",
                        "previous": {
                            "iid": "aba8603c-0543-4fb2-9118-a5ac85f3d138",
                            "text": "Can you send me the Slack workflow diagram?",
                            "ts": "1508804301.000026",
                            "type": "message",
                            "user": "U2U85HJ7R",
                            "username": "john"
                        },
                        "team": "T2U81E2FZ",
                        "text": "uploaded a file: <https://example.slack.com/files/U2U85N1RZ/F7PKF1NR7/slack_workflow_diagram.png|slack workflow diagram> and commented: Sure! Here's the workflow diagram!",
                        "ts": "1508804330.000296",
                        "type": "message",
                        "user": "U2U85N1RZ",
                        "username": "amy"
                    }
                ],
                "pagination": {
                    "first": 1,
                    "last": 1,
                    "page": 1,
                    "page_count": 1,
                    "per_page": 20,
                    "total_count": 1
                },
                "paging": {
                    "count": 20,
                    "page": 1,
                    "pages": 1,
                    "total": 1
                },
                "total": 1
            },
            "ok": true,
            "posts": {
                "matches": [],
                "total": 0
            },
            "query": "diagram"
        }
        """

        payload = {"token": self._token, "query": query}

        if count is not None:
            payload["count"] = count

        if highlight is not None:
            payload["highlight"] = highlight

        if page is not None:
            payload["page"] = page

        if sort is not None:
            payload["sort"] = sort

        if sort_dir is not None:
            payload["sort_dir"] = sort_dir

        return self._get("search.all", payload=payload, **kwargs)

    def files(
        self,
        query: str,
        count: int = None,
        highlight: bool = None,
        page: int = None,
        sort: str = None,
        sort_dir: str = None,
        **kwargs
    ) -> Response:
        """
        Searches for files matching a query.
        https://api.slack.com/methods/search.files

        :param token: Authentication token bearing required scopes.
        :type str: e.g. xxxx-xxxxxxxxx-xxxx

        :param query: Search query.
        :type str: e.g. pickleface

        :param count: Number of items to return per page.
        :type int: e.g. 20

        :param highlight: Pass a value of true to enable query highlight markers (see below).
        :type bool: e.g. true

        :param page: Page number of results to return.
        :type int: e.g. 2

        :param sort: Return matches sorted by either score or timestamp.
        :type str: e.g. timestamp

        :param sort_dir: Change sort direction to ascending (asc) or descending (desc).
        :type str: e.g. asc

        :returns response:
        :type requests.Response: e.g. <Response [200]>

        example:
        >>> client = SlackTime(token='insert-your-token-here')
        >>> response = client.search.files(**your_params)
        <Response [200]>
        >>> response.json()
        {
            "files": {
                "matches": [
                    {
                        "channels": [],
                        "comments_count": 1,
                        "created": 1507850315,
                        "deanimate_gif": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_deanimate_gif.png",
                        "display_as_bot": false,
                        "editable": false,
                        "external_type": "",
                        "filetype": "gif",
                        "groups": [],
                        "id": "F7H0D7ZBB",
                        "image_exif_rotation": 1,
                        "ims": [],
                        "is_external": false,
                        "is_public": true,
                        "mimetype": "image/gif",
                        "mode": "hosted",
                        "name": "computer.gif",
                        "original_h": 313,
                        "original_w": 500,
                        "permalink": "https://eventsdemo.slack.com/files/U2U85N1RZ/F7H0D7ZBB/computer.gif",
                        "permalink_public": "https://slack-files.com/T2U81E2BB-F7H0D7ZBB-85b7f5557e",
                        "pretty_type": "GIF",
                        "preview": null,
                        "public_url_shared": false,
                        "reactions": [
                            {
                                "count": 1,
                                "name": "stuck_out_tongue_winking_eye",
                                "users": [
                                    "U2U85N1RZ"
                                ]
                            }
                        ],
                        "score": "0.38899223746309",
                        "size": 1639034,
                        "thumb_160": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_160.png",
                        "thumb_360": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_360.png",
                        "thumb_360_gif": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_360.gif",
                        "thumb_360_h": 225,
                        "thumb_360_w": 360,
                        "thumb_480": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_480.png",
                        "thumb_480_gif": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_480.gif",
                        "thumb_480_h": 300,
                        "thumb_480_w": 480,
                        "thumb_64": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_64.png",
                        "thumb_80": "https://files.slack.com/files-tmb/T2U81E2BB-F7H0D7ZBB-21624821e6/computer_80.png",
                        "timestamp": 1507850315,
                        "title": "computer.gif",
                        "top_file": false,
                        "url_private": "https://files.slack.com/files-pri/T2U81E2BB-F7H0D7ZBB/computer.gif",
                        "url_private_download": "https://files.slack.com/files-pri/T2U81E2BB-F7H0D7ZBB/download/computer.gif",
                        "user": "U2U85N1RZ",
                        "username": ""
                    }
                ],
                "pagination": {
                    "first": 1,
                    "last": 3,
                    "page": 1,
                    "page_count": 1,
                    "per_page": 20,
                    "total_count": 3
                },
                "paging": {
                    "count": 20,
                    "page": 1,
                    "pages": 1,
                    "total": 3
                },
                "total": 3
            },
            "ok": true,
            "query": "computer.gif"
        }
        """

        payload = {"token": self._token, "query": query}

        if count is not None:
            payload["count"] = count

        if highlight is not None:
            payload["highlight"] = highlight

        if page is not None:
            payload["page"] = page

        if sort is not None:
            payload["sort"] = sort

        if sort_dir is not None:
            payload["sort_dir"] = sort_dir

        return self._get("search.files", payload=payload, **kwargs)

    def messages(
        self,
        query: str,
        count: int = None,
        highlight: bool = None,
        page: int = None,
        sort: str = None,
        sort_dir: str = None,
        **kwargs
    ) -> Response:
        """
        Searches for messages matching a query.
        https://api.slack.com/methods/search.messages

        :param token: Authentication token bearing required scopes.
        :type str: e.g. xxxx-xxxxxxxxx-xxxx

        :param query: Search query.
        :type str: e.g. pickleface

        :param count: Number of items to return per page.
        :type int: e.g. 20

        :param highlight: Pass a value of true to enable query highlight markers (see below).
        :type bool: e.g. true

        :param page: Page number of results to return.
        :type int: e.g. 2

        :param sort: Return matches sorted by either score or timestamp.
        :type str: e.g. timestamp

        :param sort_dir: Change sort direction to ascending (asc) or descending (desc).
        :type str: e.g. asc

        :returns response:
        :type requests.Response: e.g. <Response [200]>

        example:
        >>> client = SlackTime(token='insert-your-token-here')
        >>> response = client.search.messages(**your_params)
        <Response [200]>
        >>> response.json()
        {
            "messages": {
                "matches": [
                    {
                        "channel": {
                            "id": "C12345678",
                            "is_ext_shared": false,
                            "is_mpim": false,
                            "is_org_shared": false,
                            "is_pending_ext_shared": false,
                            "is_private": false,
                            "is_shared": false,
                            "name": "general",
                            "pending_shared": []
                        },
                        "iid": "cb64bdaa-c1e8-4631-8a91-0f78080113e9",
                        "permalink": "https://hitchhikers.slack.com/archives/C12345678/p1508284197000015",
                        "team": "T12345678",
                        "text": "The meaning of life the universe and everything is 42.",
                        "ts": "1508284197.000015",
                        "type": "message",
                        "user": "U2U85N1RV",
                        "username": "roach"
                    },
                    {
                        "channel": {
                            "id": "C12345678",
                            "is_ext_shared": false,
                            "is_mpim": false,
                            "is_org_shared": false,
                            "is_pending_ext_shared": false,
                            "is_private": false,
                            "is_shared": false,
                            "name": "random",
                            "pending_shared": []
                        },
                        "iid": "9a00d3c9-bd2d-45b0-988b-6cff99ae2a90",
                        "permalink": "https://hitchhikers.slack.com/archives/C12345678/p1508795665000236",
                        "team": "T12345678",
                        "text": "The meaning of life the universe and everything is 101010",
                        "ts": "1508795665.000236",
                        "type": "message",
                        "user": "",
                        "username": "robot overlord"
                    }
                ],
                "pagination": {
                    "first": 1,
                    "last": 2,
                    "page": 1,
                    "page_count": 1,
                    "per_page": 20,
                    "total_count": 2
                },
                "paging": {
                    "count": 20,
                    "page": 1,
                    "pages": 1,
                    "total": 2
                },
                "total": 2
            },
            "ok": true,
            "query": "The meaning of life the universe and everything"
        }
        """

        payload = {"token": self._token, "query": query}

        if count is not None:
            payload["count"] = count

        if highlight is not None:
            payload["highlight"] = highlight

        if page is not None:
            payload["page"] = page

        if sort is not None:
            payload["sort"] = sort

        if sort_dir is not None:
            payload["sort_dir"] = sort_dir

        return self._get("search.messages", payload=payload, **kwargs)
