from marshmallow import (
    Schema,
    fields,
    validate,
    pre_load,
)

from ...utils.utils import pre_load_date_fields


class PatentResourceSchema(Schema):
    not_blank = validate.Length(min=1, error='Field cannot be blank')

    id = fields.Integer(dump_only=True)
    patent_number = fields.String(required=True)
    jurisdiction = fields.String(required=True)
    patent_application_id = fields.Integer(allow_none=True)
    appl_id = fields.String(allow_none=True)
    app_grp_art_number = fields.String(allow_none=True)
    app_type = fields.String(allow_none=True)
    country_code = fields.String(allow_none=True)
    document_number = fields.String(allow_none=True)
    kind_code = fields.String(allow_none=True)
    primary_identifier = fields.String(allow_none=True)
    abstract_text = fields.String(allow_none=True)
    description = fields.String(allow_none=True)
    applicant = fields.String(allow_none=True)
    inventors = fields.String(allow_none=True)
    title = fields.String(allow_none=True)
    url = fields.String(allow_none=True)
    app_class = fields.String(allow_none=True)
    app_sub_class = fields.String(allow_none=True)
    grant_date = fields.DateTime(allow_none=True)
    submission_date = fields.DateTime(allow_none=True)
    app_filing_date = fields.DateTime(allow_none=True)
    app_status_date = fields.DateTime(allow_none=True)
    app_early_pub_date = fields.DateTime(allow_none=True)
    filing_date_us = fields.DateTime(allow_none=True)
    expiration_date = fields.DateTime(allow_none=True)
    pto_adjustments = fields.String(allow_none=True)
    appl_delay = fields.String(allow_none=True)
    total_pto_days = fields.String(allow_none=True)
    assignee = fields.String(allow_none=True)
    patent_pdf_url = fields.String(allow_none=True)
    espace_url = fields.String(allow_none=True)
    docdb_family_id = fields.Integer(allow_none=True)
    inpadoc_family_id = fields.Integer(allow_none=True)

    updated_at = fields.DateTime()

    @pre_load
    def convert_string_to_datetime(self, in_data, **kwargs):
        date_fields = [
            'grant_date',
            'submission_date',
            'app_filing_date',
            'app_status_date',
            'app_early_pub_date',
            'filing_date_us',
            'expiration_date',
        ]
        in_data = pre_load_date_fields(
            in_data,
            date_fields,
            date_format='%Y/%m/%d',
        )
        return in_data


class PatentQueryParamsSchema(Schema):
    not_blank = validate.Length(min=1, error='Field cannot be blank')

    id = fields.Integer()
    patent_number = fields.String()
    app_grp_art_number = fields.String(validate=not_blank)
    jurisdiction = fields.String(validate=not_blank)
    primary_identifier = fields.String(validate=not_blank)
    applicant = fields.String(validate=not_blank)
    inventors = fields.String(validate=not_blank)


class PatentPatchSchema(Schema):
    not_blank = validate.Length(min=1, error='Field cannot be blank')

    app_grp_art_number = fields.String(allow_none=True)
    app_type = fields.String(allow_none=True)
    appl_id = fields.String(allow_none=True)
    country_code = fields.String()
    document_number = fields.String()
    kind_code = fields.String()
    primary_identifier = fields.String()
    abstract_text = fields.String()
    applicant = fields.String()
    inventors = fields.String()
    title = fields.String(allow_none=True)
    url = fields.String(allow_none=True)
    grant_date = fields.DateTime(allow_none=True)
    submission_date = fields.DateTime(allow_none=True)
    app_filing_date = fields.DateTime(allow_none=True)
    app_status_date = fields.DateTime(allow_none=True)
    app_early_pub_date = fields.DateTime(allow_none=True)
    filing_date_us = fields.DateTime(allow_none=True)
    expiration_date = fields.DateTime(allow_none=True)
    pto_adjustments = fields.String(allow_none=True)
    appl_delay = fields.String(allow_none=True)
    total_pto_days = fields.String(allow_none=True)
    assignee = fields.String()
    patent_pdf_url = fields.String(allow_none=True)
    espace_url = fields.String(allow_none=True)
