# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['context_engine',
 'context_engine.commands',
 'context_engine.commands.components',
 'context_engine.commands.components.all',
 'context_engine.commands.expressions',
 'context_engine.commands.expressions.all',
 'context_engine.commands.flows',
 'context_engine.commands.flows.all']

package_data = \
{'': ['*']}

install_requires = \
['ctx>=0.1.2,<0.2.0', 'decorator==4.0.2']

setup_kwargs = {
    'name': 'context-engine',
    'version': '0.1.8',
    'description': 'Simple way to organize processing jobs. generally best when one or more specifications is involved.',
    'long_description': '# Context Engine\n\n\nSimple way to organize processing jobs. generally best when one or more specifications is involved. Processes are designed in a JSON document process broken down into steps and flow steps that work to assemble,transform, score, or create on a shared context. Excellent for mapping/templating/interfacing dynamic repetitive processes.\n\nengine and context are fully customizable to make processing to the point and fully testable. steps and custom expressions are defined in python also support valid python expressions for working with context.\n\n## Intro \nA basic process document \n\n\n````jsonc\n{\n    "process":[\n        {\n            /*\n            Steps reference engine components with the signature in this case of:\n                init(context:Context)\n            note the name of context can be anything but must map to the name of\n            engine component to do processing. while not required each step can \n            only have one step.\n            */\n            "step":"init",\n            \n            // args are passed to the step code as context.args you probably shouldn\'t\n            // hard code a value here but you can\n            "args":"models/sample.json"\n        },\n        {\n            /*\n            The most basic step is just an array of one or more python/context\n            expressions. assignments are not allowed but can be achieved with set\n            expression.\n            */\n            "expressions":[\n                "set(\'default_project_dirs\',[])",\n                "default_project_dirs.append(\'{project.path}\')"\n            ]\n        },\n        {\n            /*\n            steps can contain a single step and a list of expressions when this\n            occurs the expressions are always run first. They can set arguments for\n            the step. In this case a string is created on the context a = \'a thing\'\n            argsAsReff() resolves the name in arguments as a reference.\n            */\n            "expressions":[\n                "set(\'a\',\'a thing\')",\n                "argsAsReff()"\n            ],\n            "step":"other_step",\n            "args":"a"\n        },\n        {\n            /*\n            flow steps change the processing control dynamically and can stack\n            according to application a for each loops over a collection dynamically\n            placing the current item in var.Flow steps also support expressions\n            however these execute once before flow logic. flow steps also share locals\n            of all sub steps.\n            */\n            "flow": "for each",\n            "collection":"default_project_dirs",\n            "var": "i",\n            /*\n            vars are created at locals.varName or context.locals.varName. i is removed\n            after the block is complete.behavior uses locals so var = locals.i. locals\n            are reset with each step block unless steps are nested in flow. when\n            nested in a flow locals persist through the duration of the flow for all\n            subsequently nested blocks\n            */\n            "steps":[\n                {\n                    /*\n                    steps of a flow can contain any number of flow steps or \n                    steps and expressions.expressions are run first and can work on\n                    same args as other expressions and the step.\n\n                    a copy of locals is shared with step in a flow block.\n                    */\n                    "expressions":[\n                        "custom_expression(i)"\n                    ],\n                    "step":"another_step",\n                    "args":"i"\n                }\n            ]\n        }\n    ]\n}\n\n````\n\n## Processing \nis done by the engine on the context. The first thing todo is to setup a process document while also creating engine and context. \n\n`init_engine(processJson)` expects the follow schema as a process document\n\n````jsonc\n{\n    "process":[\n        // list of steps and flow steps.\n    ]\n}\n````\n\n\nThen best way is by creating a factory function that calls the factory init_engine(processJson) this creates and returns an instance of Engine, and Context.\n\n\n````python\n# Example factory \ndef my_data_engine_factory():\n    \n    processJson = loadFromSomewhere()\n    # returns configured engine and context with basic expressions added to \n    # generic context.\n    engine context init_engine(processJson)\n\n    #setup steps each is created with a decorator and passed context in signature\n    @engine.component(name=\'init\')\n    def initialize_project(context:Context):\n        pass\n\n    @engine.component(name=\'another_step\')\n    def some_weird_name(context:Context):\n        pass\n    \n    # setup expressions. expressions are just python functions that can be called \n    # in addition to regular python expressions.\n    @context.expression(name="log")\n    def log_process(name:str,level:str,**kwargs):\n        pass\n\n\n    return (engine, context)\n\ndef do_the_thing():\n    engine context = my_data_engine_factory()\n\n    engine.run()\n\n    # context does not need to have these properties\n    if context.error:\n        pass\n    else:\n        return context.final_product\n\n````\n\n## Steps:\n\nA step is executed on the the context in order of how it appears in the the json config document.\nno member is required but each step shall supply either a step or expressions block or both.\nargs are allowed and passed to context.args for each step. each step has access to local variables on context.locals or just locals.any_var that are available only for the duration of the step block.\nexpressions are executed first in-order before execution of step if both are present. this is designed to work with loosely typed json so members that are blank are ignored and don\'t need to be specified in the document. \n        \nSteps are added with the syntax:\n````python\n@engine_instance_name.component(name=optional__name__if_None)\ndef name_of_step(context:Context):\n    pass\n````\n````json\n{\n    "expressions":[\n        "some_expression()"\n    ],    \n    "step": "name_of_step",\n    "args":"arg value for step"\n}\n````\n        \n## Flowsteps:\na flow step must contain a member flow with one of the following flows.\n`if, while, do while, for each, try, block`. A copy of flow step locals are available for all sub (step and flow steps). flow steps can also contain a list of expressions these are executed before flow logic and do not have access to flow control variables \n\n````jsonc\n{\n    "expressions":[\n        "some_expression()"\n    ],    \n    "flow": "name_of_flow",\n    "Steps":[\n        //... steps / flow steps ...\n        \n    ],\n    //...\n}\n````\n\n<div>\n&nbsp\n</div>\n\n### **If**\nrequire a list of python expressions that must all return true\n````json\n"conditions":[ \n    "i.type == \'video\'"\n],\n````\n* steps are executed when conditions all return true.\n    ````json\n    {\n        "flow": "if",\n        "conditions":[\n        "i.type == \'video\'"\n        ],\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n            {\n                "expressions":[\n                    "argsAsReff()"\n                ],\n                "step":"evaluate_video_actions",\n                "args":"i"\n\n            }\n        ]\n    }\n    ````\n* if can optionally contain a block for `"elsesteps":[]` as block to execute when conditions fail.\n    ````json\n    {\n        "flow": "if",\n        "conditions":[\n        "i.type == \'video\'"\n        ],\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n        ],\n        "elsesteps":[\n            {\n                "expressions":[\n                    "do_something_else"\n                ]\n            }\n        ]\n    }\n    ````\n<div>\n&nbsp\n</div>\n\n### **While**\nrequire a list of python expressions that must all return true. Supports adding a variable for loop counter. if none specified locals._ is used\n````json\n"conditions":[ \n    "i.type == \'video\'"\n],\n````\n* steps are executed while conditions all return true.\n    ````json\n    {\n        "flow": "while",\n        "conditions":[\n        "i.type == \'video\'"\n        ],\n        "var": "loop_counter",\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n            {\n                "expressions":[\n                    "argsAsReff()"\n                ],\n                "step":"evaluate_video_actions",\n                "args":"i"\n\n            }\n        ]\n    }\n    ````\n<div>\n&nbsp\n</div>\n\n### **Do while**\nrequire a list of python expressions that must all return true. Supports adding a variable for loop counter. if none specified locals._ is used\n````json\n"conditions":[ \n    "i.type == \'video\'"\n],\n````\n* executes steps prior to evaluating conditions then while conditions all return true.\n    ````json\n    {\n        "flow": "do while",\n        "conditions":[\n        "i.type == \'video\'"\n        ],\n        "var": "loop_counter",\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n            {\n                "expressions":[\n                    "argsAsReff()"\n                ],\n                "step":"evaluate_video_actions",\n                "args":"i"\n\n            }\n        ]\n    }\n    ````\n<div>\n&nbsp\n</div>\n\n### **Try**\n* steps are executed when until exception is raised. \n    ````json\n    {\n        "flow": "try",\n        "var":"exception_name",\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n            {\n                "expressions":[\n                    "argsAsReff()"\n                ],\n                "step":"evaluate_video_actions",\n                "args":"i"\n\n            }\n        ]\n    }\n    ````\n* Try can optionally contain a block for `"catchsteps":[]` as block to execute after exception and/or var to store the exception. \nThe raised exception is stored in locals._ when var is omitted\n    ````jsonc\n    {\n        "flow": "try",\n        "var":"exception_name",\n        "steps":[\n            {\n                "expressions":[\n                    "create_video(i.name,i.path)"\n                ]\n            },\n        ],\n        "catchsteps":[\n            {\n                "expressions":[\n                    "cleanupThing(locals.exception_name)"\n                ]\n            }\n        ]\n    }\n    ````\n<div>\n&nbsp\n</div>\n\n### **For each**\nrequires the name of an iterable collection somewhere on the context and a variable name to store current item. the variable is created by the block at `locals.[variableName]` or `context.locals.[variableName]` if the collection is enumerable add a list of variable names.\n\n````json\n    "collection":"default_project_dirs",\n    "var": "i",\n````\n* steps are executed while conditions all return true.\n    ````json\n    {\n        "flow": "for each",\n        "collection":"default_project_dirs",\n        "var": ["k","v"],\n        "steps":[\n            {\n                "expressions":[\n                    "argsAsReff()"\n                ],\n                "step":"create_dir_structure",\n                "args":"i"\n            }\n        ]\n    }\n    ````\n\n### **Block**\nsimple way to group related activities and share locals requires an array of steps also supports expressions.\n\n````json\n    "steps":[],\n````\n* steps are executed while conditions all return true.\n    ````jsonc\n    {\n        "flow": "block",\n        "steps":[\n            {\n                "expressions":[\n                    "set(\'locals.thing\',\'otherThing\')"\n                ],\n\n            },\n            {\n                "expressions":[\n                    // Locals.thing still has otherThing\n                    "set(\'test\',locals.thing)"\n                ],\n\n            }\n\n        ]\n    },\n    {\n        "expressions":[\n            // Error Locals.thing gone.\n            "set(\'test\',locals.thing)"\n        ],\n\n    }\n    ```` \n\n\n\n## Steps and Expressions\nEach named step executes a configured code piece for that step. Steps are passed a copy of the context.\n\nExpressions can be run in a step `context.expressionName(signature)` \n\n***Note context is injected so no need to specify context in expression signature. when calling from either python through `context.expressionName(signature)` or `expressionName(signature)` but a variable representing context must be the first argument to the expression definition in python***\n\n\n````python\n# Start by creating a new context engine context\n\nengine,context = engine_builder() \n\n# context here is basically a workspace where jobs can be worked on.\n\n\n # The context is based on a dictionary that allows both self.property and self[\'prop\'] access and assignment\n # setup area for processing\n\ncontext.dbc = sqlite3.connect(db)\n\ncontext.list_of_things = []\n\n# this expression will be called as name_other(name,other) context is managed internally\n@context.expression(name=\'name_other\'):\ndef fun_with_names(context,name:str,other:str)\n    context.list_of_things.append(f\'{name}_{other}\')\n\n#Defining a engine component if the name is omitted the function name is used. engine components are not available\n# as expressions but accept args that can be any type.\n@engine.component(name="init")\ndef init(context):\n    pass\n\n````',
    'author': 'vkldwwk',
    'author_email': 'jeff.elliott.dev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vkldwwk/context-engine',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
