# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nbtlib', 'nbtlib.contrib', 'nbtlib.contrib.minecraft', 'nbtlib.literal']

package_data = \
{'': ['*']}

install_requires = \
['numpy']

entry_points = \
{'console_scripts': ['nbt = nbtlib.cli:main']}

setup_kwargs = {
    'name': 'nbtlib',
    'version': '2.0.4',
    'description': 'A python package to read and edit nbt data',
    'long_description': '# nbtlib\n\n[![GitHub Actions](https://github.com/vberlier/nbtlib/workflows/CI/badge.svg)](https://github.com/vberlier/nbtlib/actions)\n[![PyPI](https://img.shields.io/pypi/v/nbtlib.svg)](https://pypi.org/project/nbtlib/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/nbtlib.svg)](https://pypi.org/project/nbtlib/)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/ambv/black)\n[![Discord](https://img.shields.io/discord/900530660677156924?color=7289DA&label=discord&logo=discord&logoColor=fff)](https://discord.gg/98MdSGMm8j)\n\n> A python library to read and edit [nbt data](http://wiki.vg/NBT). Requires\n> python 3.8.\n\n**Features**\n\n- Create, read and edit nbt files\n- Supports gzipped and uncompressed files\n- Supports big-endian and little-endian files\n- Parse and serialize raw nbt data\n- Define tag schemas that automatically enforce predefined tag types\n- Convert nbt between binary form and literal notation\n- Use nbt paths to access deeply nested properties\n- Includes a CLI to quickly perform read/write/merge operations\n\n## Installation\n\n> **Notice 🚧**\n>\n> Version `2.0` is actively being worked on and is not stable yet. You should probably keep using version `1.12.1` for the moment. You can check out the `2.0` roadmap in the [`nbtlib 2.0` issue](https://github.com/vberlier/nbtlib/issues/156).\n\nThe package can be installed with `pip`.\n\n```bash\n$ pip install "nbtlib==1.12.1"\n```\n\n## Basic usage\n\nThe following examples will give you a very basic overview of what you\ncan do. For more advanced examples, check out the\n"[Usage](https://github.com/vberlier/nbtlib/blob/main/docs/Usage.ipynb)"\nnotebook in the docs folder.\n\n### Reading files\n\nThe `nbtlib.load` function can be used to load nbt files as `nbtlib.File` objects. Every nbt tag inherits from\nits python counterpart. This means that all the builtin operations defined on the python counterpart can be used on nbt tags.\n\n```python\nimport nbtlib\n\nnbt_file = nbtlib.load(\'bigtest.nbt\')\nassert nbt_file[\'intTest\'] == 2147483647\n```\n\nFor example, instances of `nbtlib.File` inherit from regular `Compound` tags, which themselves inherit from the builtin python dictionary `dict`. Similarly, instances of `Int` tags inherit from the builtin class `int`.\n\nFor more details on loading nbt files and how to work with nbt tags check out the "[Usage](https://github.com/vberlier/nbtlib/blob/main/docs/Usage.ipynb)"\nnotebook.\n\n### Editing files\n\nYou can use instances of `nbtlib.File` as context managers in order to save modifications\nautomatically at the end of the `with` statement.\n\n```python\nimport nbtlib\nfrom nbtlib.tag import Int\n\nwith nbtlib.load(\'demo.nbt\') as demo:\n    demo[\'counter\'] = Int(demo[\'counter\'] + 1)\n```\n\nYou can also call the `save` method manually.\n\n```python\nimport nbtlib\nfrom nbtlib.tag import Int\n\ndemo = nbtlib.load(\'demo.nbt\')\ndemo[\'counter\'] = Int(demo[\'counter\'] + 1)\ndemo.save()\n```\n\nFor more details on the `save` method check out the "[Usage](https://github.com/vberlier/nbtlib/blob/main/docs/Usage.ipynb)"\nnotebook.\n\n### Using schemas\n\n`nbtlib` allows you to define `Compound` schemas that enforce a specific tag type\nfor any given key.\n\n```python\nfrom nbtlib import schema\nfrom nbtlib.tag import Short, String\n\nMySchema = schema(\'MySchema\', {\n    \'foo\': String,\n    \'bar\': Short\n})\n\nmy_object = MySchema({\'foo\': \'hello world\', \'bar\': 21})\n\nassert isinstance(my_object, MySchema)\nassert isinstance(my_object[\'foo\'], String)\n```\n\nFor more details on schemas check out the "[Usage](https://github.com/vberlier/nbtlib/blob/main/docs/Usage.ipynb)"\nnotebook.\n\n### Nbt literals\n\nYou can parse nbt literals using the `nbtlib.parse_nbt` function.\n\n```python\nfrom nbtlib import parse_nbt\nfrom nbtlib.tag import String, List, Compound, IntArray\n\nmy_compound = parse_nbt(\'{foo: [hello, world], bar: [I; 1, 2, 3]}\')\nassert my_compound == Compound({\n    \'foo\': List[String]([\'hello\', \'world\']),\n    \'bar\': IntArray([1, 2, 3])\n})\n```\n\nNbt tags can be serialized to their literal representation with the `nbtlib.serialize_tag` function.\n\n```python\nfrom nbtlib import serialize_tag\nfrom nbtlib.tag import String, List, Compound, IntArray\n\nmy_compound = Compound({\n    \'foo\': List[String]([\'hello\', \'world\']),\n    \'bar\': IntArray([1, 2, 3])\n})\nassert serialize_tag(my_compound) == \'{foo: ["hello", "world"], bar: [I; 1, 2, 3]}\'\n```\n\nFor more details on nbt literals check out the "[Usage](https://github.com/vberlier/nbtlib/blob/main/docs/Usage.ipynb)"\nnotebook.\n\n### Nbt paths\n\nNbt paths can be used to access deeply nested properties in nbt data. The implementation is based on information available on the [Minecraft wiki](https://minecraft.gamepedia.com/Commands/data#NBT_path).\n\n```python\nfrom nbtlib import parse_nbt, Path\n\ndata = parse_nbt(\'{a: [{b: {c: 42}}]}\')\nassert data[\'a\'][0][\'b\'][\'c\'] == 42\nassert data[Path(\'a[0].b.c\')] == 42\n```\n\nYou can retrieve, modify and delete multiple properties at the same time.\n\n```python\nfrom nbtlib import parse_nbt, Path\nfrom nbtlib.tag import Int\n\ndata = parse_nbt(\'{foo: [{a: 1, b: {c: 42}}, {a: 2, b: {c: 0}}]}\')\n\ndata[Path(\'foo[].a\')] = Int(99)\nassert str(data) == \'{foo: [{a: 99, b: {c: 42}}, {a: 99, b: {c: 0}}]}\'\n\nassert data.get_all(Path(\'foo[].b.c\')) == [42, 0]\n\ndel data[Path(\'foo[].b{c: 0}\')]\nassert str(data) == \'{foo: [{a: 99, b: {c: 42}}, {a: 99}]}\'\n```\n\nNbt paths are immutable but can be manipulated and combined together to form new paths.\n\n```python\nfrom nbtlib import Path\nfrom nbtlib.tag import Compound\n\npath = Path()[\'hello\'][\'world\']\nassert path[:][Compound({\'a\': Int(0)})] == \'hello.world[{a: 0}]\'\n\nassert path + path == \'hello.world.hello.world\'\nassert sum(\'abcdef\', Path()) == \'a.b.c.d.e.f\'\n\nassert Path()[0] + \'foo{a: 1}\' + \'{b: 2}.bar\' == \'[0].foo{a: 1, b: 2}.bar\'\n\nassert path[\'key.with.dots\'] == \'hello.world."key.with.dots"\'\nassert path + \'key.with.dots\' == \'hello.world.key.with.dots\'\n\n```\n\n## Command-line interface\n\nThe package comes with a small CLI that makes it easy to quickly perform\nbasic operations on nbt files.\n\n```\n$ nbt --help\nusage: nbt [-h] [-r | -s] [-w <nbt> | -m <nbt>] [--plain] [--little]\n           [--compact] [--pretty] [--unpack] [--json]\n           [--path <path>] [--find <path>]\n           <file>\n\nPerform operations on nbt files.\n\npositional arguments:\n  <file>         the target file\n\noptional arguments:\n  -h, --help     show this help message and exit\n  -r             read nbt data from a file\n  -s             read snbt from a file\n  -w <nbt>       write nbt to a file\n  -m <nbt>       merge nbt into a file\n  --plain        don\'t use gzip compression\n  --little       use little-endian format\n  --compact      output compact snbt\n  --pretty       output indented snbt\n  --unpack       output interpreted nbt\n  --json         output nbt as json\n  --path <path>  output all the matching tags\n  --find <path>  recursively find the first matching tag\n```\n\n### Read nbt data\n\nYou can read nbt files by using the `-r` option. This will print the\nliteral notation of the binary nbt data.\n\n```bash\n$ nbt -r my_file.nbt\n{foo: [1, 2, 3], bar: "Hello, world!"}\n```\n\nYou can use the following command if you want to save the output into a\nfile.\n\n```bash\n$ nbt -r my_file.nbt > my_file.snbt\n```\n\nUsing the `--compact` argument will remove all the extra whitespace from the output.\n\n```bash\n$ nbt -r my_file.nbt --compact\n{foo:[1,2,3],bar:"Hello, world!"}\n```\n\nYou can use the `--pretty` argument if you want the command to output indented snbt.\n\n```bash\n$ nbt -r my_file.nbt --pretty\n{\n    foo: [1, 2, 3],\n    bar: "Hello, world!"\n}\n```\n\nThe output can be converted to json with the `--json` flag.\n\n```bash\n$ nbt -r my_file.nbt --json\n{"foo": [1, 2, 3], "bar": "Hello, world!"}\n```\n\nThe `--path` option lets you output tags that match a given path.\n\n```bash\n$ nbt -r my_file.nbt --path "bar"\n"Hello, world!"\n$ nbt -r my_file.nbt --path "foo[]"\n1\n2\n3\n```\n\nYou can combine this with the `--unpack` flag to print out the unpacked python objects.\n\n```bash\n$ nbt -r my_file.nbt --path "bar" --unpack\nHello, world!\n```\n\nIf you don\'t know exactly how to access the data you\'re interested in you can use the `--find` option to recursively try to match a given path.\n\n```bash\n$ nbt -r my_file.nbt --find "[1]"\n2\n```\n\nYou can also perform all these operations on snbt by using the `-s` option instead of the `-r` option.\n\n```bash\n$ nbt -s foo.snbt --path bar\n"Hello, world!"\n```\n\n### Write nbt data\n\nYou can write nbt data to a file by using the `-w` option. This will\nconvert the literal nbt notation to its binary form and save it in the\nspecified file.\n\n```bash\n$ nbt -w \'{foo:[1,2,3],bar:{hello:[B;1b,1b,0b,1b]}}\' my_file.nbt\n```\n\nThe file will be created if it doesn\'t already exist.\n\nYou can combine the `-w` flag with other flags to read, filter and write nbt data.\n\n```bash\n$ nbt -r my_file.nbt --path "bar" -w bar.nbt\n```\n\n### Merge nbt data\n\nFinally, you can merge some nbt data into an already existing file by\nusing the `-m` option. This will recursively update the file with\nthe values parsed from the literal argument.\n\n```bash\n$ nbt -m \'{bar:{"new key":56f}}\' my_file.nbt\n```\n\nYou can check the result by using the `-r` option.\n\n```bash\n$ nbt -r my_file.nbt\n{foo: [1, 2, 3], bar: {hello: [B; 1B, 1B, 0B, 1B], "new key": 56.0f}}\n```\n\nHere, the compound values that aren\'t present in the input literal are\nleft untouched. Using the `-w` option instead of `-m` would\noverwrite the whole file.\n\nYou can combine the `-m` flag with other flags to read, filter and merge nbt data.\n\n```bash\n$ nbt -s foo.snbt -m my_file.nbt\n```\n\n### Compression and byte order\n\nBy default, the CLI will assume that you\'re working with gzipped nbt\nfiles. If you want to read, write or merge uncompressed nbt files, you\ncan use the `--plain` option. Similarly, the default byte order is\nbig-endian so you\'ll need to use the `--little` option to perform\noperations on little-endian files.\n\n**Reading**\n\n```bash\n$ nbt -r my_file.nbt --plain --little\n{name: "Reading an uncompressed little-endian file"}\n```\n\n**Writing**\n\n```bash\n$ nbt -w \'{name:"Writing in an uncompressed little-endian file"}\' my_file.nbt --plain --little\n```\n\n**Merging**\n\n```bash\n$ nbt -m \'{name:"Merging in an uncompressed little-endian file"}\' my_file.nbt --plain --little\n```\n\n## Contributing\n\nContributions are welcome. This project uses [`poetry`](https://poetry.eustace.io/) so you\'ll need to install it first if you want to be able to work with the project locally.\n\n```sh\n$ curl -sSL https://raw.githubusercontent.com/sdispater/poetry/main/get-poetry.py | python\n```\n\nYou should now be able to install the required dependencies.\n\n```sh\n$ poetry install\n```\n\nYou can run the tests with `poetry run pytest`.\n\n```sh\n$ poetry run pytest\n```\n\n---\n\nLicense - [MIT](https://github.com/vberlier/nbtlib/blob/main/LICENSE)\n',
    'author': 'Valentin Berlier',
    'author_email': 'berlier.v@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vberlier/nbtlib',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
