# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sysrsync', 'sysrsync.helpers']

package_data = \
{'': ['*']}

install_requires = \
['toml>=0.10.0,<0.11.0']

setup_kwargs = {
    'name': 'sysrsync',
    'version': '1.1.0',
    'description': 'Simple and safe python wrapper for calling system rsync',
    'long_description': '# sysrsync\n<!-- ALL-CONTRIBUTORS-BADGE:START - Do not remove or modify this section -->\n[![All Contributors](https://img.shields.io/badge/all_contributors-1-orange.svg?style=flat-square)](#contributors-)\n<!-- ALL-CONTRIBUTORS-BADGE:END -->\nSimple and safe native rsync wrapper for Python 3\n\n[![Quality Gate Status](https://sonarcloud.io/api/project_badges/measure?project=gchamon_sysrsync&metric=alert_status)](https://sonarcloud.io/dashboard?id=gchamon_sysrsync)\n\n## Requirements\n\n* rsync\n* python 3.6+\n\n**development**:\n\n* poetry (be sure to have both poetry and pip upgraded to the latest version)\n\n## Installation\n\n`pip install sysrsync`\n\n## Basic rules\n\n- Syncs source contents by default, so it adds a trailing slash to the end of source, unless `sync_source_contents=False` is specified\n- Removes trailing slash from destination\n- Extra arguments are put right after `rsync`\n- Breaks if `source_ssh` and `destination_ssh` are both set\n\n## Usage\n\n* Basic file sync\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/foo.txt\',\n             destination=\'/home/server/bar\')\n# runs \'rsync /home/users/foo.txt /home/server/files\'\n```\n\n* sync whole folder\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/files\',\n             destination=\'/home/server/\',\n             sync_source_contents=False)\n# runs \'rsync /home/user/files /home/server\'\n```\n\n* sync folder contents\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/files\',\n             destination=\'/home/server/\',\n             sync_source_contents=True)\n# runs \'rsync /home/user/files/ /home/server\'\n```\n\n* ssh with options\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/files\',\n             destination=\'/home/server/files\',\n             destination_ssh=\'myserver\',\n             options=[\'-a\'])\n# runs \'rsync -a /home/users/files/ myserver:/home/server/files\'\n```\n\n* exclusions\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/files\',\n             destination=\'/home/server/files\',\n             destination_ssh=\'myserver\',\n             options=[\'-a\'],\n             exclusions=[\'file_to_exclude\', \'unwanted_file\'])\n# runs \'rsync -a /home/user/files/ myserver:/home/server/files --exclude file_to_exclude --exclude unwanted_file\'\n```\n* Private key\n\n```python\nimport sysrsync\n\nsysrsync.run(source=\'/home/user/files\',\n             destination=\'/home/server/files\',\n             destination_ssh=\'myserver\',\n             private_key="totally_secure_key")\n# runs \'rsync --rsh=\'ssh -i totally_secure_key\' /home/user/files/ myserver:/home/server/files\'\n```\n\n## API\n\n`sysrsync.run`\n\n| argument  | type | default | description |\n| --------- | ---- | ------- | ----------- |\n| cwd  | str  | `os.getcwd()` | working directory in which subprocess will run the rsync command |\n| strict  | bool | `True` | raises `RsyncError` when rsync return code is different than 0  |\n| verbose | bool | `False` | verbose mode: currently prints rsync command before executing |\n| **kwargs | dict | Not Applicable | arguments that will be forwarded to call to `sysrsync.get_rsync_command` |\n\n**returns**: `subprocess.CompletedProcess`\n\n**raises**:\n- `RsyncError` when `strict = True` and rsync return code is different than 0 ([Success](https://lxadm.com/Rsync_exit_codes#List_of_standard_rsync_exit_codes))\n\n`sysrsync.get_rsync_command`\n\n| argument  | type | default | description |\n| --------- | ---- | ------- | ----------- |\n| source | str | - | Source folder or file |\n| destination | str | - | Destination folder |\n| source_ssh | Optional[str] | None | Remote ssh client where source is located |\n| destination_ssh | Optional[str] | None | Remote ssh client where destination is located |\n| exclusions | Optional[Iterable[str]] | None | List of excluded patterns as in rsync\'s `--exclude` |\n| sync_source_contents | bool | True | Abstracts the elusive trailing slash behaviour that `source` normally has when using rsync directly, i.e. when a trailing slash is present in `source`, the folder\'s content is synchronized with destination. When no trailing slash is present, the folder itself is synchronized with destination. |\n| options | Optional[Iterable[str]] | None | List of options to be used right after rsync call, e.g. `[\'-a\', \'-v\']` translates to `rsync -a -v` |\n| private_key | Optional[str] | None | Configures an explicit key to be used with rsync --rsh command |\n| rsh_port |\xa0Optional[int] | None | Specify port to be used for --rsh command |\n| strict_host_key_checking | Optional[bool] | None | set StrictHostKeyChecking property for rsh #cf. https://superuser.com/questions/125324/how-can-i-avoid-sshs-host-verification-for-known-hosts |\n\n**returns**: `List[str]` -> the compiled list of commands to be used directly in `subprocess.run`\n\n**raises**:\n- `RemotesError` when both `source_ssh` and `destination_ssh` are set. Normally linux rsync distribution disallows source and destination to be both remotes.\n- `PrivateKeyError` when `private_key` doesn\'t exist\n\n# Contributing\n\n- Fork project\n- Install dependencies with `poetry install`\n- Make changes\n- Lint with `poetry run pylint ./sysrsync`\n- Test with `poetry run python -m unittest`\n- Run end-to-end tests with `bash end-to-end-tests/run-tests.sh`\n- Submit changes with a pull request\n\n## Contributors ✨\n\nThanks goes to these wonderful people ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="https://github.com/plazmakeks"><img src="https://avatars.githubusercontent.com/u/25690073?v=4?s=100" width="100px;" alt=""/><br /><sub><b>plazmakeks</b></sub></a><br /><a href="https://github.com/gchamon/sysrsync/commits?author=plazmakeks" title="Code">💻</a></td>\n  </tr>\n</table>\n\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors) specification. Contributions of any kind welcome!\n',
    'author': 'Gabriel Chamon',
    'author_email': 'gchamon@live.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/gchamon/sysrsync',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
