"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
async function handler(event, context) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        const dotJsii = await new Promise((ok, ko) => {
            tar_stream_1.extract()
                .on('entry', (headers, stream, next) => {
                if (headers.name !== 'package/.jsii') {
                    // Skip on next runLoop iteration so we avoid filling the stack.
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('error', ko)
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('error', ko)
                .once('close', () => ko(new Error('No .jsii file found in tarball!')))
                .write(tar, (err) => {
                if (err != null) {
                    ko(err);
                }
            });
        });
        const { name: packageName, version: packageVersion } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const [assembly, pkg] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: assemblyKey,
                Body: dotJsii,
                ContentType: 'text/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        result.push(created);
    }
    return result;
}
exports.handler = handler;
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
//# sourceMappingURL=data:application/json;base64,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