"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', async (batch) => {
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                // Process all remaining updates
                await Promise.all(versionInfos.map(processUpdatedVersion));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                if (context.getRemainingTimeInMillis() >= 120000 /* 2 minutes */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
        })
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${constants.STAGED_KEY_PREFIX}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            await putObject(`${constants.FAILED_KEY_PREFIX}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'text/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes) {
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isRelevantPackageVersion(infos)) {
                    result.push({ infos, modified, seq: change.seq });
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isRelevantPackageVersion(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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