"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshot = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) &&
        !project.outdir.includes("project-temp-dir")) {
        throw new Error("Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project");
    }
    const synthed = Symbol.for("synthed");
    if (synthed in project) {
        throw new Error("duplicate synth()");
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = "true";
        project.synth();
        const ignoreExts = ["png", "ico"];
        return directorySnapshot(project.outdir, {
            excludeGlobs: ignoreExts.map((ext) => `**/*.${ext}`),
        });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function directorySnapshot(root, options = {}) {
    const output = {};
    const files = glob_1.glob.sync("**", {
        ignore: [".git/**", ...(options.excludeGlobs ?? [])],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (!options.onlyFileNames) {
            if (path.extname(filePath) === ".json") {
                content = fs.readJsonSync(filePath);
            }
            else {
                content = fs.readFileSync(filePath, "utf-8");
            }
        }
        else {
            content = true;
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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