"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isAwsCodeArtifactRegistry = exports.Publisher = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11.x" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.16.0" } },
    dotnet: { dotnet: { version: "3.x" } },
};
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.runsOn = options.workflowRunsOn ?? ["ubuntu-latest"];
        this.publishTasks = options.publishTasks ?? false;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: '! git log --oneline -1 | grep -q "chore(release):"',
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghRelease = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_REF",
        ].join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        this.addPublishJob(() => {
            return {
                name: "github",
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                    GITHUB_REPOSITORY: "${{ github.repository }}",
                    GITHUB_REF: "${{ github.ref }}",
                },
                run: idempotentRelease,
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const npmToken = node_package_1.defaultNpmToken(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        this.addPublishJob((_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            return {
                name: "npm",
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps: options.prePublishSteps ?? [],
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "nuget",
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: secret(isGitHubPackages
                    ? "GITHUB_TOKEN"
                    : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "maven",
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "pypi",
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            prePublishSteps: options.prePublishSteps ?? [],
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv: {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            },
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob((_branch, _branchOptions) => ({
            name: "golang",
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GITHUB_REPO: options.githubRepo,
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(factory) {
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            const jobname = `release_${opts.name}`;
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [name, expression] of workflowEnvEntries) {
                requiredEnv.push(name);
                jobEnv[name] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${opts.name.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v3",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR,
                    },
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        run: 'echo "::set-output name=VERSION::$(cat dist/version.txt)"',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    tools: {
                        node: { version: "14.x" },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId],
                    runsOn: this.runsOn,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.60.2" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
//# sourceMappingURL=data:application/json;base64,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