"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - CHANGELOG: name of changelog file to create
 * - RELEASE_TAG_PREFIX: (optional) a prefix to apply to the release tag
 *
 */
const logging = require("../logging");
const bump_version_1 = require("./bump-version");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.RELEASE_TAG_PREFIX;
const versionrcOptions = process.env.VERSIONRCOPTIONS;
if (!versionFile) {
    throw new Error("OUTFILE is required");
}
if (!changelog) {
    throw new Error("CHANGELOG is required");
}
if (!bumpFile) {
    throw new Error("BUMPFILE is required");
}
if (!releaseTagFile) {
    throw new Error("RELEASETAG is required");
}
const majorVersion = major == null || major === "" ? undefined : parseInt(major);
if (Number.isNaN(majorVersion)) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
    // doesn't work with long customization
    versionrcOptions: JSON.parse(versionrcOptions ?? "{}"),
};
logging.debug(opts);
bump_version_1.bump(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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