"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const prefix = options.tagPrefix ?? "";
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    await fs_extra_1.mkdirp(path_1.dirname(releaseTagFile));
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({
        cwd,
        major,
        prerelease,
        prefix,
    });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture("git tag --points-at HEAD", { cwd })
        .toString("utf8")
        .split("\n");
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info("Skipping bump...");
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, ".versionrc.json");
    await generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, options.versionrcOptions);
    const cmd = ["npx", "standard-version@^9"];
    if (isFirstRelease) {
        cmd.push("--first-release");
    }
    if (prefix) {
        cmd.push(`--tag-prefix ${prefix}v`);
    }
    util_1.exec(cmd.join(" "), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_extra_1.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, "utf8"));
}
function generateVersionrcFile(rcfile, versionFile, changelogFile, skipBump, prerelease, configOptions) {
    return fs_extra_1.writeFile(rcfile, JSON.stringify({
        ...{
            packageFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            bumpFiles: [
                {
                    filename: versionFile,
                    type: "json",
                },
            ],
            commitAll: false,
            infile: changelogFile,
            prerelease: prerelease,
            header: "",
            skip: {
                commit: true,
                tag: true,
                bump: skipBump,
            },
            ...configOptions,
        },
    }, undefined, 2));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major !== undefined ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        "git",
        '-c "versionsort.suffix=-"',
        "tag",
        '--sort="-version:refname"',
        "--list",
        `"${prefixFilter}"`,
    ].join(" ");
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString("utf8");
    let tags = stdout?.split("\n");
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter((x) => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter((x) => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major ?? 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith("v")) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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