"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const github_1 = require("../github");
const projenrc_1 = require("../javascript/projenrc");
const project_1 = require("../project");
const util_1 = require("../util");
const pip_1 = require("./pip");
const poetry_1 = require("./poetry");
const projenrc_2 = require("./projenrc");
const pytest_1 = require("./pytest");
const pytest_sample_1 = require("./pytest-sample");
const python_sample_1 = require("./python-sample");
const setuptools_1 = require("./setuptools");
const venv_1 = require("./venv");
/** Allowed characters in python project names */
const PYTHON_PROJECT_NAME_REGEX = /^[A-Za-z0-9-_\.]+$/;
/**
 * Python project.
 *
 * @pjid python
 */
class PythonProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        if (!PYTHON_PROJECT_NAME_REGEX.test(options.name)) {
            throw new Error("Python projects must only consist of alphanumeric characters, hyphens, and underscores.");
        }
        this.moduleName = options.moduleName;
        this.version = options.version;
        const rcFileTypeOptions = [
            options.projenrcPython,
            options.projenrcJs,
            options.projenrcJson,
        ];
        if (util_1.multipleSelected(rcFileTypeOptions)) {
            throw new Error("Only one of projenrcPython, projenrcJs, and projenrcJson can be selected.");
        }
        // default to projenrc.py if no other projenrc type was elected
        if (options.projenrcPython ?? !util_1.anySelected(rcFileTypeOptions)) {
            new projenrc_2.Projenrc(this, options.projenrcPythonOptions);
        }
        if (options.projenrcJs ?? false) {
            new projenrc_1.Projenrc(this, options.projenrcJsOptions);
        }
        if (options.venv ?? true) {
            this.envManager = new venv_1.Venv(this, options.venvOptions);
        }
        if (options.pip ?? true) {
            this.depsManager = new pip_1.Pip(this);
        }
        if (options.setuptools ?? this.projectType === project_1.ProjectType.LIB) {
            this.packagingManager = new setuptools_1.Setuptools(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                setupConfig: options.setupConfig,
            });
        }
        // if (options.conda ?? false) {
        //   this.depsManager = new Conda(this, options);
        //   this.envManager = this.depsManager;
        // }
        // if (options.pipenv ?? false) {
        //   this.depsManager = new Pipenv(this, options);
        //   this.envManager = this.depsManager;
        // }
        if (options.poetry ?? false) {
            const poetry = new poetry_1.Poetry(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                poetryOptions: {
                    readme: options.readme?.filename ?? "README.md",
                    ...options.poetryOptions,
                },
            });
            this.depsManager = poetry;
            this.envManager = poetry;
            this.packagingManager = poetry;
        }
        if (!this.envManager) {
            throw new Error("At least one tool must be chosen for managing the environment (venv, conda, pipenv, or poetry).");
        }
        if (!this.depsManager) {
            throw new Error("At least one tool must be chosen for managing dependencies (pip, conda, pipenv, or poetry).");
        }
        if (!this.packagingManager && this.projectType === project_1.ProjectType.LIB) {
            throw new Error("At least one tool must be chosen for managing packaging (setuptools or poetry).");
        }
        if (Number(options.venv ?? true) + Number(options.poetry ?? false) > 1) {
            throw new Error("More than one component has been chosen for managing the environment (venv, conda, pipenv, or poetry)");
        }
        if (Number(options.pip ?? true) + Number(options.poetry ?? false) > 1) {
            throw new Error("More than one component has been chosen for managing dependencies (pip, conda, pipenv, or poetry)");
        }
        if (Number(options.setuptools ?? true) + Number(options.poetry ?? false) >
            1) {
            throw new Error("More than one component has been chosen for managing packaging (setuptools or poetry)");
        }
        if (options.pytest ?? true) {
            this.pytest = new pytest_1.Pytest(this, options.pytestOptions);
            new pytest_sample_1.PytestSample(this, {
                moduleName: this.moduleName,
                testdir: this.pytest.testdir,
            });
        }
        if (options.sample ?? true) {
            new python_sample_1.PythonSample(this, {
                dir: this.moduleName,
            });
        }
        for (const dep of options.deps ?? []) {
            this.addDependency(dep);
        }
        for (const dep of options.devDeps ?? []) {
            this.addDevDependency(dep);
        }
        this.addDefaultGitIgnore();
    }
    /**
     * Adds default gitignore options for a Python project based on
     * https://github.com/github/gitignore/blob/master/Python.gitignore
     */
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Byte-compiled / optimized / DLL files", "__pycache__/", "*.py[cod]", "*$py.class", "", "# C extensions", "*.so", "", "# Distribution / packaging", ".Python", "build/", "develop-eggs/", "dist/", "downloads/", "eggs/", ".eggs/", "lib/", "lib64/", "parts/", "sdist/", "var/", "wheels/", "share/python-wheels/", "*.egg-info/", ".installed.cfg", "*.egg", "MANIFEST", "", "# PyInstaller", "#  Usually these files are written by a python script from a template", "#  before PyInstaller builds the exe, so as to inject date/other infos into it.", "*.manifest", "*.spec", "", "# Installer logs", "pip-log.txt", "pip-delete-this-directory.txt", "", "# Unit test / coverage reports", "htmlcov/", ".tox/", ".nox/", ".coverage", ".coverage.*", ".cache", "nosetests.xml", "coverage.xml", "*.cover", "*.py,cover", ".hypothesis/", ".pytest_cache/", "cover/", "", "# Translations", "*.mo", "*.pot", "", "# Django stuff:", "*.log", "local_settings.py", "db.sqlite3", "db.sqlite3-journal", "", "# Flask stuff:", "instance/", ".webassets-cache", "", "# Scrapy stuff:", ".scrapy", "", "# Sphinx documentation", "docs/_build/", "", "# PyBuilder", ".pybuilder/", "target/", "", "# Jupyter Notebook", ".ipynb_checkpoints", "", "# IPython", "profile_default/", "ipython_config.py", "", "# PEP 582; used by e.g. github.com/David-OConnor/pyflow", "__pypackages__/", "", "# Celery stuff", "celerybeat-schedule", "celerybeat.pid", "", "# SageMath parsed files", "*.sage.py", "", "# Environments", ".env", ".venv", "env/", "venv/", "ENV/", "env.bak/", "venv.bak/", "", "# Spyder project settings", ".spyderproject", ".spyproject", "", "# Rope project settings", ".ropeproject", "", "# mkdocs documentation", "/site", "", "# mypy", ".mypy_cache/", ".dmypy.json", "dmypy.json", "", "# Pyre type checker", ".pyre/", "", "# pytype static type analyzer", ".pytype/", "", "# Cython debug symbols", "cython_debug/");
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        return this.depsManager.addDependency(spec);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        return this.depsManager.addDevDependency(spec);
    }
    postSynthesize() {
        super.postSynthesize();
        this.envManager.setupEnvironment();
        this.depsManager.installDependencies();
    }
}
exports.PythonProject = PythonProject;
_a = JSII_RTTI_SYMBOL_1;
PythonProject[_a] = { fqn: "projen.python.PythonProject", version: "0.60.2" };
//# sourceMappingURL=data:application/json;base64,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