"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvePythonImportName = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.rcfile = options.filename ?? ".projenrc.py";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec("python .projenrc.py");
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = path_1.join(this.project.outdir, this.rcfile);
        // skip if file exists
        if (fs_extra_1.existsSync(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        const importName = resolvePythonImportName(jsiiFqn, jsiiManifest);
        emit(toPythonImport(importName));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        fs_extra_1.mkdirpSync(path_1.dirname(pythonFile));
        fs_extra_1.writeFileSync(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.60.2" };
function resolvePythonImportName(jsiiFqn, jsiiManifest) {
    const moduleName = jsiiManifest?.targets?.python?.module;
    // Module name prefix should take precedence in the event moduleName !== fqn prefix
    return moduleName
        ? [moduleName, ...jsiiFqn.split(".").slice(1)].join(".")
        : jsiiFqn;
}
exports.resolvePythonImportName = resolvePythonImportName;
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return case_1.snake(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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