"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ObjectFile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const file_1 = require("./file");
const util_1 = require("./util");
/**
 * Represents an Object file.
 */
class ObjectFile extends file_1.FileBase {
    constructor(project, filePath, options) {
        super(project, filePath, options);
        this.obj = options.obj ?? {};
        this.omitEmpty = options.omitEmpty ?? false;
        this.rawOverrides = {};
    }
    /**
     * Adds an override to the synthesized object file.
     *
     * If the override is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * To include a literal `.` in the property name, prefix with a `\`. In most
     * programming languages you will need to write this as `"\\."` because the
     * `\` itself will need to be escaped.
     *
     * For example,
     * ```typescript
     * project.tsconfig.file.addOverride('compilerOptions.alwaysStrict', true);
     * project.tsconfig.file.addOverride('compilerOptions.lib', ['dom', 'dom.iterable', 'esnext']);
     * ```
     * would add the overrides
     * ```json
     * "compilerOptions": {
     *   "alwaysStrict": true,
     *   "lib": [
     *     "dom",
     *     "dom.iterable",
     *     "esnext"
     *   ]
     *   ...
     * }
     * ...
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        override values in complex types. Any intermediate keys
     *        will be created as needed.
     * @param value - The value. Could be primitive or complex.
     */
    addOverride(path, value) {
        const parts = splitOnPeriods(path);
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null &&
                typeof curr[key] === "object" &&
                !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    /**
     * Adds to an array in the synthesized object file.
     *
     * If the array is nested, separate each nested level using a dot (.) in the path parameter.
     * If there is an array as part of the nesting, specify the index in the path.
     *
     * To include a literal `.` in the property name, prefix with a `\`. In most
     * programming languages you will need to write this as `"\\."` because the
     * `\` itself will need to be escaped.
     *
     * For example, with the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules"],
     *   "lib": ["es2019"]
     *   ...
     * }
     * ...
     * ```
     *
     * ```typescript
     * project.tsconfig.file.addToArray('compilerOptions.exclude', 'coverage');
     * project.tsconfig.file.addToArray('compilerOptions.lib', 'dom', 'dom.iterable', 'esnext');
     * ```
     * would result in the following object file
     * ```json
     * "compilerOptions": {
     *   "exclude": ["node_modules", "coverage"],
     *   "lib": ["es2019", "dom", "dom.iterable", "esnext"]
     *   ...
     * }
     * ...
     * ```
     *
     * @param path - The path of the property, you can use dot notation to
     *        att to arrays in complex types. Any intermediate keys
     *        will be created as needed.
     * @param values - The values to add. Could be primitive or complex.
     */
    addToArray(path, ...values) {
        const parts = splitOnPeriods(path);
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null &&
                typeof curr[key] === "object" &&
                !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        const lastKey = parts.shift();
        if (Array.isArray(curr[lastKey])) {
            curr[lastKey].push(...values);
        }
        else {
            curr[lastKey] = { __$APPEND: values };
        }
    }
    /**
     * Syntactic sugar for `addOverride(path, undefined)`.
     * @param path The path of the value to delete
     */
    addDeletionOverride(path) {
        this.addOverride(path, undefined);
    }
    synthesizeContent(resolver) {
        const obj = this.obj;
        const resolved = resolver.resolve(obj, {
            omitEmpty: this.omitEmpty,
        }) ?? undefined;
        if (resolved) {
            util_1.deepMerge([resolved, this.rawOverrides], true);
        }
        return resolved ? JSON.stringify(resolved, undefined, 2) : undefined;
    }
}
exports.ObjectFile = ObjectFile;
_a = JSII_RTTI_SYMBOL_1;
ObjectFile[_a] = { fqn: "projen.ObjectFile", version: "0.60.2" };
/**
 * Split on periods while processing escape characters \
 */
function splitOnPeriods(x) {
    // Build this list in reverse because it's more convenient to get the "current"
    // item by doing ret[0] than by ret[ret.length - 1].
    const ret = [""];
    for (let i = 0; i < x.length; i++) {
        if (x[i] === "\\" && i + 1 < x.length) {
            ret[0] += x[i + 1];
            i++;
        }
        else if (x[i] === ".") {
            ret.unshift("");
        }
        else {
            ret[0] += x[i];
        }
    }
    ret.reverse();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2JqZWN0LWZpbGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvb2JqZWN0LWZpbGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBOEQ7QUFFOUQsaUNBQW1DO0FBcUJuQzs7R0FFRztBQUNILE1BQXNCLFVBQVcsU0FBUSxlQUFRO0lBaUIvQyxZQUFZLE9BQWdCLEVBQUUsUUFBZ0IsRUFBRSxPQUEwQjtRQUN4RSxLQUFLLENBQUMsT0FBTyxFQUFFLFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVsQyxJQUFJLENBQUMsR0FBRyxHQUFHLE9BQU8sQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDO1FBQzdCLElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUM7UUFDNUMsSUFBSSxDQUFDLFlBQVksR0FBRyxFQUFFLENBQUM7SUFDekIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQ0c7SUFDSSxXQUFXLENBQUMsSUFBWSxFQUFFLEtBQVU7UUFDekMsTUFBTSxLQUFLLEdBQUcsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ25DLElBQUksSUFBSSxHQUFRLElBQUksQ0FBQyxZQUFZLENBQUM7UUFFbEMsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUN2QixNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFHLENBQUM7WUFFM0IsOERBQThEO1lBQzlELHNDQUFzQztZQUN0QyxNQUFNLFFBQVEsR0FDWixJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSTtnQkFDakIsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssUUFBUTtnQkFDN0IsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQzVCLElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ2IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQzthQUNoQjtZQUVELElBQUksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDbEI7UUFFRCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFHLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0NHO0lBQ0ksVUFBVSxDQUFDLElBQVksRUFBRSxHQUFHLE1BQVc7UUFDNUMsTUFBTSxLQUFLLEdBQUcsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ25DLElBQUksSUFBSSxHQUFRLElBQUksQ0FBQyxZQUFZLENBQUM7UUFFbEMsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUN2QixNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFHLENBQUM7WUFFM0IsOERBQThEO1lBQzlELHNDQUFzQztZQUN0QyxNQUFNLFFBQVEsR0FDWixJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSTtnQkFDakIsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssUUFBUTtnQkFDN0IsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQzVCLElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ2IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQzthQUNoQjtZQUVELElBQUksR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDbEI7UUFFRCxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsS0FBSyxFQUFHLENBQUM7UUFDL0IsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxFQUFFO1lBQ2hDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsQ0FBQztTQUMvQjthQUFNO1lBQ0wsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLE1BQU0sRUFBRSxDQUFDO1NBQ3ZDO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNJLG1CQUFtQixDQUFDLElBQVk7UUFDckMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVTLGlCQUFpQixDQUFDLFFBQW1CO1FBQzdDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUM7UUFFckIsTUFBTSxRQUFRLEdBQ1osUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUU7WUFDcEIsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO1NBQzFCLENBQUMsSUFBSSxTQUFTLENBQUM7UUFFbEIsSUFBSSxRQUFRLEVBQUU7WUFDWixnQkFBUyxDQUFDLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNoRDtRQUVELE9BQU8sUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUN2RSxDQUFDOztBQTNLSCxnQ0E0S0M7OztBQUVEOztHQUVHO0FBQ0gsU0FBUyxjQUFjLENBQUMsQ0FBUztJQUMvQiwrRUFBK0U7SUFDL0Usb0RBQW9EO0lBQ3BELE1BQU0sR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDakIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7UUFDakMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRTtZQUNyQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNuQixDQUFDLEVBQUUsQ0FBQztTQUNMO2FBQU0sSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssR0FBRyxFQUFFO1lBQ3ZCLEdBQUcsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDakI7YUFBTTtZQUNMLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDaEI7S0FDRjtJQUVELEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNkLE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEZpbGVCYXNlLCBGaWxlQmFzZU9wdGlvbnMsIElSZXNvbHZlciB9IGZyb20gXCIuL2ZpbGVcIjtcbmltcG9ydCB7IFByb2plY3QgfSBmcm9tIFwiLi9wcm9qZWN0XCI7XG5pbXBvcnQgeyBkZWVwTWVyZ2UgfSBmcm9tIFwiLi91dGlsXCI7XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYE9iamVjdEZpbGVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9iamVjdEZpbGVPcHRpb25zIGV4dGVuZHMgRmlsZUJhc2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBvYmplY3QgdGhhdCB3aWxsIGJlIHNlcmlhbGl6ZWQuIFlvdSBjYW4gbW9kaWZ5IHRoZSBvYmplY3QncyBjb250ZW50c1xuICAgKiBiZWZvcmUgc3ludGhlc2lzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB7fSBhbiBlbXB0eSBvYmplY3QgKHVzZSBgZmlsZS5vYmpgIHRvIG11dGF0ZSkuXG4gICAqL1xuICByZWFkb25seSBvYmo/OiBhbnk7XG5cbiAgLyoqXG4gICAqIE9taXRzIGVtcHR5IG9iamVjdHMgYW5kIGFycmF5cy5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9taXRFbXB0eT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBPYmplY3QgZmlsZS5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIE9iamVjdEZpbGUgZXh0ZW5kcyBGaWxlQmFzZSB7XG4gIC8qKlxuICAgKiBUaGUgb3V0cHV0IG9iamVjdC4gVGhpcyBvYmplY3QgY2FuIGJlIG11dGF0ZWQgdW50aWwgdGhlIHByb2plY3QgaXNcbiAgICogc3ludGhlc2l6ZWQuXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IG9iajogb2JqZWN0O1xuXG4gIC8qKlxuICAgKiBBbiBvYmplY3QgdG8gYmUgbWVyZ2VkIG9uIHRvcCBvZiBgb2JqYCBhZnRlciB0aGUgcmVzb2x2ZXIgaXMgY2FsbGVkXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHJhd092ZXJyaWRlczogb2JqZWN0O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgZW1wdHkgb2JqZWN0cyBhbmQgYXJyYXlzIGFyZSBvbWl0dGVkIGZyb20gdGhlIG91dHB1dCBvYmplY3QuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgb21pdEVtcHR5OiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFByb2plY3QsIGZpbGVQYXRoOiBzdHJpbmcsIG9wdGlvbnM6IE9iamVjdEZpbGVPcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCwgZmlsZVBhdGgsIG9wdGlvbnMpO1xuXG4gICAgdGhpcy5vYmogPSBvcHRpb25zLm9iaiA/PyB7fTtcbiAgICB0aGlzLm9taXRFbXB0eSA9IG9wdGlvbnMub21pdEVtcHR5ID8/IGZhbHNlO1xuICAgIHRoaXMucmF3T3ZlcnJpZGVzID0ge307XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBvdmVycmlkZSB0byB0aGUgc3ludGhlc2l6ZWQgb2JqZWN0IGZpbGUuXG4gICAqXG4gICAqIElmIHRoZSBvdmVycmlkZSBpcyBuZXN0ZWQsIHNlcGFyYXRlIGVhY2ggbmVzdGVkIGxldmVsIHVzaW5nIGEgZG90ICguKSBpbiB0aGUgcGF0aCBwYXJhbWV0ZXIuXG4gICAqIElmIHRoZXJlIGlzIGFuIGFycmF5IGFzIHBhcnQgb2YgdGhlIG5lc3RpbmcsIHNwZWNpZnkgdGhlIGluZGV4IGluIHRoZSBwYXRoLlxuICAgKlxuICAgKiBUbyBpbmNsdWRlIGEgbGl0ZXJhbCBgLmAgaW4gdGhlIHByb3BlcnR5IG5hbWUsIHByZWZpeCB3aXRoIGEgYFxcYC4gSW4gbW9zdFxuICAgKiBwcm9ncmFtbWluZyBsYW5ndWFnZXMgeW91IHdpbGwgbmVlZCB0byB3cml0ZSB0aGlzIGFzIGBcIlxcXFwuXCJgIGJlY2F1c2UgdGhlXG4gICAqIGBcXGAgaXRzZWxmIHdpbGwgbmVlZCB0byBiZSBlc2NhcGVkLlxuICAgKlxuICAgKiBGb3IgZXhhbXBsZSxcbiAgICogYGBgdHlwZXNjcmlwdFxuICAgKiBwcm9qZWN0LnRzY29uZmlnLmZpbGUuYWRkT3ZlcnJpZGUoJ2NvbXBpbGVyT3B0aW9ucy5hbHdheXNTdHJpY3QnLCB0cnVlKTtcbiAgICogcHJvamVjdC50c2NvbmZpZy5maWxlLmFkZE92ZXJyaWRlKCdjb21waWxlck9wdGlvbnMubGliJywgWydkb20nLCAnZG9tLml0ZXJhYmxlJywgJ2VzbmV4dCddKTtcbiAgICogYGBgXG4gICAqIHdvdWxkIGFkZCB0aGUgb3ZlcnJpZGVzXG4gICAqIGBgYGpzb25cbiAgICogXCJjb21waWxlck9wdGlvbnNcIjoge1xuICAgKiAgIFwiYWx3YXlzU3RyaWN0XCI6IHRydWUsXG4gICAqICAgXCJsaWJcIjogW1xuICAgKiAgICAgXCJkb21cIixcbiAgICogICAgIFwiZG9tLml0ZXJhYmxlXCIsXG4gICAqICAgICBcImVzbmV4dFwiXG4gICAqICAgXVxuICAgKiAgIC4uLlxuICAgKiB9XG4gICAqIC4uLlxuICAgKiBgYGBcbiAgICpcbiAgICogQHBhcmFtIHBhdGggLSBUaGUgcGF0aCBvZiB0aGUgcHJvcGVydHksIHlvdSBjYW4gdXNlIGRvdCBub3RhdGlvbiB0b1xuICAgKiAgICAgICAgb3ZlcnJpZGUgdmFsdWVzIGluIGNvbXBsZXggdHlwZXMuIEFueSBpbnRlcm1lZGlhdGUga2V5c1xuICAgKiAgICAgICAgd2lsbCBiZSBjcmVhdGVkIGFzIG5lZWRlZC5cbiAgICogQHBhcmFtIHZhbHVlIC0gVGhlIHZhbHVlLiBDb3VsZCBiZSBwcmltaXRpdmUgb3IgY29tcGxleC5cbiAgICovXG4gIHB1YmxpYyBhZGRPdmVycmlkZShwYXRoOiBzdHJpbmcsIHZhbHVlOiBhbnkpIHtcbiAgICBjb25zdCBwYXJ0cyA9IHNwbGl0T25QZXJpb2RzKHBhdGgpO1xuICAgIGxldCBjdXJyOiBhbnkgPSB0aGlzLnJhd092ZXJyaWRlcztcblxuICAgIHdoaWxlIChwYXJ0cy5sZW5ndGggPiAxKSB7XG4gICAgICBjb25zdCBrZXkgPSBwYXJ0cy5zaGlmdCgpITtcblxuICAgICAgLy8gaWYgd2UgY2FuJ3QgcmVjdXJzZSBmdXJ0aGVyIG9yIHRoZSBwcmV2aW91cyB2YWx1ZSBpcyBub3QgYW5cbiAgICAgIC8vIG9iamVjdCBvdmVyd3JpdGUgaXQgd2l0aCBhbiBvYmplY3QuXG4gICAgICBjb25zdCBpc09iamVjdCA9XG4gICAgICAgIGN1cnJba2V5XSAhPSBudWxsICYmXG4gICAgICAgIHR5cGVvZiBjdXJyW2tleV0gPT09IFwib2JqZWN0XCIgJiZcbiAgICAgICAgIUFycmF5LmlzQXJyYXkoY3VycltrZXldKTtcbiAgICAgIGlmICghaXNPYmplY3QpIHtcbiAgICAgICAgY3VycltrZXldID0ge307XG4gICAgICB9XG5cbiAgICAgIGN1cnIgPSBjdXJyW2tleV07XG4gICAgfVxuXG4gICAgY29uc3QgbGFzdEtleSA9IHBhcnRzLnNoaWZ0KCkhO1xuICAgIGN1cnJbbGFzdEtleV0gPSB2YWx1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIHRvIGFuIGFycmF5IGluIHRoZSBzeW50aGVzaXplZCBvYmplY3QgZmlsZS5cbiAgICpcbiAgICogSWYgdGhlIGFycmF5IGlzIG5lc3RlZCwgc2VwYXJhdGUgZWFjaCBuZXN0ZWQgbGV2ZWwgdXNpbmcgYSBkb3QgKC4pIGluIHRoZSBwYXRoIHBhcmFtZXRlci5cbiAgICogSWYgdGhlcmUgaXMgYW4gYXJyYXkgYXMgcGFydCBvZiB0aGUgbmVzdGluZywgc3BlY2lmeSB0aGUgaW5kZXggaW4gdGhlIHBhdGguXG4gICAqXG4gICAqIFRvIGluY2x1ZGUgYSBsaXRlcmFsIGAuYCBpbiB0aGUgcHJvcGVydHkgbmFtZSwgcHJlZml4IHdpdGggYSBgXFxgLiBJbiBtb3N0XG4gICAqIHByb2dyYW1taW5nIGxhbmd1YWdlcyB5b3Ugd2lsbCBuZWVkIHRvIHdyaXRlIHRoaXMgYXMgYFwiXFxcXC5cImAgYmVjYXVzZSB0aGVcbiAgICogYFxcYCBpdHNlbGYgd2lsbCBuZWVkIHRvIGJlIGVzY2FwZWQuXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCB3aXRoIHRoZSBmb2xsb3dpbmcgb2JqZWN0IGZpbGVcbiAgICogYGBganNvblxuICAgKiBcImNvbXBpbGVyT3B0aW9uc1wiOiB7XG4gICAqICAgXCJleGNsdWRlXCI6IFtcIm5vZGVfbW9kdWxlc1wiXSxcbiAgICogICBcImxpYlwiOiBbXCJlczIwMTlcIl1cbiAgICogICAuLi5cbiAgICogfVxuICAgKiAuLi5cbiAgICogYGBgXG4gICAqXG4gICAqIGBgYHR5cGVzY3JpcHRcbiAgICogcHJvamVjdC50c2NvbmZpZy5maWxlLmFkZFRvQXJyYXkoJ2NvbXBpbGVyT3B0aW9ucy5leGNsdWRlJywgJ2NvdmVyYWdlJyk7XG4gICAqIHByb2plY3QudHNjb25maWcuZmlsZS5hZGRUb0FycmF5KCdjb21waWxlck9wdGlvbnMubGliJywgJ2RvbScsICdkb20uaXRlcmFibGUnLCAnZXNuZXh0Jyk7XG4gICAqIGBgYFxuICAgKiB3b3VsZCByZXN1bHQgaW4gdGhlIGZvbGxvd2luZyBvYmplY3QgZmlsZVxuICAgKiBgYGBqc29uXG4gICAqIFwiY29tcGlsZXJPcHRpb25zXCI6IHtcbiAgICogICBcImV4Y2x1ZGVcIjogW1wibm9kZV9tb2R1bGVzXCIsIFwiY292ZXJhZ2VcIl0sXG4gICAqICAgXCJsaWJcIjogW1wiZXMyMDE5XCIsIFwiZG9tXCIsIFwiZG9tLml0ZXJhYmxlXCIsIFwiZXNuZXh0XCJdXG4gICAqICAgLi4uXG4gICAqIH1cbiAgICogLi4uXG4gICAqIGBgYFxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCAtIFRoZSBwYXRoIG9mIHRoZSBwcm9wZXJ0eSwgeW91IGNhbiB1c2UgZG90IG5vdGF0aW9uIHRvXG4gICAqICAgICAgICBhdHQgdG8gYXJyYXlzIGluIGNvbXBsZXggdHlwZXMuIEFueSBpbnRlcm1lZGlhdGUga2V5c1xuICAgKiAgICAgICAgd2lsbCBiZSBjcmVhdGVkIGFzIG5lZWRlZC5cbiAgICogQHBhcmFtIHZhbHVlcyAtIFRoZSB2YWx1ZXMgdG8gYWRkLiBDb3VsZCBiZSBwcmltaXRpdmUgb3IgY29tcGxleC5cbiAgICovXG4gIHB1YmxpYyBhZGRUb0FycmF5KHBhdGg6IHN0cmluZywgLi4udmFsdWVzOiBhbnkpIHtcbiAgICBjb25zdCBwYXJ0cyA9IHNwbGl0T25QZXJpb2RzKHBhdGgpO1xuICAgIGxldCBjdXJyOiBhbnkgPSB0aGlzLnJhd092ZXJyaWRlcztcblxuICAgIHdoaWxlIChwYXJ0cy5sZW5ndGggPiAxKSB7XG4gICAgICBjb25zdCBrZXkgPSBwYXJ0cy5zaGlmdCgpITtcblxuICAgICAgLy8gaWYgd2UgY2FuJ3QgcmVjdXJzZSBmdXJ0aGVyIG9yIHRoZSBwcmV2aW91cyB2YWx1ZSBpcyBub3QgYW5cbiAgICAgIC8vIG9iamVjdCBvdmVyd3JpdGUgaXQgd2l0aCBhbiBvYmplY3QuXG4gICAgICBjb25zdCBpc09iamVjdCA9XG4gICAgICAgIGN1cnJba2V5XSAhPSBudWxsICYmXG4gICAgICAgIHR5cGVvZiBjdXJyW2tleV0gPT09IFwib2JqZWN0XCIgJiZcbiAgICAgICAgIUFycmF5LmlzQXJyYXkoY3VycltrZXldKTtcbiAgICAgIGlmICghaXNPYmplY3QpIHtcbiAgICAgICAgY3VycltrZXldID0ge307XG4gICAgICB9XG5cbiAgICAgIGN1cnIgPSBjdXJyW2tleV07XG4gICAgfVxuXG4gICAgY29uc3QgbGFzdEtleSA9IHBhcnRzLnNoaWZ0KCkhO1xuICAgIGlmIChBcnJheS5pc0FycmF5KGN1cnJbbGFzdEtleV0pKSB7XG4gICAgICBjdXJyW2xhc3RLZXldLnB1c2goLi4udmFsdWVzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY3VycltsYXN0S2V5XSA9IHsgX18kQVBQRU5EOiB2YWx1ZXMgfTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogU3ludGFjdGljIHN1Z2FyIGZvciBgYWRkT3ZlcnJpZGUocGF0aCwgdW5kZWZpbmVkKWAuXG4gICAqIEBwYXJhbSBwYXRoIFRoZSBwYXRoIG9mIHRoZSB2YWx1ZSB0byBkZWxldGVcbiAgICovXG4gIHB1YmxpYyBhZGREZWxldGlvbk92ZXJyaWRlKHBhdGg6IHN0cmluZykge1xuICAgIHRoaXMuYWRkT3ZlcnJpZGUocGF0aCwgdW5kZWZpbmVkKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBzeW50aGVzaXplQ29udGVudChyZXNvbHZlcjogSVJlc29sdmVyKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBvYmogPSB0aGlzLm9iajtcblxuICAgIGNvbnN0IHJlc29sdmVkID1cbiAgICAgIHJlc29sdmVyLnJlc29sdmUob2JqLCB7XG4gICAgICAgIG9taXRFbXB0eTogdGhpcy5vbWl0RW1wdHksXG4gICAgICB9KSA/PyB1bmRlZmluZWQ7XG5cbiAgICBpZiAocmVzb2x2ZWQpIHtcbiAgICAgIGRlZXBNZXJnZShbcmVzb2x2ZWQsIHRoaXMucmF3T3ZlcnJpZGVzXSwgdHJ1ZSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc29sdmVkID8gSlNPTi5zdHJpbmdpZnkocmVzb2x2ZWQsIHVuZGVmaW5lZCwgMikgOiB1bmRlZmluZWQ7XG4gIH1cbn1cblxuLyoqXG4gKiBTcGxpdCBvbiBwZXJpb2RzIHdoaWxlIHByb2Nlc3NpbmcgZXNjYXBlIGNoYXJhY3RlcnMgXFxcbiAqL1xuZnVuY3Rpb24gc3BsaXRPblBlcmlvZHMoeDogc3RyaW5nKTogc3RyaW5nW10ge1xuICAvLyBCdWlsZCB0aGlzIGxpc3QgaW4gcmV2ZXJzZSBiZWNhdXNlIGl0J3MgbW9yZSBjb252ZW5pZW50IHRvIGdldCB0aGUgXCJjdXJyZW50XCJcbiAgLy8gaXRlbSBieSBkb2luZyByZXRbMF0gdGhhbiBieSByZXRbcmV0Lmxlbmd0aCAtIDFdLlxuICBjb25zdCByZXQgPSBbXCJcIl07XG4gIGZvciAobGV0IGkgPSAwOyBpIDwgeC5sZW5ndGg7IGkrKykge1xuICAgIGlmICh4W2ldID09PSBcIlxcXFxcIiAmJiBpICsgMSA8IHgubGVuZ3RoKSB7XG4gICAgICByZXRbMF0gKz0geFtpICsgMV07XG4gICAgICBpKys7XG4gICAgfSBlbHNlIGlmICh4W2ldID09PSBcIi5cIikge1xuICAgICAgcmV0LnVuc2hpZnQoXCJcIik7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldFswXSArPSB4W2ldO1xuICAgIH1cbiAgfVxuXG4gIHJldC5yZXZlcnNlKCk7XG4gIHJldHVybiByZXQ7XG59XG4iXX0=