"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const build_1 = require("../build");
const common_1 = require("../common");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/util");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const release_1 = require("../release");
const util_2 = require("../util");
const version_1 = require("../version");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease = exports.AutoRelease || (exports.AutoRelease = {}));
/**
 * Node.js project
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.artifactsJavascriptDirectory = path_1.join(this.artifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", '$(npx -c "node -e \\"console.log(process.env.PATH)\\"")');
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore");
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_RC}`);
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        this.gitignore.include(`/${common_1.PROJEN_RC}`);
        const projen = options.projenDevDependency ?? true;
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        if (buildEnabled && this.github) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                preBuildSteps: this.renderWorkflowSetup({
                    mutable: options.mutableBuild ?? true,
                }),
                postBuildSteps: options.postBuildSteps,
                runsOn: options.workflowRunsOn,
                workflowTriggers: options.buildWorkflowTriggers,
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.addDevDeps(version_1.Version.STANDARD_VERSION);
            this.release = new release_1.Release(this, {
                versionFile: "package.json",
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                artifactsDirectory: this.artifactsDirectory,
                ...options,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({ mutable: false }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
            });
            this.publisher = this.release.publisher;
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = release_1.isAwsCodeArtifactRegistry(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot(util_2.deepMerge([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, util_2.deepMerge([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (projenrcJs) {
            new projenrc_1.Projenrc(this, options.projenrcJsOptions);
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        if (options.package ?? true) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            // always use npm here - yarn doesn't add much value
            // sadly we cannot use --pack-destination because it is not supported by older npm
            this.packageTask.exec(`mv $(npm pack) ${this.artifactsJavascriptDirectory}/`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * Minimum node.js version required by this package.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version required by this pacakge.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v1",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
        };
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v1",
                    with: {
                        "aws-access-key-id": util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: util_1.secretToString(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: util_1.secretToString(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.nodeVersion) {
            install.push({
                name: "Setup Node.js",
                uses: "actions/setup-node@v2.2.0",
                with: { "node-version": this.nodeVersion },
            });
        }
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.0.1",
                with: { version: "6.14.7" },
            });
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.60.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibm9kZS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2phdmFzY3JpcHQvbm9kZS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsK0JBQTRCO0FBQzVCLG9DQUF5QztBQUN6QyxzQ0FBa0Q7QUFDbEQsc0NBTW1CO0FBQ25CLG1EQUFrRTtBQUNsRSx5Q0FBZ0Q7QUFFaEQsZ0RBQTRDO0FBQzVDLDhDQUt1QjtBQUN2Qix3Q0FBcUM7QUFDckMsd0NBS29CO0FBRXBCLGtDQUFvQztBQUNwQyx3Q0FBcUM7QUFDckMsdUNBQW9EO0FBQ3BELGlDQUEyQztBQUMzQyxpREFLd0I7QUFDeEIseUNBQXVEO0FBRXZELE1BQU0sYUFBYSxHQUFHLFFBQVEsQ0FBQztBQThQL0I7O0dBRUc7QUFDSCxJQUFZLFdBVVg7QUFWRCxXQUFZLFdBQVc7SUFDckI7O09BRUc7SUFDSCw2REFBWSxDQUFBO0lBRVo7O09BRUc7SUFDSCwrQ0FBSyxDQUFBO0FBQ1AsQ0FBQyxFQVZXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBVXRCO0FBRUQ7Ozs7R0FJRztBQUNILE1BQWEsV0FBWSxTQUFRLHNCQUFhO0lBaUg1QyxZQUFZLE9BQTJCO1FBQ3JDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVmLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSwwQkFBVyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsT0FBTyxDQUFDLHNCQUFzQixJQUFJLEVBQUUsQ0FBQztRQUNuRSxJQUFJLENBQUMsbUJBQW1CO1lBQ3RCLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSx1Q0FBMkIsQ0FBQztRQUM3RCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsT0FBTyxDQUFDLGtCQUFrQixJQUFJLE1BQU0sQ0FBQztRQUMvRCxJQUFJLENBQUMsNEJBQTRCLEdBQUcsV0FBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUV4RSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsQ0FBQyxHQUFHLEVBQUU7WUFDNUIsUUFBUSxJQUFJLENBQUMsY0FBYyxFQUFFO2dCQUMzQixLQUFLLGlDQUFrQixDQUFDLEdBQUc7b0JBQ3pCLE9BQU8sU0FBUyxDQUFDO2dCQUNuQixLQUFLLGlDQUFrQixDQUFDLElBQUk7b0JBQzFCLE9BQU8sVUFBVSxDQUFDO2dCQUNwQixLQUFLLGlDQUFrQixDQUFDLElBQUk7b0JBQzFCLE9BQU8sVUFBVSxDQUFDO2dCQUNwQjtvQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixJQUFJLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQzthQUN4RTtRQUNILENBQUMsQ0FBQyxFQUFFLENBQUM7UUFFTCxJQUFJLENBQUMsV0FBVztZQUNkLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztRQUU3RCxvRUFBb0U7UUFDcEUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQ3ZCLE1BQU0sRUFDTix5REFBeUQsQ0FDMUQsQ0FBQztRQUVGLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFekIsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLElBQUksSUFBSSxFQUFFO1lBQ3BDLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSx3QkFBVSxDQUFDLElBQUksRUFBRSxZQUFZLENBQUMsQ0FBQztTQUNyRDtRQUVELElBQUksQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBRTNCLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxNQUFNLEVBQUU7WUFDN0IsS0FBSyxNQUFNLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFO2dCQUNqQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUMzQjtTQUNGO1FBRUQsSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRTtZQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRTtnQkFDbkIsTUFBTSxJQUFJLEtBQUssQ0FDYiwyRkFBMkYsQ0FDNUYsQ0FBQzthQUNIO1lBRUQsS0FBSyxNQUFNLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxFQUFFO2dCQUNqQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUMzQjtTQUNGO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDakIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxhQUFhLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUMzRDtRQUVELElBQUksQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLElBQUksa0JBQVMsRUFBRSxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxtQkFBVSxHQUFHLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLGtCQUFTLEVBQUUsQ0FBQyxDQUFDO1FBRXhDLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLENBQUM7UUFDbkQsSUFBSSxNQUFNLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFO1lBQzNCLE1BQU0sT0FBTyxHQUFHLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDekUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDckM7UUFFRCxJQUFJLENBQUMsT0FBTyxDQUFDLG9CQUFvQixFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQ2Isb0hBQW9ILENBQ3JILENBQUM7U0FDSDtRQUVELE1BQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxhQUFhLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTNFLDRCQUE0QjtRQUM1Qiw2Q0FBNkM7UUFDN0MsSUFBSSxPQUFPLENBQUMsSUFBSSxJQUFJLElBQUksRUFBRTtZQUN4QixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksV0FBSSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDakQ7UUFFRCxJQUFJLFlBQVksSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQy9CLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxxQkFBYSxDQUFDLElBQUksRUFBRTtnQkFDM0MsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN6QixrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCO2dCQUMzQyxjQUFjLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjtnQkFDOUMsV0FBVyxFQUFFLElBQUksQ0FBQyxtQkFBbUI7Z0JBQ3JDLFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDbEMsYUFBYSxFQUFFLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztvQkFDdEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxZQUFZLElBQUksSUFBSTtpQkFDdEMsQ0FBQztnQkFDRixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7Z0JBQ3RDLE1BQU0sRUFBRSxPQUFPLENBQUMsY0FBYztnQkFDOUIsZ0JBQWdCLEVBQUUsT0FBTyxDQUFDLHFCQUFxQjthQUNoRCxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUNsQyxHQUFHLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxPQUFPLENBQUMsQ0FDN0MsQ0FBQztTQUNIO1FBRUQsTUFBTSxPQUFPLEdBQ1gsT0FBTyxDQUFDLE9BQU87WUFDZixPQUFPLENBQUMsZUFBZTtZQUN2QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDL0IsSUFBSSxPQUFPLEVBQUU7WUFDWCxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUUxQyxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksaUJBQU8sQ0FBQyxJQUFJLEVBQUU7Z0JBQy9CLFdBQVcsRUFBRSxjQUFjO2dCQUMzQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3BCLE1BQU0sRUFBRSxPQUFPLENBQUMsb0JBQW9CLElBQUksTUFBTTtnQkFDOUMsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtnQkFDM0MsR0FBRyxPQUFPO2dCQUVWLHlCQUF5QixFQUFFO29CQUN6QixHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQztvQkFDL0MsR0FBRyxDQUFDLE9BQU8sQ0FBQyx5QkFBeUIsSUFBSSxFQUFFLENBQUM7aUJBQzdDO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7b0JBQ2pDLEdBQUcsSUFBSSxDQUFDLDJCQUEyQixDQUFDLE9BQU8sQ0FBQztpQkFDN0M7YUFDRixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO1lBRXhDLElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxLQUFLLEVBQUU7Z0JBQ2pDLE1BQU0sbUJBQW1CLEdBQUcsbUNBQXlCLENBQ25ELElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUN6QjtvQkFDQyxDQUFDLENBQUM7d0JBQ0UsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLG1CQUFtQixFQUFFLGlCQUFpQjt3QkFDakUscUJBQXFCLEVBQ25CLE9BQU8sQ0FBQyxtQkFBbUIsRUFBRSxxQkFBcUI7d0JBQ3BELFlBQVksRUFBRSxPQUFPLENBQUMsbUJBQW1CLEVBQUUsWUFBWTtxQkFDeEQ7b0JBQ0gsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDUCxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUM7b0JBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVc7b0JBQ2xDLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWM7b0JBQzNDLG1CQUFtQjtpQkFDcEIsQ0FBQyxDQUFDO2FBQ0o7U0FDRjthQUFNO1lBQ0wscUZBQXFGO1lBQ3JGLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRTtnQkFDeEIsTUFBTSxJQUFJLEtBQUssQ0FDYix5REFBeUQsQ0FDMUQsQ0FBQzthQUNIO1lBRUQsSUFBSSxPQUFPLENBQUMsa0JBQWtCLEVBQUU7Z0JBQzlCLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStELENBQ2hFLENBQUM7YUFDSDtZQUVELElBQUksT0FBTyxDQUFDLGVBQWUsRUFBRTtnQkFDM0IsTUFBTSxJQUFJLEtBQUssQ0FDYiw0REFBNEQsQ0FDN0QsQ0FBQzthQUNIO1NBQ0Y7UUFFRCxJQUNFLENBQUMsT0FBTyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUM7WUFDM0IsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPO1lBQ3BCLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxFQUMvQjtZQUNBLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxrQkFBUyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7WUFDdEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDaEMsTUFBTSxFQUFFLEdBQUcsRUFBRSxDQUNYLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxDQUFDO29CQUNuRSxFQUFFO2FBQ0wsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUMvQyxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBRXZELElBQUksVUFBVSxJQUFJLFdBQVcsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUNiLCtEQUErRCxDQUNoRSxDQUFDO1NBQ0g7UUFFRCxNQUFNLGVBQWUsR0FBRyxPQUFPLENBQUMsbUJBQW1CLElBQUksS0FBSyxDQUFDO1FBRTdELElBQUksZUFBZSxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQ2IsdUZBQXVGLENBQ3hGLENBQUM7U0FDSDtRQUVELE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxTQUFrQixFQUFFLEVBQUUsQ0FDOUMsU0FBUyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSztZQUNsQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQztZQUMxQixDQUFDLENBQUMsU0FBUyxDQUFDO1FBRWhCLElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxjQUFjLEdBQUc7Z0JBQ3JCLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7YUFDMUMsQ0FBQztZQUNGLElBQUksQ0FBQyxNQUFNLEVBQUUsYUFBYSxDQUN4QixnQkFBUyxDQUFDLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUM3RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLFdBQVcsRUFBRTtZQUNmLE1BQU0sY0FBYyxHQUErQjtnQkFDakQsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRSxPQUFPLENBQUMsc0JBQXNCO3dCQUN2QyxDQUFDLENBQUM7NEJBQ0UsS0FBSyxFQUFFLE9BQU8sQ0FBQyxzQkFBc0I7eUJBQ3RDO3dCQUNILENBQUMsQ0FBQyxTQUFTO29CQUNiLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxlQUFlLENBQUM7b0JBQ3pDLFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2lCQUN0QzthQUNGLENBQUM7WUFDRixJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksZ0NBQW1CLENBQzVDLElBQUksRUFDSixnQkFBUyxDQUFDLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUM5RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLEVBQUU7WUFDdkMsSUFBSSxDQUFDLE1BQU0sRUFBRSxzQkFBc0IsQ0FDakMsR0FBRyxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsSUFBSSxFQUFFLENBQUMsQ0FDL0MsQ0FBQztTQUNIO1FBRUQsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUM7UUFDL0QsSUFBSSxVQUFVLEVBQUU7WUFDZCxJQUFJLG1CQUFRLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1NBQy9DO1FBRUQscUdBQXFHO1FBQ3JHLElBQUksQ0FBQyxPQUFPLEdBQUcsSUFBSSxpQkFBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUM7UUFFekQsSUFBSSxPQUFPLENBQUMsT0FBTyxJQUFJLElBQUksRUFBRTtZQUMzQixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxZQUFZLElBQUksQ0FBQyw0QkFBNEIsRUFBRSxDQUFDLENBQUM7WUFFdkUsb0RBQW9EO1lBQ3BELGtGQUFrRjtZQUNsRixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FDbkIsa0JBQWtCLElBQUksQ0FBQyw0QkFBNEIsR0FBRyxDQUN2RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLE9BQU8sQ0FBQyxRQUFRLElBQUksS0FBSyxFQUFFO1lBQzdCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxxQkFBUSxDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsT0FBTyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUM7U0FDcEU7SUFDSCxDQUFDO0lBeldEOztPQUVHO0lBQ0gsSUFBVyx3QkFBd0I7UUFDakMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLHdCQUF3QixDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsVUFBVTtRQUNuQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQ2pDLENBQUM7SUF5QkQ7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLGNBQWM7UUFDdkIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQztJQUNyQyxDQUFDO0lBSUQ7Ozs7T0FJRztJQUNILElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO0lBQ3JDLENBQUM7SUFZRDs7T0FFRztJQUNILElBQVcsUUFBUTtRQUNqQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO0lBQy9CLENBQUM7SUE4Uk8sMkJBQTJCLENBQUMsT0FBMkI7UUFDN0QsNkRBQTZEO1FBQzdELDhCQUE4QjtRQUM5QixJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxPQUFPLENBQUMsa0JBQWtCLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUN4RSxPQUFPO2dCQUNMO29CQUNFLElBQUksRUFBRSw0QkFBNEI7b0JBQ2xDLElBQUksRUFBRSwyQkFBMkI7b0JBQ2pDLElBQUksRUFBRSxPQUFPLENBQUMsa0JBQWtCO3dCQUM5QixDQUFDLENBQUM7NEJBQ0UsS0FBSyxFQUFFLGdCQUFnQixPQUFPLENBQUMsa0JBQWtCLEtBQUs7NEJBQ3RELFNBQVMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUI7eUJBQzlDO3dCQUNILENBQUMsQ0FBQzs0QkFDRSxTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCO3lCQUM5QztpQkFDTjthQUNGLENBQUM7U0FDSDthQUFNO1lBQ0wsT0FBTyxFQUFFLENBQUM7U0FDWDtJQUNILENBQUM7SUFFTSxPQUFPLENBQUMsSUFBNEI7UUFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksU0FBUyxDQUFDLElBQVksRUFBRSxPQUFlO1FBQzVDLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksWUFBWSxDQUFDLElBQVk7UUFDOUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFNBQVMsQ0FBQyxJQUFZO1FBQzNCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGlCQUFpQixDQUFDLEdBQUcsUUFBa0I7UUFDNUMsS0FBSyxNQUFNLENBQUMsSUFBSSxRQUFRLEVBQUU7WUFDeEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDMUI7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksY0FBYyxDQUFDLEdBQUcsUUFBa0I7UUFDekMsS0FBSyxNQUFNLENBQUMsSUFBSSxRQUFRLEVBQUU7WUFDeEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDdkI7SUFDSCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksU0FBUyxDQUFDLE1BQStCO1FBQzlDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ2xELElBQUksQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztTQUNwQztJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxXQUFXLENBQUMsR0FBRyxRQUFrQjtRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLHFCQUFxQixDQUMzQixtQkFBb0Q7UUFFcEQsTUFBTSx5QkFBeUIsR0FBRztZQUNoQyxpQkFBaUIsRUFDZixtQkFBbUIsRUFBRSxpQkFBaUIsSUFBSSxtQkFBbUI7WUFDL0QscUJBQXFCLEVBQ25CLG1CQUFtQixFQUFFLHFCQUFxQixJQUFJLHVCQUF1QjtZQUN2RSxZQUFZLEVBQUUsbUJBQW1CLEVBQUUsWUFBWTtTQUNoRCxDQUFDO1FBRUYsSUFBSSx5QkFBeUIsQ0FBQyxZQUFZLEVBQUU7WUFDMUMsT0FBTztnQkFDTDtvQkFDRSxJQUFJLEVBQUUsMkJBQTJCO29CQUNqQyxJQUFJLEVBQUUsMENBQTBDO29CQUNoRCxJQUFJLEVBQUU7d0JBQ0osbUJBQW1CLEVBQUUscUJBQWMsQ0FDakMseUJBQXlCLENBQUMsaUJBQWlCLENBQzVDO3dCQUNELHVCQUF1QixFQUFFLHFCQUFjLENBQ3JDLHlCQUF5QixDQUFDLHFCQUFxQixDQUNoRDt3QkFDRCxZQUFZLEVBQUUsV0FBVzt3QkFDekIsZ0JBQWdCLEVBQUUseUJBQXlCLENBQUMsWUFBWTt3QkFDeEQsdUJBQXVCLEVBQUUsR0FBRztxQkFDN0I7aUJBQ0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLHdCQUF3QjtvQkFDOUIsR0FBRyxFQUFFLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixXQUFXO2lCQUN6QzthQUNGLENBQUM7U0FDSDtRQUVELE9BQU87WUFDTDtnQkFDRSxJQUFJLEVBQUUsd0JBQXdCO2dCQUM5QixHQUFHLEVBQUUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLFdBQVc7Z0JBQ3hDLEdBQUcsRUFBRTtvQkFDSCxpQkFBaUIsRUFBRSxxQkFBYyxDQUMvQix5QkFBeUIsQ0FBQyxpQkFBaUIsQ0FDNUM7b0JBQ0QscUJBQXFCLEVBQUUscUJBQWMsQ0FDbkMseUJBQXlCLENBQUMscUJBQXFCLENBQ2hEO2lCQUNGO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQixDQUN4QixVQUFzQyxFQUFFO1FBRXhDLE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxFQUFXLENBQUM7UUFFckMseUNBQXlDO1FBQ3pDLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUU3QyxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUU7WUFDcEIsT0FBTyxDQUFDLElBQUksQ0FBQztnQkFDWCxJQUFJLEVBQUUsZUFBZTtnQkFDckIsSUFBSSxFQUFFLDJCQUEyQjtnQkFDakMsSUFBSSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUU7YUFDM0MsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxLQUFLLGlDQUFrQixDQUFDLElBQUksRUFBRTtZQUMzRCxPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUNYLElBQUksRUFBRSxZQUFZO2dCQUNsQixJQUFJLEVBQUUsMEJBQTBCO2dCQUNoQyxJQUFJLEVBQUUsRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFO2FBQzVCLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUM7UUFFekMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFxQixFQUFFO1lBQ3RDLE9BQU8sQ0FBQyxJQUFJLENBQ1YsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxDQUNoRSxDQUFDO1NBQ0g7UUFFRCxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQ1gsSUFBSSxFQUFFLHNCQUFzQjtZQUM1QixHQUFHLEVBQUUsT0FBTztnQkFDVixDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQywrQkFBK0I7Z0JBQzlDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWM7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksT0FBTyxDQUFDLEdBQUcsSUFBYztRQUM5QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksVUFBVSxDQUFDLEdBQUcsSUFBYztRQUNqQyxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFdBQVcsQ0FBQyxHQUFHLElBQWM7UUFDbEMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGNBQWMsQ0FBQyxHQUFHLElBQWM7UUFDckMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFTSxnQkFBZ0IsQ0FBQyxPQUFlO1FBQ3JDLElBQUksQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFTyxVQUFVLENBQUMsT0FBMkI7UUFDNUMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRTtZQUN4QixJQUFJLGlCQUFPLENBQUMsSUFBSSxFQUFFO2dCQUNoQixJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPO2dCQUMxQixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWMsSUFBSSxPQUFPLENBQUMsVUFBVTtnQkFDNUQsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlO2FBQ3pDLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztJQUVPLG1CQUFtQjtRQUN6QixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FDcEIsUUFBUSxFQUNSLE1BQU0sRUFDTixPQUFPLEVBQ1AsZ0JBQWdCLEVBQ2hCLGlCQUFpQixFQUNqQixpQkFBaUIsRUFDakIsa0JBQWtCLEVBRWxCLDJEQUEyRCxFQUMzRCx5Q0FBeUMsRUFFekMsZ0JBQWdCLEVBQ2hCLE1BQU0sRUFDTixPQUFPLEVBQ1AsUUFBUSxFQUNSLFlBQVksRUFFWixtRUFBbUUsRUFDbkUsU0FBUyxFQUVULGtEQUFrRCxFQUNsRCxVQUFVLEVBQ1YsUUFBUSxFQUVSLHFCQUFxQixFQUNyQixhQUFhLEVBRWIsK0RBQStELEVBQy9ELGVBQWUsRUFFZiwwQkFBMEIsRUFDMUIsZUFBZSxFQUNmLGdCQUFnQixFQUVoQixvQkFBb0IsRUFDcEIsZUFBZSxFQUVmLHlCQUF5QixFQUN6QixjQUFjLEVBRWQsd0JBQXdCLEVBQ3hCLE9BQU8sRUFFUCx1QkFBdUIsRUFDdkIsaUJBQWlCLEVBRWpCLGdEQUFnRCxFQUNoRCxRQUFRLENBQ1QsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGNBQWMsQ0FBQyxJQUFVO1FBQzlCLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDdEQsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxrQkFBa0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1QyxDQUFDOztBQXhzQkgsa0NBeXNCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGpvaW4gfSBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgQnVpbGRXb3JrZmxvdyB9IGZyb20gXCIuLi9idWlsZFwiO1xuaW1wb3J0IHsgUFJPSkVOX0RJUiwgUFJPSkVOX1JDIH0gZnJvbSBcIi4uL2NvbW1vblwiO1xuaW1wb3J0IHtcbiAgQXV0b01lcmdlLFxuICBEZXBlbmRhYm90T3B0aW9ucyxcbiAgR2l0SHViUHJvamVjdCxcbiAgR2l0SHViUHJvamVjdE9wdGlvbnMsXG4gIEdpdElkZW50aXR5LFxufSBmcm9tIFwiLi4vZ2l0aHViXCI7XG5pbXBvcnQgeyBERUZBVUxUX0dJVEhVQl9BQ1RJT05TX1VTRVIgfSBmcm9tIFwiLi4vZ2l0aHViL2NvbnN0YW50c1wiO1xuaW1wb3J0IHsgc2VjcmV0VG9TdHJpbmcgfSBmcm9tIFwiLi4vZ2l0aHViL3V0aWxcIjtcbmltcG9ydCB7IEpvYlN0ZXAsIFRyaWdnZXJzIH0gZnJvbSBcIi4uL2dpdGh1Yi93b3JrZmxvd3MtbW9kZWxcIjtcbmltcG9ydCB7IElnbm9yZUZpbGUgfSBmcm9tIFwiLi4vaWdub3JlLWZpbGVcIjtcbmltcG9ydCB7XG4gIFByZXR0aWVyLFxuICBQcmV0dGllck9wdGlvbnMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXMsXG4gIFVwZ3JhZGVEZXBlbmRlbmNpZXNPcHRpb25zLFxufSBmcm9tIFwiLi4vamF2YXNjcmlwdFwiO1xuaW1wb3J0IHsgTGljZW5zZSB9IGZyb20gXCIuLi9saWNlbnNlXCI7XG5pbXBvcnQge1xuICBpc0F3c0NvZGVBcnRpZmFjdFJlZ2lzdHJ5LFxuICBQdWJsaXNoZXIsXG4gIFJlbGVhc2UsXG4gIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyxcbn0gZnJvbSBcIi4uL3JlbGVhc2VcIjtcbmltcG9ydCB7IFRhc2sgfSBmcm9tIFwiLi4vdGFza1wiO1xuaW1wb3J0IHsgZGVlcE1lcmdlIH0gZnJvbSBcIi4uL3V0aWxcIjtcbmltcG9ydCB7IFZlcnNpb24gfSBmcm9tIFwiLi4vdmVyc2lvblwiO1xuaW1wb3J0IHsgQnVuZGxlciwgQnVuZGxlck9wdGlvbnMgfSBmcm9tIFwiLi9idW5kbGVyXCI7XG5pbXBvcnQgeyBKZXN0LCBKZXN0T3B0aW9ucyB9IGZyb20gXCIuL2plc3RcIjtcbmltcG9ydCB7XG4gIENvZGVBcnRpZmFjdE9wdGlvbnMsXG4gIE5vZGVQYWNrYWdlLFxuICBOb2RlUGFja2FnZU1hbmFnZXIsXG4gIE5vZGVQYWNrYWdlT3B0aW9ucyxcbn0gZnJvbSBcIi4vbm9kZS1wYWNrYWdlXCI7XG5pbXBvcnQgeyBQcm9qZW5yYywgUHJvamVucmNPcHRpb25zIH0gZnJvbSBcIi4vcHJvamVucmNcIjtcblxuY29uc3QgUFJPSkVOX1NDUklQVCA9IFwicHJvamVuXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgTm9kZVByb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgR2l0SHViUHJvamVjdE9wdGlvbnMsXG4gICAgTm9kZVBhY2thZ2VPcHRpb25zLFxuICAgIFJlbGVhc2VQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBMaWNlbnNlIGNvcHlyaWdodCBvd25lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0cyB0byB0aGUgdmFsdWUgb2YgYXV0aG9yTmFtZSBvciBcIlwiIGlmIGBhdXRob3JOYW1lYCBpcyB1bmRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBjb3B5cmlnaHRPd25lcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvcHlyaWdodCB5ZWFycyB0byBwdXQgaW4gdGhlIExJQ0VOU0UgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBjdXJyZW50IHllYXJcbiAgICovXG4gIHJlYWRvbmx5IGNvcHlyaWdodFBlcmlvZD86IHN0cmluZztcblxuICAvKipcbiAgICogVmVyc2lvbiBvZiBwcm9qZW4gdG8gaW5zdGFsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0cyB0byB0aGUgbGF0ZXN0IHZlcnNpb24uXG4gICAqL1xuICByZWFkb25seSBwcm9qZW5WZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgb2YgXCJwcm9qZW5cIiBzaG91bGQgYmUgaW5zdGFsbGVkIGFzIGEgZGV2RGVwZW5kZW5jeS5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVuRGV2RGVwZW5kZW5jeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZSBhIEdpdEh1YiB3b3JrZmxvdyBmb3IgYnVpbGRpbmcgUFJzLlxuICAgKiBAZGVmYXVsdCAtIHRydWUgaWYgbm90IGEgc3VicHJvamVjdFxuICAgKi9cbiAgcmVhZG9ubHkgYnVpbGRXb3JrZmxvdz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgdXBkYXRlIGZpbGVzIG1vZGlmaWVkIGR1cmluZyBidWlsZHMgdG8gcHVsbC1yZXF1ZXN0IGJyYW5jaGVzLiBUaGlzIG1lYW5zXG4gICAqIHRoYXQgYW55IGZpbGVzIHN5bnRoZXNpemVkIGJ5IHByb2plbiBvciBlLmcuIHRlc3Qgc25hcHNob3RzIHdpbGwgYWx3YXlzIGJlIHVwLXRvLWRhdGVcbiAgICogYmVmb3JlIGEgUFIgaXMgbWVyZ2VkLlxuICAgKlxuICAgKiBJbXBsaWVzIHRoYXQgUFIgYnVpbGRzIGRvIG5vdCBoYXZlIGFudGktdGFtcGVyIGNoZWNrcy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbXV0YWJsZUJ1aWxkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVmaW5lIGEgR2l0SHViIHdvcmtmbG93IHN0ZXAgZm9yIHNlbmRpbmcgY29kZSBjb3ZlcmFnZSBtZXRyaWNzIHRvIGh0dHBzOi8vY29kZWNvdi5pby9cbiAgICogVXNlcyBjb2RlY292L2NvZGVjb3YtYWN0aW9uQHYxXG4gICAqIEEgc2VjcmV0IGlzIHJlcXVpcmVkIGZvciBwcml2YXRlIHJlcG9zLiBDb25maWd1cmVkIHdpdGggQGNvZGVDb3ZUb2tlblNlY3JldFxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY29kZUNvdj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIERlZmluZSB0aGUgc2VjcmV0IG5hbWUgZm9yIGEgc3BlY2lmaWVkIGh0dHBzOi8vY29kZWNvdi5pby8gdG9rZW5cbiAgICogQSBzZWNyZXQgaXMgcmVxdWlyZWQgdG8gc2VuZCBjb3ZlcmFnZSBmb3IgcHJpdmF0ZSByZXBvc2l0b3JpZXNcbiAgICogQGRlZmF1bHQgLSBpZiB0aGlzIG9wdGlvbiBpcyBub3Qgc3BlY2lmaWVkLCBvbmx5IHB1YmxpYyByZXBvc2l0b3JpZXMgYXJlIHN1cHBvcnRlZFxuICAgKi9cbiAgcmVhZG9ubHkgY29kZUNvdlRva2VuU2VjcmV0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBERVBSRUNBVEVEOiByZW5hbWVkIHRvIGByZWxlYXNlYC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIG5vdCBhIHN1YnByb2plY3RcbiAgICogQGRlcHJlY2F0ZWQgc2VlIGByZWxlYXNlYC5cbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VXb3JrZmxvdz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEFkZCByZWxlYXNlIG1hbmFnZW1lbnQgdG8gdGhpcyBwcm9qZWN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgKGZhbHNlIGZvciBzdWJwcm9qZWN0cylcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2U/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgbWFpbiByZWxlYXNlIGJyYW5jaC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJtYWluXCJcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRSZWxlYXNlQnJhbmNoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdvcmtmbG93IHN0ZXBzIHRvIHVzZSBpbiBvcmRlciB0byBib290c3RyYXAgdGhpcyByZXBvLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcInlhcm4gaW5zdGFsbCAtLWZyb3plbi1sb2NrZmlsZSAmJiB5YXJuIHByb2plblwiXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzPzogSm9iU3RlcFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgZ2l0IGlkZW50aXR5IHRvIHVzZSBpbiB3b3JrZmxvd3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gR2l0SHViIEFjdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IHdvcmtmbG93R2l0SWRlbnRpdHk/OiBHaXRJZGVudGl0eTtcblxuICAvKipcbiAgICogQXV0b21hdGljYWxseSByZWxlYXNlIHRvIG5wbSB3aGVuIG5ldyB2ZXJzaW9ucyBhcmUgaW50cm9kdWNlZC5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlbGVhc2VUb05wbT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBub2RlIHZlcnNpb24gdG8gdXNlIGluIEdpdEh1YiB3b3JrZmxvd3MuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gc2FtZSBhcyBgbWluTm9kZVZlcnNpb25gXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd05vZGVWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBVc2UgZGVwZW5kYWJvdCB0byBoYW5kbGUgZGVwZW5kZW5jeSB1cGdyYWRlcy5cbiAgICogQ2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBgZGVwc1VwZ3JhZGVgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwZW5kYWJvdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGRlcGVuZGFib3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBkZXBlbmRhYm90T3B0aW9ucz86IERlcGVuZGFib3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBVc2UgZ2l0aHViIHdvcmtmbG93cyB0byBoYW5kbGUgZGVwZW5kZW5jeSB1cGdyYWRlcy5cbiAgICogQ2Fubm90IGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBgZGVwZW5kYWJvdGAuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNVcGdyYWRlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT3B0aW9ucyBmb3IgYFVwZ3JhZGVEZXBlbmRlbmNpZXNgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgZGVwc1VwZ3JhZGVPcHRpb25zPzogVXBncmFkZURlcGVuZGVuY2llc09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgYXBwcm92ZSBkZXBzIHVwZ3JhZGUgUFJzLCBhbGxvd2luZyB0aGVtIHRvIGJlXG4gICAqIG1lcmdlZCBieSBtZXJnaWZ5IChpZiBjb25maWd1ZWQpLlxuICAgKlxuICAgKiBUaHJvdyBpZiBzZXQgdG8gdHJ1ZSBidXQgYGF1dG9BcHByb3ZlT3B0aW9uc2AgYXJlIG5vdCBkZWZpbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9BcHByb3ZlVXBncmFkZXM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIC5ucG1pZ25vcmUgZmlsZS4gTm9ybWFsbHkgdGhpcyBpcyBvbmx5IG5lZWRlZCBmb3IgbGlicmFyaWVzIHRoYXRcbiAgICogYXJlIHBhY2thZ2VkIGFzIHRhcmJhbGxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBucG1pZ25vcmVFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBlbnRyaWVzIHRvIC5ucG1pZ25vcmUuXG4gICAqIEBkZXByZWNhdGVkIC0gdXNlIGBwcm9qZWN0LmFkZFBhY2thZ2VJZ25vcmVgXG4gICAqL1xuICByZWFkb25seSBucG1pZ25vcmU/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogSW5jbHVkZSBhIEdpdEh1YiBwdWxsIHJlcXVlc3QgdGVtcGxhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHB1bGxSZXF1ZXN0VGVtcGxhdGU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgY29udGVudHMgb2YgdGhlIHB1bGwgcmVxdWVzdCB0ZW1wbGF0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IGNvbnRlbnRcbiAgICovXG4gIHJlYWRvbmx5IHB1bGxSZXF1ZXN0VGVtcGxhdGVDb250ZW50cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBTZXR1cCBwcmV0dGllci5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByZXR0aWVyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUHJldHRpZXIgb3B0aW9uc1xuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJldHRpZXJPcHRpb25zPzogUHJldHRpZXJPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIGVudHJpZXMgdG8gLmdpdGlnbm9yZVxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0aWdub3JlPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFNldHVwIGplc3QgdW5pdCB0ZXN0c1xuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBqZXN0PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSmVzdCBvcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBvcHRpb25zXG4gICAqL1xuICByZWFkb25seSBqZXN0T3B0aW9ucz86IEplc3RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSAob25jZSkgLnByb2plbnJjLmpzIChpbiBKYXZhU2NyaXB0KS4gU2V0IHRvIGBmYWxzZWAgaW4gb3JkZXIgdG8gZGlzYWJsZVxuICAgKiAucHJvamVucmMuanMgZ2VuZXJhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlIGlmIHByb2plbnJjSnNvbiBpcyBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNKcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIC5wcm9qZW5yYy5qc1xuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgb3B0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVucmNKc09wdGlvbnM/OiBQcm9qZW5yY09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIGBCdW5kbGVyYC5cbiAgICovXG4gIHJlYWRvbmx5IGJ1bmRsZXJPcHRpb25zPzogQnVuZGxlck9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEEgZGlyZWN0b3J5IHdoaWNoIHdpbGwgY29udGFpbiBidWlsZCBhcnRpZmFjdHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiZGlzdFwiXG4gICAqL1xuICByZWFkb25seSBhcnRpZmFjdHNEaXJlY3Rvcnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBgcGFja2FnZWAgdGFzayB0aGF0IHdpbGwgcHJvZHVjZSBhbiBucG0gdGFyYmFsbCB1bmRlciB0aGVcbiAgICogYXJ0aWZhY3RzIGRpcmVjdG9yeSAoZS5nLiBgZGlzdGApLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBwYWNrYWdlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQnVpbGQgd29ya2Zsb3cgdHJpZ2dlcnNcbiAgICogQGRlZmF1bHQgXCJ7IHB1bGxSZXF1ZXN0OiB7fSwgd29ya2Zsb3dEaXNwYXRjaDoge30gfVwiXG4gICAqL1xuICByZWFkb25seSBidWlsZFdvcmtmbG93VHJpZ2dlcnM/OiBUcmlnZ2Vycztcbn1cblxuLyoqXG4gKiBBdXRvbWF0aWMgYnVtcCBtb2Rlc1xuICovXG5leHBvcnQgZW51bSBBdXRvUmVsZWFzZSB7XG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gZm9yIGV2ZXJ5IGNvbW1pdCB0byBcIm1haW5cIlxuICAgKi9cbiAgRVZFUllfQ09NTUlULFxuXG4gIC8qKlxuICAgKiBBdXRvbWF0aWNhbGx5IGJ1bXAgJiByZWxlYXNlIGEgbmV3IHZlcnNpb24gb24gYSBkYWlseSBiYXNpcy5cbiAgICovXG4gIERBSUxZLFxufVxuXG4vKipcbiAqIE5vZGUuanMgcHJvamVjdFxuICpcbiAqIEBwamlkIG5vZGVcbiAqL1xuZXhwb3J0IGNsYXNzIE5vZGVQcm9qZWN0IGV4dGVuZHMgR2l0SHViUHJvamVjdCB7XG4gIC8qKlxuICAgKiBBUEkgZm9yIG1hbmFnaW5nIHRoZSBub2RlIHBhY2thZ2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFja2FnZTogTm9kZVBhY2thZ2U7XG5cbiAgLyoqXG4gICAqIFRoZSAubnBtaWdub3JlIGZpbGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbnBtaWdub3JlPzogSWdub3JlRmlsZTtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLmFsbG93TGlicmFyeURlcGVuZGVuY2llc2BcbiAgICovXG4gIHB1YmxpYyBnZXQgYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzKCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWxsb3dMaWJyYXJ5RGVwZW5kZW5jaWVzO1xuICB9XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgcGFja2FnZS5lbnRyeXBvaW50YFxuICAgKi9cbiAgcHVibGljIGdldCBlbnRyeXBvaW50KCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5lbnRyeXBvaW50O1xuICB9XG5cbiAgLyoqXG4gICAqIENvbXBvbmVudCB0aGF0IHNldHMgdXAgbWVyZ2lmeSBmb3IgbWVyZ2luZyBhcHByb3ZlZCBwdWxsIHJlcXVlc3RzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGF1dG9NZXJnZT86IEF1dG9NZXJnZTtcblxuICAvKipcbiAgICogVGhlIFBSIGJ1aWxkIEdpdEh1YiB3b3JrZmxvdy4gYHVuZGVmaW5lZGAgaWYgYGJ1aWxkV29ya2Zsb3dgIGlzIGRpc2FibGVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGJ1aWxkV29ya2Zsb3c/OiBCdWlsZFdvcmtmbG93O1xuXG4gIC8qKlxuICAgKiBQYWNrYWdlIHB1Ymxpc2hlci4gVGhpcyB3aWxsIGJlIGB1bmRlZmluZWRgIGlmIHRoZSBwcm9qZWN0IGRvZXMgbm90IGhhdmUgYVxuICAgKiByZWxlYXNlIHdvcmtmbG93LlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHJlbGVhc2UucHVibGlzaGVyYC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaXNoZXI/OiBQdWJsaXNoZXI7XG5cbiAgLyoqXG4gICAqIFJlbGVhc2UgbWFuYWdlbWVudC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZWxlYXNlPzogUmVsZWFzZTtcblxuICAvKipcbiAgICogTWluaW11bSBub2RlLmpzIHZlcnNpb24gcmVxdWlyZWQgYnkgdGhpcyBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIGdldCBtaW5Ob2RlVmVyc2lvbigpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UubWluTm9kZVZlcnNpb247XG4gIH1cblxuICAvKipcbiAgICogTWF4aW11bSBub2RlIHZlcnNpb24gcmVxdWlyZWQgYnkgdGhpcyBwYWNha2dlLlxuICAgKi9cbiAgcHVibGljIGdldCBtYXhOb2RlVmVyc2lvbigpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UubWF4Tm9kZVZlcnNpb247XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IG5vZGVWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFja2FnZSBtYW5hZ2VyIHRvIHVzZS5cbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwYWNrYWdlLnBhY2thZ2VNYW5hZ2VyYFxuICAgKi9cbiAgcHVibGljIGdldCBwYWNrYWdlTWFuYWdlcigpOiBOb2RlUGFja2FnZU1hbmFnZXIge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXI7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGNvbW1hbmQgdG8gdXNlIHRvIHJ1biBzY3JpcHRzIChlLmcuIGB5YXJuIHJ1bmAgb3IgYG5wbSBydW5gIGRlcGVuZHMgb24gdGhlIHBhY2thZ2UgbWFuYWdlcikuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcnVuU2NyaXB0Q29tbWFuZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSmVzdCBjb25maWd1cmF0aW9uIChpZiBlbmFibGVkKVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGplc3Q/OiBKZXN0O1xuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHBhY2thZ2UuYWRkRmllbGQoeCwgeSlgXG4gICAqL1xuICBwdWJsaWMgZ2V0IG1hbmlmZXN0KCkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UubWFuaWZlc3Q7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgYnVuZGxlcjogQnVuZGxlcjtcblxuICAvKipcbiAgICogVGhlIGJ1aWxkIG91dHB1dCBkaXJlY3RvcnkuIEFuIG5wbSB0YXJiYWxsIHdpbGwgYmUgY3JlYXRlZCB1bmRlciB0aGUgYGpzYFxuICAgKiBzdWJkaXJlY3RvcnkuIEZvciBleGFtcGxlLCBpZiB0aGlzIGlzIHNldCB0byBgZGlzdGAgKHRoZSBkZWZhdWx0KSwgdGhlIG5wbVxuICAgKiB0YXJiYWxsIHdpbGwgYmUgcGxhY2VkIHVuZGVyIGBkaXN0L2pzL2Jvb20tYm9vbS0xLjIuMy50Z2AuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJ0aWZhY3RzRGlyZWN0b3J5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBsb2NhdGlvbiBvZiB0aGUgbnBtIHRhcmJhbGwgYWZ0ZXIgYnVpbGQgKGAke2FydGlmYWN0c0RpcmVjdG9yeX0vanNgKS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcnRpZmFjdHNKYXZhc2NyaXB0RGlyZWN0b3J5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB1cGdyYWRlIHdvcmtmbG93LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHVwZ3JhZGVXb3JrZmxvdz86IFVwZ3JhZGVEZXBlbmRlbmNpZXM7XG5cbiAgcHJpdmF0ZSByZWFkb25seSB3b3JrZmxvd0Jvb3RzdHJhcFN0ZXBzOiBKb2JTdGVwW107XG4gIHByaXZhdGUgcmVhZG9ubHkgd29ya2Zsb3dHaXRJZGVudGl0eTogR2l0SWRlbnRpdHk7XG4gIHB1YmxpYyByZWFkb25seSBwcmV0dGllcj86IFByZXR0aWVyO1xuXG4gIGNvbnN0cnVjdG9yKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKG9wdGlvbnMpO1xuXG4gICAgdGhpcy5wYWNrYWdlID0gbmV3IE5vZGVQYWNrYWdlKHRoaXMsIG9wdGlvbnMpO1xuICAgIHRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyA9IG9wdGlvbnMud29ya2Zsb3dCb290c3RyYXBTdGVwcyA/PyBbXTtcbiAgICB0aGlzLndvcmtmbG93R2l0SWRlbnRpdHkgPVxuICAgICAgb3B0aW9ucy53b3JrZmxvd0dpdElkZW50aXR5ID8/IERFRkFVTFRfR0lUSFVCX0FDVElPTlNfVVNFUjtcbiAgICB0aGlzLmFydGlmYWN0c0RpcmVjdG9yeSA9IG9wdGlvbnMuYXJ0aWZhY3RzRGlyZWN0b3J5ID8/IFwiZGlzdFwiO1xuICAgIHRoaXMuYXJ0aWZhY3RzSmF2YXNjcmlwdERpcmVjdG9yeSA9IGpvaW4odGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksIFwianNcIik7XG5cbiAgICB0aGlzLnJ1blNjcmlwdENvbW1hbmQgPSAoKCkgPT4ge1xuICAgICAgc3dpdGNoICh0aGlzLnBhY2thZ2VNYW5hZ2VyKSB7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLk5QTTpcbiAgICAgICAgICByZXR1cm4gXCJucG0gcnVuXCI7XG4gICAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLllBUk46XG4gICAgICAgICAgcmV0dXJuIFwieWFybiBydW5cIjtcbiAgICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTTpcbiAgICAgICAgICByZXR1cm4gXCJwbnBtIHJ1blwiO1xuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgdW5leHBlY3RlZCBwYWNrYWdlIG1hbmFnZXIgJHt0aGlzLnBhY2thZ2VNYW5hZ2VyfWApO1xuICAgICAgfVxuICAgIH0pKCk7XG5cbiAgICB0aGlzLm5vZGVWZXJzaW9uID1cbiAgICAgIG9wdGlvbnMud29ya2Zsb3dOb2RlVmVyc2lvbiA/PyB0aGlzLnBhY2thZ2UubWluTm9kZVZlcnNpb247XG5cbiAgICAvLyBhZGQgUEFUSCBmb3IgYWxsIHRhc2tzIHdoaWNoIGluY2x1ZGVzIHRoZSBwcm9qZWN0J3MgbnBtIC5iaW4gbGlzdFxuICAgIHRoaXMudGFza3MuYWRkRW52aXJvbm1lbnQoXG4gICAgICBcIlBBVEhcIixcbiAgICAgICckKG5weCAtYyBcIm5vZGUgLWUgXFxcXFwiY29uc29sZS5sb2cocHJvY2Vzcy5lbnYuUEFUSClcXFxcXCJcIiknXG4gICAgKTtcblxuICAgIHRoaXMuYWRkTGljZW5zZShvcHRpb25zKTtcblxuICAgIGlmIChvcHRpb25zLm5wbWlnbm9yZUVuYWJsZWQgPz8gdHJ1ZSkge1xuICAgICAgdGhpcy5ucG1pZ25vcmUgPSBuZXcgSWdub3JlRmlsZSh0aGlzLCBcIi5ucG1pZ25vcmVcIik7XG4gICAgfVxuXG4gICAgdGhpcy5hZGREZWZhdWx0R2l0SWdub3JlKCk7XG5cbiAgICBpZiAob3B0aW9ucy5naXRpZ25vcmU/Lmxlbmd0aCkge1xuICAgICAgZm9yIChjb25zdCBpIG9mIG9wdGlvbnMuZ2l0aWdub3JlKSB7XG4gICAgICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoaSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMubnBtaWdub3JlPy5sZW5ndGgpIHtcbiAgICAgIGlmICghdGhpcy5ucG1pZ25vcmUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICcubnBtaWdub3JlIGlzIG5vdCBkZWZpbmVkIGZvciBhbiBBUFAgcHJvamVjdCB0eXBlLiBBZGQgXCJucG1JZ25vcmU6IHRydWVcIiB0byBvdmVycmlkZSB0aGlzJ1xuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICBmb3IgKGNvbnN0IGkgb2Ygb3B0aW9ucy5ucG1pZ25vcmUpIHtcbiAgICAgICAgdGhpcy5ucG1pZ25vcmUuZXhjbHVkZShpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuZWplY3RlZCkge1xuICAgICAgdGhpcy5zZXRTY3JpcHQoUFJPSkVOX1NDUklQVCwgdGhpcy5wYWNrYWdlLnByb2plbkNvbW1hbmQpO1xuICAgIH1cblxuICAgIHRoaXMubnBtaWdub3JlPy5leGNsdWRlKGAvJHtQUk9KRU5fUkN9YCk7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmV4Y2x1ZGUoYC8ke1BST0pFTl9ESVJ9L2ApO1xuICAgIHRoaXMuZ2l0aWdub3JlLmluY2x1ZGUoYC8ke1BST0pFTl9SQ31gKTtcblxuICAgIGNvbnN0IHByb2plbiA9IG9wdGlvbnMucHJvamVuRGV2RGVwZW5kZW5jeSA/PyB0cnVlO1xuICAgIGlmIChwcm9qZW4gJiYgIXRoaXMuZWplY3RlZCkge1xuICAgICAgY29uc3QgcG9zdGZpeCA9IG9wdGlvbnMucHJvamVuVmVyc2lvbiA/IGBAJHtvcHRpb25zLnByb2plblZlcnNpb259YCA6IFwiXCI7XG4gICAgICB0aGlzLmFkZERldkRlcHMoYHByb2plbiR7cG9zdGZpeH1gKTtcbiAgICB9XG5cbiAgICBpZiAoIW9wdGlvbnMuZGVmYXVsdFJlbGVhc2VCcmFuY2gpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgJ1wiZGVmYXVsdFJlbGVhc2VCcmFuY2hcIiBpcyB0ZW1wb3JhcmlseSBhIHJlcXVpcmVkIG9wdGlvbiB3aGlsZSB3ZSBtaWdyYXRlIGl0cyBkZWZhdWx0IHZhbHVlIGZyb20gXCJtYXN0ZXJcIiB0byBcIm1haW5cIidcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgYnVpbGRFbmFibGVkID0gb3B0aW9ucy5idWlsZFdvcmtmbG93ID8/ICh0aGlzLnBhcmVudCA/IGZhbHNlIDogdHJ1ZSk7XG5cbiAgICAvLyBjb25maWd1cmUgamVzdCBpZiBlbmFibGVkXG4gICAgLy8gbXVzdCBiZSBiZWZvcmUgdGhlIGJ1aWxkL3JlbGVhc2Ugd29ya2Zsb3dzXG4gICAgaWYgKG9wdGlvbnMuamVzdCA/PyB0cnVlKSB7XG4gICAgICB0aGlzLmplc3QgPSBuZXcgSmVzdCh0aGlzLCBvcHRpb25zLmplc3RPcHRpb25zKTtcbiAgICB9XG5cbiAgICBpZiAoYnVpbGRFbmFibGVkICYmIHRoaXMuZ2l0aHViKSB7XG4gICAgICB0aGlzLmJ1aWxkV29ya2Zsb3cgPSBuZXcgQnVpbGRXb3JrZmxvdyh0aGlzLCB7XG4gICAgICAgIGJ1aWxkVGFzazogdGhpcy5idWlsZFRhc2ssXG4gICAgICAgIGFydGlmYWN0c0RpcmVjdG9yeTogdGhpcy5hcnRpZmFjdHNEaXJlY3RvcnksXG4gICAgICAgIGNvbnRhaW5lckltYWdlOiBvcHRpb25zLndvcmtmbG93Q29udGFpbmVySW1hZ2UsXG4gICAgICAgIGdpdElkZW50aXR5OiB0aGlzLndvcmtmbG93R2l0SWRlbnRpdHksXG4gICAgICAgIG11dGFibGVCdWlsZDogb3B0aW9ucy5tdXRhYmxlQnVpbGQsXG4gICAgICAgIHByZUJ1aWxkU3RlcHM6IHRoaXMucmVuZGVyV29ya2Zsb3dTZXR1cCh7XG4gICAgICAgICAgbXV0YWJsZTogb3B0aW9ucy5tdXRhYmxlQnVpbGQgPz8gdHJ1ZSxcbiAgICAgICAgfSksXG4gICAgICAgIHBvc3RCdWlsZFN0ZXBzOiBvcHRpb25zLnBvc3RCdWlsZFN0ZXBzLFxuICAgICAgICBydW5zT246IG9wdGlvbnMud29ya2Zsb3dSdW5zT24sXG4gICAgICAgIHdvcmtmbG93VHJpZ2dlcnM6IG9wdGlvbnMuYnVpbGRXb3JrZmxvd1RyaWdnZXJzLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRTdGVwcyhcbiAgICAgICAgLi4udGhpcy5yZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9ucylcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVsZWFzZSA9XG4gICAgICBvcHRpb25zLnJlbGVhc2UgPz9cbiAgICAgIG9wdGlvbnMucmVsZWFzZVdvcmtmbG93ID8/XG4gICAgICAodGhpcy5wYXJlbnQgPyBmYWxzZSA6IHRydWUpO1xuICAgIGlmIChyZWxlYXNlKSB7XG4gICAgICB0aGlzLmFkZERldkRlcHMoVmVyc2lvbi5TVEFOREFSRF9WRVJTSU9OKTtcblxuICAgICAgdGhpcy5yZWxlYXNlID0gbmV3IFJlbGVhc2UodGhpcywge1xuICAgICAgICB2ZXJzaW9uRmlsZTogXCJwYWNrYWdlLmpzb25cIiwgLy8gdGhpcyBpcyB3aGVyZSBcInZlcnNpb25cIiBpcyBzZXQgYWZ0ZXIgYnVtcFxuICAgICAgICB0YXNrOiB0aGlzLmJ1aWxkVGFzayxcbiAgICAgICAgYnJhbmNoOiBvcHRpb25zLmRlZmF1bHRSZWxlYXNlQnJhbmNoID8/IFwibWFpblwiLFxuICAgICAgICBhcnRpZmFjdHNEaXJlY3Rvcnk6IHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LFxuICAgICAgICAuLi5vcHRpb25zLFxuXG4gICAgICAgIHJlbGVhc2VXb3JrZmxvd1NldHVwU3RlcHM6IFtcbiAgICAgICAgICAuLi50aGlzLnJlbmRlcldvcmtmbG93U2V0dXAoeyBtdXRhYmxlOiBmYWxzZSB9KSxcbiAgICAgICAgICAuLi4ob3B0aW9ucy5yZWxlYXNlV29ya2Zsb3dTZXR1cFN0ZXBzID8/IFtdKSxcbiAgICAgICAgXSxcbiAgICAgICAgcG9zdEJ1aWxkU3RlcHM6IFtcbiAgICAgICAgICAuLi4ob3B0aW9ucy5wb3N0QnVpbGRTdGVwcyA/PyBbXSksXG4gICAgICAgICAgLi4udGhpcy5yZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9ucyksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5wdWJsaXNoZXIgPSB0aGlzLnJlbGVhc2UucHVibGlzaGVyO1xuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0gPz8gZmFsc2UpIHtcbiAgICAgICAgY29uc3QgY29kZUFydGlmYWN0T3B0aW9ucyA9IGlzQXdzQ29kZUFydGlmYWN0UmVnaXN0cnkoXG4gICAgICAgICAgdGhpcy5wYWNrYWdlLm5wbVJlZ2lzdHJ5XG4gICAgICAgIClcbiAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgYWNjZXNzS2V5SWRTZWNyZXQ6IG9wdGlvbnMuY29kZUFydGlmYWN0T3B0aW9ucz8uYWNjZXNzS2V5SWRTZWNyZXQsXG4gICAgICAgICAgICAgIHNlY3JldEFjY2Vzc0tleVNlY3JldDpcbiAgICAgICAgICAgICAgICBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LnNlY3JldEFjY2Vzc0tleVNlY3JldCxcbiAgICAgICAgICAgICAgcm9sZVRvQXNzdW1lOiBvcHRpb25zLmNvZGVBcnRpZmFjdE9wdGlvbnM/LnJvbGVUb0Fzc3VtZSxcbiAgICAgICAgICAgIH1cbiAgICAgICAgICA6IHt9O1xuICAgICAgICB0aGlzLnJlbGVhc2UucHVibGlzaGVyLnB1Ymxpc2hUb05wbSh7XG4gICAgICAgICAgcmVnaXN0cnk6IHRoaXMucGFja2FnZS5ucG1SZWdpc3RyeSxcbiAgICAgICAgICBucG1Ub2tlblNlY3JldDogdGhpcy5wYWNrYWdlLm5wbVRva2VuU2VjcmV0LFxuICAgICAgICAgIGNvZGVBcnRpZmFjdE9wdGlvbnMsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICAvLyB2YWxpZGF0ZSB0aGF0IG5vIHJlbGVhc2Ugb3B0aW9ucyBhcmUgc2VsZWN0ZWQgaWYgdGhlIHJlbGVhc2Ugd29ya2Zsb3cgaXMgZGlzYWJsZWQuXG4gICAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdcInJlbGVhc2VUb05wbVwiIGlzIG5vdCBzdXBwb3J0ZWQgaWYgXCJyZWxlYXNlXCIgaXMgbm90IHNldCdcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgaWYgKG9wdGlvbnMucmVsZWFzZUV2ZXJ5Q29tbWl0KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnXCJyZWxlYXNlRXZlcnlDb21taXRcIiBpcyBub3Qgc3VwcG9ydGVkIGlmIFwicmVsZWFzZVwiIGlzIG5vdCBzZXQnXG4gICAgICAgICk7XG4gICAgICB9XG5cbiAgICAgIGlmIChvcHRpb25zLnJlbGVhc2VTY2hlZHVsZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ1wicmVsZWFzZVNjaGVkdWxlXCIgaXMgbm90IHN1cHBvcnRlZCBpZiBcInJlbGVhc2VcIiBpcyBub3Qgc2V0J1xuICAgICAgICApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChcbiAgICAgIChvcHRpb25zLmF1dG9NZXJnZSA/PyB0cnVlKSAmJlxuICAgICAgdGhpcy5naXRodWI/Lm1lcmdpZnkgJiZcbiAgICAgIHRoaXMuYnVpbGRXb3JrZmxvdz8uYnVpbGRKb2JJZHNcbiAgICApIHtcbiAgICAgIHRoaXMuYXV0b01lcmdlID0gbmV3IEF1dG9NZXJnZSh0aGlzLmdpdGh1Yiwgb3B0aW9ucy5hdXRvTWVyZ2VPcHRpb25zKTtcbiAgICAgIHRoaXMuYXV0b01lcmdlLmFkZENvbmRpdGlvbnNMYXRlcih7XG4gICAgICAgIHJlbmRlcjogKCkgPT5cbiAgICAgICAgICB0aGlzLmJ1aWxkV29ya2Zsb3c/LmJ1aWxkSm9iSWRzLm1hcCgoaWQpID0+IGBzdGF0dXMtc3VjY2Vzcz0ke2lkfWApID8/XG4gICAgICAgICAgW10sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBkZXBlbmRhYm90ID0gb3B0aW9ucy5kZXBlbmRhYm90ID8/IGZhbHNlO1xuICAgIGNvbnN0IGRlcHNVcGdyYWRlID0gb3B0aW9ucy5kZXBzVXBncmFkZSA/PyAhZGVwZW5kYWJvdDtcblxuICAgIGlmIChkZXBlbmRhYm90ICYmIGRlcHNVcGdyYWRlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiJ2RlcGVuZGFib3QnIGNhbm5vdCBiZSBjb25maWd1cmVkIHRvZ2V0aGVyIHdpdGggJ2RlcHNVcGdyYWRlJ1wiXG4gICAgICApO1xuICAgIH1cblxuICAgIGNvbnN0IGRlcHNBdXRvQXBwcm92ZSA9IG9wdGlvbnMuYXV0b0FwcHJvdmVVcGdyYWRlcyA/PyBmYWxzZTtcblxuICAgIGlmIChkZXBzQXV0b0FwcHJvdmUgJiYgIXRoaXMuYXV0b0FwcHJvdmUgJiYgdGhpcy5naXRodWIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgXCJBdXRvbWF0aWMgYXBwcm92YWwgb2YgZGVwZW5kZW5jaWVzIHVwZ3JhZGVzIHJlcXVpcmVzIGNvbmZpZ3VyaW5nIGBhdXRvQXBwcm92ZU9wdGlvbnNgXCJcbiAgICAgICk7XG4gICAgfVxuXG4gICAgY29uc3QgYXV0b0FwcHJvdmVMYWJlbCA9IChjb25kaXRpb246IGJvb2xlYW4pID0+XG4gICAgICBjb25kaXRpb24gJiYgdGhpcy5hdXRvQXBwcm92ZT8ubGFiZWxcbiAgICAgICAgPyBbdGhpcy5hdXRvQXBwcm92ZS5sYWJlbF1cbiAgICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICBpZiAoZGVwZW5kYWJvdCkge1xuICAgICAgY29uc3QgZGVmYXVsdE9wdGlvbnMgPSB7XG4gICAgICAgIGxhYmVsczogYXV0b0FwcHJvdmVMYWJlbChkZXBzQXV0b0FwcHJvdmUpLFxuICAgICAgfTtcbiAgICAgIHRoaXMuZ2l0aHViPy5hZGREZXBlbmRhYm90KFxuICAgICAgICBkZWVwTWVyZ2UoW2RlZmF1bHRPcHRpb25zLCBvcHRpb25zLmRlcGVuZGFib3RPcHRpb25zID8/IHt9XSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKGRlcHNVcGdyYWRlKSB7XG4gICAgICBjb25zdCBkZWZhdWx0T3B0aW9uczogVXBncmFkZURlcGVuZGVuY2llc09wdGlvbnMgPSB7XG4gICAgICAgIHdvcmtmbG93T3B0aW9uczoge1xuICAgICAgICAgIGNvbnRhaW5lcjogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlXG4gICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICBpbWFnZTogb3B0aW9ucy53b3JrZmxvd0NvbnRhaW5lckltYWdlLFxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBsYWJlbHM6IGF1dG9BcHByb3ZlTGFiZWwoZGVwc0F1dG9BcHByb3ZlKSxcbiAgICAgICAgICBnaXRJZGVudGl0eTogdGhpcy53b3JrZmxvd0dpdElkZW50aXR5LFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICAgIHRoaXMudXBncmFkZVdvcmtmbG93ID0gbmV3IFVwZ3JhZGVEZXBlbmRlbmNpZXMoXG4gICAgICAgIHRoaXMsXG4gICAgICAgIGRlZXBNZXJnZShbZGVmYXVsdE9wdGlvbnMsIG9wdGlvbnMuZGVwc1VwZ3JhZGVPcHRpb25zID8/IHt9XSlcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucHVsbFJlcXVlc3RUZW1wbGF0ZSA/PyB0cnVlKSB7XG4gICAgICB0aGlzLmdpdGh1Yj8uYWRkUHVsbFJlcXVlc3RUZW1wbGF0ZShcbiAgICAgICAgLi4uKG9wdGlvbnMucHVsbFJlcXVlc3RUZW1wbGF0ZUNvbnRlbnRzID8/IFtdKVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm9qZW5yY0pzID0gb3B0aW9ucy5wcm9qZW5yY0pzID8/ICFvcHRpb25zLnByb2plbnJjSnNvbjtcbiAgICBpZiAocHJvamVucmNKcykge1xuICAgICAgbmV3IFByb2plbnJjKHRoaXMsIG9wdGlvbnMucHJvamVucmNKc09wdGlvbnMpO1xuICAgIH1cblxuICAgIC8vIGFkZCBhIGJ1bmRsZXIgY29tcG9uZW50IC0gdGhpcyBlbmFibGVzIHRoaW5ncyBsaWtlIExhbWJkYSBidW5kbGluZyBhbmQgaW4gdGhlIGZ1dHVyZSB3ZWIgYnVuZGxpbmcuXG4gICAgdGhpcy5idW5kbGVyID0gbmV3IEJ1bmRsZXIodGhpcywgb3B0aW9ucy5idW5kbGVyT3B0aW9ucyk7XG5cbiAgICBpZiAob3B0aW9ucy5wYWNrYWdlID8/IHRydWUpIHtcbiAgICAgIHRoaXMucGFja2FnZVRhc2suZXhlYyhgbWtkaXIgLXAgJHt0aGlzLmFydGlmYWN0c0phdmFzY3JpcHREaXJlY3Rvcnl9YCk7XG5cbiAgICAgIC8vIGFsd2F5cyB1c2UgbnBtIGhlcmUgLSB5YXJuIGRvZXNuJ3QgYWRkIG11Y2ggdmFsdWVcbiAgICAgIC8vIHNhZGx5IHdlIGNhbm5vdCB1c2UgLS1wYWNrLWRlc3RpbmF0aW9uIGJlY2F1c2UgaXQgaXMgbm90IHN1cHBvcnRlZCBieSBvbGRlciBucG1cbiAgICAgIHRoaXMucGFja2FnZVRhc2suZXhlYyhcbiAgICAgICAgYG12ICQobnBtIHBhY2spICR7dGhpcy5hcnRpZmFjdHNKYXZhc2NyaXB0RGlyZWN0b3J5fS9gXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLnByZXR0aWVyID8/IGZhbHNlKSB7XG4gICAgICB0aGlzLnByZXR0aWVyID0gbmV3IFByZXR0aWVyKHRoaXMsIHsgLi4ub3B0aW9ucy5wcmV0dGllck9wdGlvbnMgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJVcGxvYWRDb3ZlcmFnZUpvYlN0ZXAob3B0aW9uczogTm9kZVByb2plY3RPcHRpb25zKTogSm9iU3RlcFtdIHtcbiAgICAvLyBydW4gY29kZWNvdiBpZiBlbmFibGVkIG9yIGEgc2VjcmV0IHRva2VuIG5hbWUgaXMgcGFzc2VkIGluXG4gICAgLy8gQU5EIGplc3QgbXVzdCBiZSBjb25maWd1cmVkXG4gICAgaWYgKChvcHRpb25zLmNvZGVDb3YgfHwgb3B0aW9ucy5jb2RlQ292VG9rZW5TZWNyZXQpICYmIHRoaXMuamVzdD8uY29uZmlnKSB7XG4gICAgICByZXR1cm4gW1xuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogXCJVcGxvYWQgY292ZXJhZ2UgdG8gQ29kZWNvdlwiLFxuICAgICAgICAgIHVzZXM6IFwiY29kZWNvdi9jb2RlY292LWFjdGlvbkB2MVwiLFxuICAgICAgICAgIHdpdGg6IG9wdGlvbnMuY29kZUNvdlRva2VuU2VjcmV0XG4gICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICB0b2tlbjogYFxcJHt7IHNlY3JldHMuJHtvcHRpb25zLmNvZGVDb3ZUb2tlblNlY3JldH0gfX1gLFxuICAgICAgICAgICAgICAgIGRpcmVjdG9yeTogdGhpcy5qZXN0LmNvbmZpZy5jb3ZlcmFnZURpcmVjdG9yeSxcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgOiB7XG4gICAgICAgICAgICAgICAgZGlyZWN0b3J5OiB0aGlzLmplc3QuY29uZmlnLmNvdmVyYWdlRGlyZWN0b3J5LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIFtdO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhZGRCaW5zKGJpbnM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4pIHtcbiAgICB0aGlzLnBhY2thZ2UuYWRkQmluKGJpbnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlcGxhY2VzIHRoZSBjb250ZW50cyBvZiBhbiBucG0gcGFja2FnZS5qc29uIHNjcmlwdC5cbiAgICpcbiAgICogQHBhcmFtIG5hbWUgVGhlIHNjcmlwdCBuYW1lXG4gICAqIEBwYXJhbSBjb21tYW5kIFRoZSBjb21tYW5kIHRvIGV4ZWN1dGVcbiAgICovXG4gIHB1YmxpYyBzZXRTY3JpcHQobmFtZTogc3RyaW5nLCBjb21tYW5kOiBzdHJpbmcpIHtcbiAgICB0aGlzLnBhY2thZ2Uuc2V0U2NyaXB0KG5hbWUsIGNvbW1hbmQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgdGhlIG5wbSBzY3JpcHQgKGFsd2F5cyBzdWNjZXNzZnVsKS5cbiAgICogQHBhcmFtIG5hbWUgVGhlIG5hbWUgb2YgdGhlIHNjcmlwdC5cbiAgICovXG4gIHB1YmxpYyByZW1vdmVTY3JpcHQobmFtZTogc3RyaW5nKSB7XG4gICAgdGhpcy5wYWNrYWdlLnJlbW92ZVNjcmlwdChuYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgaWYgYSBzY3JpcHQgYnkgdGhlIG5hbWUgbmFtZSBpcyBkZWZpbmVkLlxuICAgKiBAcGFyYW0gbmFtZSBUaGUgbmFtZSBvZiB0aGUgc2NyaXB0XG4gICAqL1xuICBwdWJsaWMgaGFzU2NyaXB0KG5hbWU6IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuaGFzU2NyaXB0KG5hbWUpO1xuICB9XG5cbiAgLyoqXG4gICAqIERFUFJFQ0FURURcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwcm9qZWN0LmNvbXBpbGVUYXNrLmV4ZWMoKWBcbiAgICovXG4gIHB1YmxpYyBhZGRDb21waWxlQ29tbWFuZCguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICBmb3IgKGNvbnN0IGMgb2YgY29tbWFuZHMpIHtcbiAgICAgIHRoaXMuY29tcGlsZVRhc2suZXhlYyhjKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogREVQUkVDQVRFRFxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHByb2plY3QudGVzdFRhc2suZXhlYygpYFxuICAgKi9cbiAgcHVibGljIGFkZFRlc3RDb21tYW5kKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgIGZvciAoY29uc3QgYyBvZiBjb21tYW5kcykge1xuICAgICAgdGhpcy50ZXN0VGFzay5leGVjKGMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBEaXJlY3RseSBzZXQgZmllbGRzIGluIGBwYWNrYWdlLmpzb25gLlxuICAgKiBAcGFyYW0gZmllbGRzIFRoZSBmaWVsZHMgdG8gc2V0XG4gICAqL1xuICBwdWJsaWMgYWRkRmllbGRzKGZpZWxkczogeyBbbmFtZTogc3RyaW5nXTogYW55IH0pIHtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCB2YWx1ZV0gb2YgT2JqZWN0LmVudHJpZXMoZmllbGRzKSkge1xuICAgICAgdGhpcy5wYWNrYWdlLmFkZEZpZWxkKG5hbWUsIHZhbHVlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBrZXl3b3JkcyB0byBwYWNrYWdlLmpzb24gKGRlZHVwbGljYXRlZClcbiAgICogQHBhcmFtIGtleXdvcmRzIFRoZSBrZXl3b3JkcyB0byBhZGRcbiAgICovXG4gIHB1YmxpYyBhZGRLZXl3b3JkcyguLi5rZXl3b3Jkczogc3RyaW5nW10pIHtcbiAgICB0aGlzLnBhY2thZ2UuYWRkS2V5d29yZHMoLi4ua2V5d29yZHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldCBzdGVwcyBmb3Igc2NvcGVkIHBhY2thZ2UgYWNjZXNzXG4gICAqXG4gICAqIEBwYXJhbSBjb2RlQXJ0aWZhY3RPcHRpb25zIERldGFpbHMgb2YgbG9nZ2luZyBpbiB0byBBV1NcbiAgICogQHJldHVybnMgYXJyYXkgb2Ygam9iIHN0ZXBzIHJlcXVpcmVkIGZvciBlYWNoIHByaXZhdGUgc2NvcGVkIHBhY2thZ2VzXG4gICAqL1xuICBwcml2YXRlIGdldFNjb3BlZFBhY2thZ2VTdGVwcyhcbiAgICBjb2RlQXJ0aWZhY3RPcHRpb25zOiBDb2RlQXJ0aWZhY3RPcHRpb25zIHwgdW5kZWZpbmVkXG4gICk6IEpvYlN0ZXBbXSB7XG4gICAgY29uc3QgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucyA9IHtcbiAgICAgIGFjY2Vzc0tleUlkU2VjcmV0OlxuICAgICAgICBjb2RlQXJ0aWZhY3RPcHRpb25zPy5hY2Nlc3NLZXlJZFNlY3JldCA/PyBcIkFXU19BQ0NFU1NfS0VZX0lEXCIsXG4gICAgICBzZWNyZXRBY2Nlc3NLZXlTZWNyZXQ6XG4gICAgICAgIGNvZGVBcnRpZmFjdE9wdGlvbnM/LnNlY3JldEFjY2Vzc0tleVNlY3JldCA/PyBcIkFXU19TRUNSRVRfQUNDRVNTX0tFWVwiLFxuICAgICAgcm9sZVRvQXNzdW1lOiBjb2RlQXJ0aWZhY3RPcHRpb25zPy5yb2xlVG9Bc3N1bWUsXG4gICAgfTtcblxuICAgIGlmIChwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLnJvbGVUb0Fzc3VtZSkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQ29uZmlndXJlIEFXUyBDcmVkZW50aWFsc1wiLFxuICAgICAgICAgIHVzZXM6IFwiYXdzLWFjdGlvbnMvY29uZmlndXJlLWF3cy1jcmVkZW50aWFsc0B2MVwiLFxuICAgICAgICAgIHdpdGg6IHtcbiAgICAgICAgICAgIFwiYXdzLWFjY2Vzcy1rZXktaWRcIjogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICAgIHBhcnNlZENvZGVBcnRpZmFjdE9wdGlvbnMuYWNjZXNzS2V5SWRTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1zZWNyZXQtYWNjZXNzLWtleVwiOiBzZWNyZXRUb1N0cmluZyhcbiAgICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICAgICksXG4gICAgICAgICAgICBcImF3cy1yZWdpb25cIjogXCJ1cy1lYXN0LTJcIixcbiAgICAgICAgICAgIFwicm9sZS10by1hc3N1bWVcIjogcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5yb2xlVG9Bc3N1bWUsXG4gICAgICAgICAgICBcInJvbGUtZHVyYXRpb24tc2Vjb25kc1wiOiA5MDAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICAgIHJ1bjogYCR7dGhpcy5ydW5TY3JpcHRDb21tYW5kfSBjYTpsb2dpbmAsXG4gICAgICAgIH0sXG4gICAgICBdO1xuICAgIH1cblxuICAgIHJldHVybiBbXG4gICAgICB7XG4gICAgICAgIG5hbWU6IFwiQVdTIENvZGVBcnRpZmFjdCBMb2dpblwiLFxuICAgICAgICBydW46IGAke3RoaXMucnVuU2NyaXB0Q29tbWFuZH0gY2E6bG9naW5gLFxuICAgICAgICBlbnY6IHtcbiAgICAgICAgICBBV1NfQUNDRVNTX0tFWV9JRDogc2VjcmV0VG9TdHJpbmcoXG4gICAgICAgICAgICBwYXJzZWRDb2RlQXJ0aWZhY3RPcHRpb25zLmFjY2Vzc0tleUlkU2VjcmV0XG4gICAgICAgICAgKSxcbiAgICAgICAgICBBV1NfU0VDUkVUX0FDQ0VTU19LRVk6IHNlY3JldFRvU3RyaW5nKFxuICAgICAgICAgICAgcGFyc2VkQ29kZUFydGlmYWN0T3B0aW9ucy5zZWNyZXRBY2Nlc3NLZXlTZWNyZXRcbiAgICAgICAgICApLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHNldCBvZiB3b3JrZmxvdyBzdGVwcyB3aGljaCBzaG91bGQgYmUgZXhlY3V0ZWQgdG8gYm9vdHN0cmFwIGFcbiAgICogd29ya2Zsb3cuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMuXG4gICAqIEByZXR1cm5zIEpvYiBzdGVwc1xuICAgKi9cbiAgcHVibGljIHJlbmRlcldvcmtmbG93U2V0dXAoXG4gICAgb3B0aW9uczogUmVuZGVyV29ya2Zsb3dTZXR1cE9wdGlvbnMgPSB7fVxuICApOiBKb2JTdGVwW10ge1xuICAgIGNvbnN0IGluc3RhbGwgPSBuZXcgQXJyYXk8Sm9iU3RlcD4oKTtcblxuICAgIC8vIGZpcnN0IHJ1biB0aGUgd29ya2Zsb3cgYm9vdHN0cmFwIHN0ZXBzXG4gICAgaW5zdGFsbC5wdXNoKC4uLnRoaXMud29ya2Zsb3dCb290c3RyYXBTdGVwcyk7XG5cbiAgICBpZiAodGhpcy5ub2RlVmVyc2lvbikge1xuICAgICAgaW5zdGFsbC5wdXNoKHtcbiAgICAgICAgbmFtZTogXCJTZXR1cCBOb2RlLmpzXCIsXG4gICAgICAgIHVzZXM6IFwiYWN0aW9ucy9zZXR1cC1ub2RlQHYyLjIuMFwiLFxuICAgICAgICB3aXRoOiB7IFwibm9kZS12ZXJzaW9uXCI6IHRoaXMubm9kZVZlcnNpb24gfSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXIgPT09IE5vZGVQYWNrYWdlTWFuYWdlci5QTlBNKSB7XG4gICAgICBpbnN0YWxsLnB1c2goe1xuICAgICAgICBuYW1lOiBcIlNldHVwIHBucG1cIixcbiAgICAgICAgdXNlczogXCJwbnBtL2FjdGlvbi1zZXR1cEB2Mi4wLjFcIixcbiAgICAgICAgd2l0aDogeyB2ZXJzaW9uOiBcIjYuMTQuN1wiIH0sIC8vIGN1cnJlbnQgbGF0ZXN0LiBTaG91bGQgcHJvYmFibHkgYmVjb21lIHR1bmFibGUuXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBtdXRhYmxlID0gb3B0aW9ucy5tdXRhYmxlID8/IGZhbHNlO1xuXG4gICAgaWYgKHRoaXMucGFja2FnZS5zY29wZWRQYWNrYWdlc09wdGlvbnMpIHtcbiAgICAgIGluc3RhbGwucHVzaChcbiAgICAgICAgLi4udGhpcy5nZXRTY29wZWRQYWNrYWdlU3RlcHModGhpcy5wYWNrYWdlLmNvZGVBcnRpZmFjdE9wdGlvbnMpXG4gICAgICApO1xuICAgIH1cblxuICAgIGluc3RhbGwucHVzaCh7XG4gICAgICBuYW1lOiBcIkluc3RhbGwgZGVwZW5kZW5jaWVzXCIsXG4gICAgICBydW46IG11dGFibGVcbiAgICAgICAgPyB0aGlzLnBhY2thZ2UuaW5zdGFsbEFuZFVwZGF0ZUxvY2tmaWxlQ29tbWFuZFxuICAgICAgICA6IHRoaXMucGFja2FnZS5pbnN0YWxsQ29tbWFuZCxcbiAgICB9KTtcblxuICAgIHJldHVybiBpbnN0YWxsO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgbm9ybWFsIGRlcGVuZGVuY2llcy5cbiAgICpcbiAgICogQHBhcmFtIGRlcHMgTmFtZXMgbW9kdWxlcyB0byBpbnN0YWxsLiBCeSBkZWZhdWx0LCB0aGUgdGhlIGRlcGVuZGVuY3kgd2lsbFxuICAgKiBiZSBpbnN0YWxsZWQgaW4gdGhlIG5leHQgYG5weCBwcm9qZW5gIHJ1biBhbmQgdGhlIHZlcnNpb24gd2lsbCBiZSByZWNvcmRlZFxuICAgKiBpbiB5b3VyIGBwYWNrYWdlLmpzb25gIGZpbGUuIFlvdSBjYW4gdXBncmFkZSBtYW51YWxseSBvciB1c2luZyBgeWFyblxuICAgKiBhZGQvdXBncmFkZWAuIElmIHlvdSB3aXNoIHRvIHNwZWNpZnkgYSB2ZXJzaW9uIHJhbmdlIHVzZSB0aGlzIHN5bnRheDpcbiAgICogYG1vZHVsZUBeN2AuXG4gICAqL1xuICBwdWJsaWMgYWRkRGVwcyguLi5kZXBzOiBzdHJpbmdbXSkge1xuICAgIHJldHVybiB0aGlzLnBhY2thZ2UuYWRkRGVwcyguLi5kZXBzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGRldmVsb3BtZW50L3Rlc3QgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBAcGFyYW0gZGVwcyBOYW1lcyBtb2R1bGVzIHRvIGluc3RhbGwuIEJ5IGRlZmF1bHQsIHRoZSB0aGUgZGVwZW5kZW5jeSB3aWxsXG4gICAqIGJlIGluc3RhbGxlZCBpbiB0aGUgbmV4dCBgbnB4IHByb2plbmAgcnVuIGFuZCB0aGUgdmVyc2lvbiB3aWxsIGJlIHJlY29yZGVkXG4gICAqIGluIHlvdXIgYHBhY2thZ2UuanNvbmAgZmlsZS4gWW91IGNhbiB1cGdyYWRlIG1hbnVhbGx5IG9yIHVzaW5nIGB5YXJuXG4gICAqIGFkZC91cGdyYWRlYC4gSWYgeW91IHdpc2ggdG8gc3BlY2lmeSBhIHZlcnNpb24gcmFuZ2UgdXNlIHRoaXMgc3ludGF4OlxuICAgKiBgbW9kdWxlQF43YC5cbiAgICovXG4gIHB1YmxpYyBhZGREZXZEZXBzKC4uLmRlcHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hZGREZXZEZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZXMgcGVlciBkZXBlbmRlbmNpZXMuXG4gICAqXG4gICAqIFdoZW4gYWRkaW5nIHBlZXIgZGVwZW5kZW5jaWVzLCBhIGRldkRlcGVuZGVuY3kgd2lsbCBhbHNvIGJlIGFkZGVkIG9uIHRoZVxuICAgKiBwaW5uZWQgdmVyc2lvbiBvZiB0aGUgZGVjbGFyZWQgcGVlci4gVGhpcyB3aWxsIGVuc3VyZSB0aGF0IHlvdSBhcmUgdGVzdGluZ1xuICAgKiB5b3VyIGNvZGUgYWdhaW5zdCB0aGUgbWluaW11bSB2ZXJzaW9uIHJlcXVpcmVkIGZyb20geW91ciBjb25zdW1lcnMuXG4gICAqXG4gICAqIEBwYXJhbSBkZXBzIE5hbWVzIG1vZHVsZXMgdG8gaW5zdGFsbC4gQnkgZGVmYXVsdCwgdGhlIHRoZSBkZXBlbmRlbmN5IHdpbGxcbiAgICogYmUgaW5zdGFsbGVkIGluIHRoZSBuZXh0IGBucHggcHJvamVuYCBydW4gYW5kIHRoZSB2ZXJzaW9uIHdpbGwgYmUgcmVjb3JkZWRcbiAgICogaW4geW91ciBgcGFja2FnZS5qc29uYCBmaWxlLiBZb3UgY2FuIHVwZ3JhZGUgbWFudWFsbHkgb3IgdXNpbmcgYHlhcm5cbiAgICogYWRkL3VwZ3JhZGVgLiBJZiB5b3Ugd2lzaCB0byBzcGVjaWZ5IGEgdmVyc2lvbiByYW5nZSB1c2UgdGhpcyBzeW50YXg6XG4gICAqIGBtb2R1bGVAXjdgLlxuICAgKi9cbiAgcHVibGljIGFkZFBlZXJEZXBzKC4uLmRlcHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIHRoaXMucGFja2FnZS5hZGRQZWVyRGVwcyguLi5kZXBzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGJ1bmRsZWQgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBCdW5kbGVkIGRlcGVuZGVuY2llcyB3aWxsIGJlIGFkZGVkIGFzIG5vcm1hbCBkZXBlbmRlbmNpZXMgYXMgd2VsbCBhcyB0byB0aGVcbiAgICogYGJ1bmRsZWREZXBlbmRlbmNpZXNgIHNlY3Rpb24gb2YgeW91ciBgcGFja2FnZS5qc29uYC5cbiAgICpcbiAgICogQHBhcmFtIGRlcHMgTmFtZXMgbW9kdWxlcyB0byBpbnN0YWxsLiBCeSBkZWZhdWx0LCB0aGUgdGhlIGRlcGVuZGVuY3kgd2lsbFxuICAgKiBiZSBpbnN0YWxsZWQgaW4gdGhlIG5leHQgYG5weCBwcm9qZW5gIHJ1biBhbmQgdGhlIHZlcnNpb24gd2lsbCBiZSByZWNvcmRlZFxuICAgKiBpbiB5b3VyIGBwYWNrYWdlLmpzb25gIGZpbGUuIFlvdSBjYW4gdXBncmFkZSBtYW51YWxseSBvciB1c2luZyBgeWFyblxuICAgKiBhZGQvdXBncmFkZWAuIElmIHlvdSB3aXNoIHRvIHNwZWNpZnkgYSB2ZXJzaW9uIHJhbmdlIHVzZSB0aGlzIHN5bnRheDpcbiAgICogYG1vZHVsZUBeN2AuXG4gICAqL1xuICBwdWJsaWMgYWRkQnVuZGxlZERlcHMoLi4uZGVwczogc3RyaW5nW10pIHtcbiAgICByZXR1cm4gdGhpcy5wYWNrYWdlLmFkZEJ1bmRsZWREZXBzKC4uLmRlcHMpO1xuICB9XG5cbiAgcHVibGljIGFkZFBhY2thZ2VJZ25vcmUocGF0dGVybjogc3RyaW5nKSB7XG4gICAgdGhpcy5ucG1pZ25vcmU/LmFkZFBhdHRlcm5zKHBhdHRlcm4pO1xuICB9XG5cbiAgcHJpdmF0ZSBhZGRMaWNlbnNlKG9wdGlvbnM6IE5vZGVQcm9qZWN0T3B0aW9ucykge1xuICAgIGlmICh0aGlzLnBhY2thZ2UubGljZW5zZSkge1xuICAgICAgbmV3IExpY2Vuc2UodGhpcywge1xuICAgICAgICBzcGR4OiB0aGlzLnBhY2thZ2UubGljZW5zZSxcbiAgICAgICAgY29weXJpZ2h0T3duZXI6IG9wdGlvbnMuY29weXJpZ2h0T3duZXIgPz8gb3B0aW9ucy5hdXRob3JOYW1lLFxuICAgICAgICBjb3B5cmlnaHRQZXJpb2Q6IG9wdGlvbnMuY29weXJpZ2h0UGVyaW9kLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBhZGREZWZhdWx0R2l0SWdub3JlKCkge1xuICAgIHRoaXMuZ2l0aWdub3JlLmV4Y2x1ZGUoXG4gICAgICBcIiMgTG9nc1wiLFxuICAgICAgXCJsb2dzXCIsXG4gICAgICBcIioubG9nXCIsXG4gICAgICBcIm5wbS1kZWJ1Zy5sb2cqXCIsXG4gICAgICBcInlhcm4tZGVidWcubG9nKlwiLFxuICAgICAgXCJ5YXJuLWVycm9yLmxvZypcIixcbiAgICAgIFwibGVybmEtZGVidWcubG9nKlwiLFxuXG4gICAgICBcIiMgRGlhZ25vc3RpYyByZXBvcnRzIChodHRwczovL25vZGVqcy5vcmcvYXBpL3JlcG9ydC5odG1sKVwiLFxuICAgICAgXCJyZXBvcnQuWzAtOV0qLlswLTldKi5bMC05XSouWzAtOV0qLmpzb25cIixcblxuICAgICAgXCIjIFJ1bnRpbWUgZGF0YVwiLFxuICAgICAgXCJwaWRzXCIsXG4gICAgICBcIioucGlkXCIsXG4gICAgICBcIiouc2VlZFwiLFxuICAgICAgXCIqLnBpZC5sb2NrXCIsXG5cbiAgICAgIFwiIyBEaXJlY3RvcnkgZm9yIGluc3RydW1lbnRlZCBsaWJzIGdlbmVyYXRlZCBieSBqc2NvdmVyYWdlL0pTQ292ZXJcIixcbiAgICAgIFwibGliLWNvdlwiLFxuXG4gICAgICBcIiMgQ292ZXJhZ2UgZGlyZWN0b3J5IHVzZWQgYnkgdG9vbHMgbGlrZSBpc3RhbmJ1bFwiLFxuICAgICAgXCJjb3ZlcmFnZVwiLFxuICAgICAgXCIqLmxjb3ZcIixcblxuICAgICAgXCIjIG55YyB0ZXN0IGNvdmVyYWdlXCIsXG4gICAgICBcIi5ueWNfb3V0cHV0XCIsXG5cbiAgICAgIFwiIyBDb21waWxlZCBiaW5hcnkgYWRkb25zIChodHRwczovL25vZGVqcy5vcmcvYXBpL2FkZG9ucy5odG1sKVwiLFxuICAgICAgXCJidWlsZC9SZWxlYXNlXCIsXG5cbiAgICAgIFwiIyBEZXBlbmRlbmN5IGRpcmVjdG9yaWVzXCIsXG4gICAgICBcIm5vZGVfbW9kdWxlcy9cIixcbiAgICAgIFwianNwbV9wYWNrYWdlcy9cIixcblxuICAgICAgXCIjIFR5cGVTY3JpcHQgY2FjaGVcIixcbiAgICAgIFwiKi50c2J1aWxkaW5mb1wiLFxuXG4gICAgICBcIiMgT3B0aW9uYWwgZXNsaW50IGNhY2hlXCIsXG4gICAgICBcIi5lc2xpbnRjYWNoZVwiLFxuXG4gICAgICBcIiMgT3V0cHV0IG9mICducG0gcGFjaydcIixcbiAgICAgIFwiKi50Z3pcIixcblxuICAgICAgXCIjIFlhcm4gSW50ZWdyaXR5IGZpbGVcIixcbiAgICAgIFwiLnlhcm4taW50ZWdyaXR5XCIsXG5cbiAgICAgIFwiIyBwYXJjZWwtYnVuZGxlciBjYWNoZSAoaHR0cHM6Ly9wYXJjZWxqcy5vcmcvKVwiLFxuICAgICAgXCIuY2FjaGVcIlxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgc2hlbGwgY29tbWFuZCB0byBleGVjdXRlIGluIG9yZGVyIHRvIHJ1biBhIHRhc2suIFRoaXMgd2lsbFxuICAgKiB0eXBpY2FsbHkgYmUgYG5weCBwcm9qZW4gVEFTS2AuXG4gICAqXG4gICAqIEBwYXJhbSB0YXNrIFRoZSB0YXNrIGZvciB3aGljaCB0aGUgY29tbWFuZCBpcyByZXF1aXJlZFxuICAgKi9cbiAgcHVibGljIHJ1blRhc2tDb21tYW5kKHRhc2s6IFRhc2spIHtcbiAgICByZXR1cm4gYCR7dGhpcy5wYWNrYWdlLnByb2plbkNvbW1hbmR9ICR7dGFzay5uYW1lfWA7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGpvYiBJRCBvZiB0aGUgYnVpbGQgd29ya2Zsb3cuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGJ1aWxkV29ya2Zsb3dKb2JJZCgpIHtcbiAgICByZXR1cm4gdGhpcy5idWlsZFdvcmtmbG93Py5idWlsZEpvYklkc1swXTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGByZW5kZXJJbnN0YWxsU3RlcHMoKWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVuZGVyV29ya2Zsb3dTZXR1cE9wdGlvbnMge1xuICAvKipcbiAgICogU2hvdWxkIHRoZSBwYWNha2dlIGxvY2tmaWxlIGJlIHVwZGF0ZWQ/XG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBtdXRhYmxlPzogYm9vbGVhbjtcbn1cbiJdfQ==