import { Component } from "../component";
import { Project } from "../project";
import { Task } from "../task";
/**
 * Options for `Bundler`.
 */
export interface BundlerOptions {
    /**
     * The semantic version requirement for `esbuild`.
     *
     * @default - no specific version (implies latest)
     */
    readonly esbuildVersion?: string;
    /**
     * Output directory for all bundles.
     * @default "assets"
     */
    readonly assetsDir?: string;
    /**
     * Install the `bundle` command as a pre-compile phase.
     *
     * @default true
     */
    readonly addToPreCompile?: boolean;
}
/**
 * Adds support for bundling JavaScript applications and dependencies into a
 * single file. In the future, this will also supports bundling websites.
 */
export declare class Bundler extends Component {
    /**
     * Returns the `Bundler` instance associated with a project or `undefined` if
     * there is no Bundler.
     * @param project The project
     * @returns A bundler
     */
    static of(project: Project): Bundler | undefined;
    /**
     * The semantic version requirement for `esbuild` (if defined).
     */
    readonly esbuildVersion: string | undefined;
    /**
     * Root bundle directory.
     */
    readonly bundledir: string;
    private _task;
    private readonly addToPreCompile;
    /**
     * Creates a `Bundler`.
     */
    constructor(project: Project, options?: BundlerOptions);
    /**
     * Gets or creates the singleton "bundle" task of the project.
     *
     * If the project doesn't have a "bundle" task, it will be created and spawned
     * during the pre-compile phase.
     */
    get bundleTask(): Task;
    /**
     * Adds a task to the project which bundles a specific entrypoint and all of
     * its dependencies into a single javascript output file.
     *
     * @param entrypoint The relative path of the artifact within the project
     * @param options Bundling options
     */
    addBundle(entrypoint: string, options: AddBundleOptions): Bundle;
    /**
     * Add bundling support to a project. This is called implicitly when
     * `bundleTask` is referenced first. It adds the dependency on `esbuild`,
     * gitignore/npmignore, etc.
     */
    private addBundlingSupport;
}
export interface Bundle {
    /**
     * The task that produces this bundle.
     */
    readonly bundleTask: Task;
    /**
     * The "watch" task for this bundle.
     */
    readonly watchTask?: Task;
    /**
     * Location of the output file (relative to project root).
     */
    readonly outfile: string;
    /**
     * Base directory containing the output file (relative to project root).
     */
    readonly outdir: string;
}
/**
 * Options for bundling.
 */
export interface BundlingOptions {
    /**
     * You can mark a file or a package as external to exclude it from your build.
     * Instead of being bundled, the import will be preserved (using require for
     * the iife and cjs formats and using import for the esm format) and will be
     * evaluated at run time instead.
     *
     * This has several uses. First of all, it can be used to trim unnecessary
     * code from your bundle for a code path that you know will never be executed.
     * For example, a package may contain code that only runs in node but you will
     * only be using that package in the browser. It can also be used to import
     * code in node at run time from a package that cannot be bundled. For
     * example, the fsevents package contains a native extension, which esbuild
     * doesn't support.
     *
     * @default []
     */
    readonly externals?: string[];
    /**
     * Include a source map in the bundle.
     *
     * @default false
     */
    readonly sourcemap?: boolean;
    /**
     * In addition to the `bundle:xyz` task, creates `bundle:xyz:watch` task which will
     * invoke the same esbuild command with the `--watch` flag. This can be used
     * to continusouly watch for changes.
     *
     * @default true
     */
    readonly watchTask?: boolean;
}
/**
 * Options for `addBundle()`.
 */
export interface AddBundleOptions extends BundlingOptions {
    /**
     * esbuild target.
     *
     * @example "node12"
     */
    readonly target: string;
    /**
     * esbuild platform.
     *
     * @example "node"
     */
    readonly platform: string;
    /**
     * Bundler output path relative to the asset's output directory.
     * @default "index.js"
     */
    readonly outfile?: string;
    /**
     * Mark the output file as executable.
     * @default false
     */
    readonly executable?: boolean;
}
