"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaImport = exports.generateJavaOptionNames = exports.Projenrc = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const common_1 = require("../common");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
/**
 * Allows writing projenrc files in java.
 *
 * This will install `org.projen/projen` as a Maven dependency and will add a
 * `synth` task which will compile & execute `main()` from
 * `src/main/java/projenrc.java`.
 */
class Projenrc extends component_1.Component {
    constructor(project, pom, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.className = options.className ?? "projenrc";
        this.testScope = options.testScope ?? true;
        const depType = this.testScope
            ? dependencies_1.DependencyType.TEST
            : dependencies_1.DependencyType.RUNTIME;
        const execOpts = this.testScope ? ' -Dexec.classpathScope="test"' : "";
        const compileGoal = this.testScope
            ? "compiler:testCompile"
            : "compiler:compile";
        project.deps.addDependency(`io.github.cdklabs/projen@${projenVersion}`, depType);
        pom.addPlugin("org.codehaus.mojo/exec-maven-plugin@3.0.0");
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`mvn ${compileGoal} --quiet`);
        project.defaultTask?.exec(`mvn exec:java --quiet -Dexec.mainClass=${this.className}${execOpts}`);
        // if this is a new project, generate a skeleton for projenrc.java
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = inventory_1.readJsiiManifest(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const javaTarget = jsiiManifest.targets.java;
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const jsiiOptionsType = jsiiManifest.types[optionsTypeFqn];
        if (!jsiiOptionsType) {
            this.project.logger.warn(`cannot find jsii type for project options: ${optionsTypeFqn}`);
            return;
        }
        const dir = this.testScope ? "src/test/java" : "src/main/java";
        const split = this.className.split(".");
        let javaClass, javaPackage;
        if (split.length === 1) {
            javaClass = split[0];
            javaPackage = [];
        }
        else {
            javaPackage = split.slice(0, split.length - 2);
            javaClass = split[split.length - 1];
        }
        const javaFile = path_1.join(this.project.outdir, dir, ...javaPackage, javaClass + ".java");
        // skip if file exists
        if (fs_extra_1.existsSync(javaFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        const openBlock = (line = "") => {
            emit(line + " {");
            indent++;
        };
        const closeBlock = () => {
            indent--;
            emit("}");
        };
        const optionFqns = generateJavaOptionNames(bootstrap.type.options, jsiiManifest);
        if (javaPackage.length > 0) {
            emit(`package ${javaPackage.join(".")};`);
            emit();
        }
        const { renderedOptions, imports } = renderJavaOptions(2, jsiiOptionsType.name, optionFqns, bootstrap.args);
        emit(`import ${getJavaImport(jsiiType, jsiiManifest)};`);
        emit(`import ${getJavaImport(jsiiOptionsType, jsiiManifest)};`);
        for (const optionTypeName of imports) {
            emit(`import ${javaTarget.package}.${optionTypeName};`);
        }
        emit();
        openBlock(`public class ${javaClass}`);
        openBlock("public static void main(String[] args)");
        emit(`${jsiiType.name} project = new ${jsiiType.name}(${renderedOptions});`);
        emit("project.synth();");
        closeBlock();
        closeBlock();
        fs_extra_1.mkdirpSync(path_1.dirname(javaFile));
        fs_extra_1.writeFileSync(javaFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${javaFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.java.Projenrc", version: "0.60.2" };
function generateJavaOptionNames(options, jsiiManifest) {
    const optionFqns = {};
    for (const option of options) {
        if (option.fqn && jsiiManifest.types[option.fqn]) {
            optionFqns[option.name] = toJavaFullTypeName(jsiiManifest.types[option.fqn]);
        }
    }
    return optionFqns;
}
exports.generateJavaOptionNames = generateJavaOptionNames;
function renderJavaOptions(indent, optionsTypeName, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        return { renderedOptions: "", imports }; // no options
    }
    const lines = [`${optionsTypeName}.builder()`];
    for (const [name, value] of Object.entries(initOptions)) {
        const { javaValue, importName } = toJavaValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`.${toJavaProperty(name)}(${javaValue})`);
    }
    lines.push(".build()");
    const renderedOptions = lines.join(`\n${" ".repeat((indent + 1) * 4)}`);
    return { renderedOptions, imports };
}
function toJavaProperty(prop) {
    return prop;
}
function toJavaValue(value, name, optionFqns) {
    if (typeof value === "string" && optionFqns[name] !== undefined) {
        const parts = optionFqns[name].split(".");
        const base = parts[parts.length - 1];
        const choice = String(value).toUpperCase().replace(/-/g, "_");
        return { javaValue: `${base}.${choice}`, importName: optionFqns[name] };
    }
    else {
        return { javaValue: JSON.stringify(value) };
    }
}
function toJavaFullTypeName(jsiiType) {
    return [jsiiType.namespace, jsiiType.name].filter((x) => x).join(".");
}
function getJavaImport(jsiiType, jsiiManifest) {
    const packageName = jsiiManifest?.submodules?.[`${jsiiType.assembly}.${jsiiType?.namespace}`]
        ?.targets?.java?.package ||
        [jsiiManifest.targets.java.package, jsiiType.namespace]
            .filter((x) => x)
            .join(".");
    return `${packageName}.${jsiiType.name}`;
}
exports.getJavaImport = getJavaImport;
//# sourceMappingURL=data:application/json;base64,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