"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.project = project;
        this.name = path.parse(name).name;
        this.path =
            this.name === "gitlab-ci"
                ? ".gitlab-ci.yml"
                : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            this.defaultCache = defaults.cache;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addJobs(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
        }
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            ...snakeCaseKeys(this.jobs),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.60.2" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null && k !== "variables") {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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