"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    constructor(github, name, options = {}) {
        super(github.project);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        this.projenCredentials = github.projenCredentials;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        // verify that job has a "permissions" statement to ensure workflow can
        // operate in repos with default tokens set to readonly
        for (const [id, job] of Object.entries(jobs)) {
            if (!job.permissions) {
                throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
            }
        }
        // verify that job has a "runsOn" statement to ensure a worker can be selected appropriately
        for (const [id, job] of Object.entries(jobs)) {
            if (!("uses" in job)) {
                if ("runsOn" in job && job.runsOn.length === 0) {
                    throw new Error(`${id}: at least one runner selector labels must be provided in "runsOn" to ensure a runner instance can be selected`);
                }
            }
        }
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.60.2" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
function renderJobs(jobs) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: util_1.kebabCaseKeys(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = util_1.kebabCaseKeys(_resolve_1.resolve(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOn),
            permissions: util_1.kebabCaseKeys(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: util_1.kebabCaseKeys(job.defaults),
            if: job.if,
            steps: steps,
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
}
function arrayOrScalar(arr) {
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v3",
            with: { distribution: "temurin", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v3",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v3",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v3",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v2",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
//# sourceMappingURL=data:application/json;base64,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