"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
class GithubCredentials {
    constructor(options) {
        this.options = options;
    }
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options = {}) {
        return new GithubCredentials({
            setupSteps: [],
            tokenRef: `\${{ secrets.${options.secret ?? "PROJEN_GITHUB_TOKEN"} }}`,
        });
    }
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @default - app id stored in "PROJEN_GITHUB_TOKEN" and private key stored in "PROJEN_APP_PRIVATE_KEY"
     */
    static fromApp(options = {}) {
        const appIdSecret = options.appIdSecret ?? "PROJEN_APP_ID";
        const privateKeySecret = options.privateKeySecret ?? "PROJEN_APP_PRIVATE_KEY";
        return new GithubCredentials({
            setupSteps: [
                {
                    name: "Generate token",
                    id: "generate_token",
                    uses: "tibdex/github-app-token@7ce9ffdcdeb2ba82b01b51d6584a6a85872336d4",
                    with: {
                        app_id: `\${{ secrets.${appIdSecret} }}`,
                        private_key: `\${{ secrets.${privateKeySecret} }}`,
                    },
                },
            ],
            tokenRef: "${{ steps.generate_token.outputs.token }}",
        });
    }
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps() {
        return [...this.options.setupSteps];
    }
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef() {
        return this.options.tokenRef;
    }
}
exports.GithubCredentials = GithubCredentials;
_a = JSII_RTTI_SYMBOL_1;
GithubCredentials[_a] = { fqn: "projen.github.GithubCredentials", version: "0.60.2" };
//# sourceMappingURL=data:application/json;base64,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