"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs_extra_1 = require("fs-extra");
const _resolve_1 = require("./_resolve");
const common_1 = require("./common");
const component_1 = require("./component");
const util_1 = require("./util");
class FileBase extends component_1.Component {
    constructor(project, filePath, options = {}) {
        super(project);
        this.readonly = !project.ejected && (options.readonly ?? true);
        this.executable = options.executable ?? false;
        this.path = filePath;
        // `marker` is empty if project is being ejected or if explicitly disabled
        this.marker =
            project.ejected || options.marker === false
                ? undefined
                : `${common_1.PROJEN_MARKER}. To modify, edit ${common_1.PROJEN_RC} and run "npx projen".`;
        const globPattern = `/${this.path}`;
        const committed = options.committed ?? true;
        if (committed && filePath !== ".gitattributes") {
            project.root.annotateGenerated(`/${filePath}`);
        }
        this.absolutePath = path.resolve(project.outdir, filePath);
        // verify file path is unique within project tree
        const existing = project.root.tryFindFile(this.absolutePath);
        if (existing && existing !== this) {
            throw new Error(`there is already a file under ${path.relative(project.root.outdir, this.absolutePath)}`);
        }
        const editGitignore = options.editGitignore ?? true;
        if (editGitignore) {
            this.project.addGitIgnore(`${committed ? "!" : ""}${globPattern}`);
        }
        else {
            if (options.committed != null) {
                throw new Error('"gitignore" is disabled, so it does not make sense to specify "committed"');
            }
        }
    }
    /**
     * Writes the file to the project's output directory
     */
    synthesize() {
        const outdir = this.project.outdir;
        const filePath = path.join(outdir, this.path);
        const resolver = {
            resolve: (obj, options) => _resolve_1.resolve(obj, options),
        };
        const content = this.synthesizeContent(resolver);
        if (content === undefined) {
            // remove file (if exists) and skip rest of synthesis
            fs_extra_1.removeSync(filePath);
            return;
        }
        // check if the file was changed.
        const prev = util_1.tryReadFileSync(filePath);
        const prevReadonly = !util_1.isWritable(filePath);
        const prevExecutable = util_1.isExecutable(filePath);
        if (prev !== undefined &&
            content === prev &&
            prevReadonly === this.readonly &&
            prevExecutable === this.executable) {
            this.project.logger.debug(`no change in ${filePath}`);
            this._changed = false;
            return;
        }
        util_1.writeFile(filePath, content, {
            readonly: this.readonly,
            executable: this.executable,
        });
        this.checkForProjenMarker();
        this._changed = true;
    }
    /**
     * For debugging, check whether a file was incorrectly generated with
     * or without the projen marker. The projen marker does not *need* to be
     * included on projen-generated files, but it's recommended since it signals
     * that it probably should not be edited directly.
     */
    checkForProjenMarker() {
        const filePath = path.join(this.project.outdir, this.path);
        const contents = util_1.tryReadFileSync(filePath);
        const containsMarker = contents?.includes(common_1.PROJEN_MARKER);
        if (this.marker && !containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to contain marker but found none.`);
        }
        else if (!this.marker && containsMarker) {
            this.project.logger.debug(`note: expected ${this.path} to not contain marker but found one anyway.`);
        }
    }
    /**
     * Indicates if the file has been changed during synthesis. This property is
     * only available in `postSynthesize()` hooks. If this is `undefined`, the
     * file has not been synthesized yet.
     */
    get changed() {
        return this._changed;
    }
}
exports.FileBase = FileBase;
_a = JSII_RTTI_SYMBOL_1;
FileBase[_a] = { fqn: "projen.FileBase", version: "0.60.2" };
//# sourceMappingURL=data:application/json;base64,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