"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = void 0;
const chalk = require("chalk");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    const tasks = runtime.manifest.tasks ?? {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, task.description ?? "", taskCommandHandler(task.name));
    }
    function taskCommandHandler(taskName) {
        return (args) => {
            args.option("inspect", {
                alias: "i",
                desc: "show all steps in this task",
            });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(taskName);
            }
            else {
                try {
                    runtime.runTask(taskName);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        const writeln = (s) => console.log(" ".repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline("description")}: ${task.description}`);
        }
        for (const [k, v] of Object.entries(task.env ?? {})) {
            writeln(`${chalk.underline("env")}: ${k}=${v}`);
        }
        for (const step of task.steps ?? []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
exports.discoverTaskCommands = discoverTaskCommands;
//# sourceMappingURL=data:application/json;base64,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