"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability = exports.Stability || (exports.Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
            jestOptions: {
                jestVersion: "^27",
            },
        };
        const forcedOptions = {
            releaseToNpm: false,
            disableTsconfig: true,
            docgen: false,
        };
        const mergedOptions = util_1.deepMerge([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"].join(" ");
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(`jsii ${jsiiFlags}`);
        this.watchTask.reset(`jsii -w ${jsiiFlags}`);
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        const prepareRepoForCI = [
            `rsync -a . .repo --exclude .git --exclude node_modules`,
            `rm -rf ${this.artifactsDirectory}`,
            `mv .repo ${this.artifactsDirectory}`,
        ].join(" && ");
        // when running inside CI we just prepare the repo for packaging, which
        // takes place in separate tasks.
        // outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset(`if [ ! -z \${CI} ]; then ${prepareRepoForCI}; else ${this.runTaskCommand(this.packageAllTask)}; fi`);
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        this.release?.publisher.addGitHubPrePublishingSteps({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Collect GitHub Metadata",
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        if (options.releaseToNpm != false) {
            const task = this.addPackagingTask("js");
            this.release?.publisher.publishToNpm({
                ...this.pacmakForLanguage("js", task),
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                codeArtifactOptions: options.codeArtifactOptions,
            });
            this.addPackagingTarget("js", task);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        if (options.publishToMaven) {
            targets.java = {
                package: options.publishToMaven.javaPackage,
                maven: {
                    groupId: options.publishToMaven.mavenGroupId,
                    artifactId: options.publishToMaven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.release?.publisher.publishToMaven({
                ...this.pacmakForLanguage("java", task),
                ...options.publishToMaven,
            });
            this.addPackagingTarget("java", task);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.release?.publisher.publishToPyPi({
                ...this.pacmakForLanguage("python", task),
                ...pypi,
            });
            this.addPackagingTarget("python", task);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
            };
            const task = this.addPackagingTask("dotnet");
            this.release?.publisher.publishToNuget({
                ...this.pacmakForLanguage("dotnet", task),
                ...nuget,
            });
            this.addPackagingTarget("dotnet", task);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
            };
            const task = this.addPackagingTask("go");
            this.release?.publisher.publishToGo({
                ...this.pacmakForLanguage("go", task),
                ...golang,
            });
            this.addPackagingTarget("go", task);
        }
        this.addDevDeps("jsii", "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            new jsii_docgen_1.JsiiDocgen(this, { filePath: options.docgenFilePath });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
        // jsii relies on typescript < 4.0, which causes build errors
        // since @types/prettier@2.6.1 only supports typescript >= 4.2.
        // add a package resolution override to fix this.
        // this should have no effect if @types/prettier is not a transitive dependency
        this.package.addPackageResolutions("@types/prettier@2.6.0");
    }
    /**
     * Adds a target language to the build workflow and creates a package task.
     * @param language
     * @returns
     */
    addPackagingTarget(language, packTask) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            runsOn: ["ubuntu-latest"],
            permissions: {},
            tools: {
                node: { version: "14.x" },
                ...pacmak.publishTools,
            },
            steps: pacmak.prePublishSteps ?? [],
        });
    }
    addPackagingTask(language) {
        const packageTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        packageTask.exec(`jsii-pacmak -v --target ${language}`);
        this.packageAllTask.spawn(packageTask);
        return packageTask;
    }
    pacmakForLanguage(target, packTask) {
        // at this stage, `artifactsDirectory` contains the prebuilt repository.
        // for the publishing to work seamlessely, that directory needs to contain the actual artifact.
        // so we move the repo, create the artifact, and put it in the expected place.
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            prePublishSteps: [
                {
                    name: "Prepare Repository",
                    run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
                },
                {
                    name: "Install Dependencies",
                    run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
                },
                {
                    name: `Create ${target} artifact`,
                    run: `cd ${REPO_TEMP_DIRECTORY} && npx projen ${packTask.name}`,
                },
                {
                    name: `Collect ${target} Artifact`,
                    run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
                },
            ],
        };
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.60.2" };
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoianNpaS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Nkay9qc2lpLXByb2plY3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFTQSw4Q0FBNEU7QUFDNUUsa0NBQW9DO0FBQ3BDLHFDQUE0RDtBQUM1RCwrQ0FBMkM7QUFFM0MsTUFBTSxXQUFXLEdBQ2YsNGFBQTRhLENBQUM7QUFDL2EsTUFBTSxTQUFTLEdBQ2Isc0xBQXNMLENBQUM7QUFDekwsTUFBTSxtQkFBbUIsR0FBRyxPQUFPLENBQUM7QUE2RnBDLElBQVksU0FJWDtBQUpELFdBQVksU0FBUztJQUNuQiwwQ0FBNkIsQ0FBQTtJQUM3Qiw4QkFBaUIsQ0FBQTtJQUNqQixzQ0FBeUIsQ0FBQTtBQUMzQixDQUFDLEVBSlcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUFJcEI7QUErQkQ7Ozs7R0FJRztBQUNILE1BQWEsV0FBWSxTQUFRLDhCQUFpQjtJQUtoRCxZQUFZLE9BQTJCO1FBQ3JDLE1BQU0sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLEdBQUcsa0JBQWtCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFL0QsTUFBTSxjQUFjLEdBQUc7WUFDckIsVUFBVSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQ2pDLFVBQVUsRUFBRSxPQUFPLENBQUMsTUFBTTtZQUMxQixXQUFXO1lBQ1gsU0FBUztZQUNULFdBQVcsRUFBRTtnQkFDWCxXQUFXLEVBQUUsS0FBSzthQUNuQjtTQUNGLENBQUM7UUFFRixNQUFNLGFBQWEsR0FBRztZQUNwQixZQUFZLEVBQUUsS0FBSztZQUNuQixlQUFlLEVBQUUsSUFBSTtZQUNyQixNQUFNLEVBQUUsS0FBSztTQUNkLENBQUM7UUFFRixNQUFNLGFBQWEsR0FBRyxnQkFBUyxDQUFDO1lBQzlCLGNBQWM7WUFDZCxPQUFPO1lBQ1AsYUFBYTtTQUNkLENBQTZCLENBQUM7UUFFL0IsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXJCLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUM7UUFDM0IsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQztRQUUzQixJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsTUFBTSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBRWxELCtCQUErQjtRQUMvQixNQUFNLFNBQVMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWpFLE1BQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksZUFBZSxDQUFDO1FBRTdELElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVMsSUFBSSxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUVyRSxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLFVBQVUsRUFBRTtZQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7U0FDdEM7UUFFRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUN4QyxXQUFXLEVBQUUsd0RBQXdEO1lBQ3JFLElBQUksRUFBRSxnRkFBZ0YsWUFBWSxnR0FBZ0csWUFBWSx5QkFBeUI7U0FDeE8sQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUM7UUFDdkMsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUNwQztRQUVELElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLFFBQVEsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxXQUFXLFNBQVMsRUFBRSxDQUFDLENBQUM7UUFDN0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUNoRCxXQUFXLEVBQUUsNkNBQTZDO1NBQzNELENBQUMsQ0FBQztRQUVILHlFQUF5RTtRQUN6RSx1RUFBdUU7UUFDdkUsTUFBTSxnQkFBZ0IsR0FBRztZQUN2Qix3REFBd0Q7WUFDeEQsVUFBVSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDbkMsWUFBWSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7U0FDdEMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFZix1RUFBdUU7UUFDdkUsaUNBQWlDO1FBQ2pDLDZEQUE2RDtRQUM3RCxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FDcEIsNEJBQTRCLGdCQUFnQixVQUFVLElBQUksQ0FBQyxjQUFjLENBQ3ZFLElBQUksQ0FBQyxjQUFjLENBQ3BCLE1BQU0sQ0FDUixDQUFDO1FBRUYsTUFBTSxPQUFPLEdBQXdCLEVBQUUsQ0FBQztRQUV4QyxNQUFNLElBQUksR0FBUTtZQUNoQixNQUFNLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtZQUMvQixPQUFPO1lBQ1AsR0FBRyxFQUFFO2dCQUNILE1BQU0sRUFBRSxNQUFNO2dCQUNkLE9BQU8sRUFBRSxNQUFNO2FBQ2hCO1NBQ0YsQ0FBQztRQUVGLElBQUksT0FBTyxDQUFDLGlCQUFpQixFQUFFO1lBQzdCLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxPQUFPLENBQUMsaUJBQWlCLENBQUM7U0FDcEQ7UUFFRCxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUV6QixJQUFJLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQywyQkFBMkIsQ0FDakQ7WUFDRSxJQUFJLEVBQUUsb0JBQW9CO1lBQzFCLEdBQUcsRUFBRSxNQUFNLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxtQkFBbUIsRUFBRTtTQUM1RCxFQUNEO1lBQ0UsSUFBSSxFQUFFLHlCQUF5QjtZQUMvQixHQUFHLEVBQUUsTUFBTSxtQkFBbUIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1NBQ3ZGLENBQ0YsQ0FBQztRQUVGLElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxLQUFLLEVBQUU7WUFDakMsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLFlBQVksQ0FBQztnQkFDbkMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQztnQkFDckMsUUFBUSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDbEMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYztnQkFDM0MsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLG1CQUFtQjthQUNqRCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ3JDO1FBRUQsMEVBQTBFO1FBQzFFLDZFQUE2RTtRQUM3RSx5REFBeUQ7UUFDekQsSUFBSSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUVELElBQUksT0FBTyxDQUFDLGNBQWMsRUFBRTtZQUMxQixPQUFPLENBQUMsSUFBSSxHQUFHO2dCQUNiLE9BQU8sRUFBRSxPQUFPLENBQUMsY0FBYyxDQUFDLFdBQVc7Z0JBQzNDLEtBQUssRUFBRTtvQkFDTCxPQUFPLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxZQUFZO29CQUM1QyxVQUFVLEVBQUUsT0FBTyxDQUFDLGNBQWMsQ0FBQyxlQUFlO2lCQUNuRDthQUNGLENBQUM7WUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUM7WUFFM0MsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDO2dCQUNyQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDO2dCQUN2QyxHQUFHLE9BQU8sQ0FBQyxjQUFjO2FBQzFCLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDdkM7UUFFRCxNQUFNLElBQUksR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDckQsSUFBSSxJQUFJLEVBQUU7WUFDUixPQUFPLENBQUMsTUFBTSxHQUFHO2dCQUNmLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtnQkFDdkIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2FBQ3BCLENBQUM7WUFFRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDN0MsSUFBSSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsYUFBYSxDQUFDO2dCQUNwQyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDO2dCQUN6QyxHQUFHLElBQUk7YUFDUixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsa0JBQWtCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO1NBQ3pDO1FBRUQsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLGNBQWMsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDO1FBQ3ZELElBQUksS0FBSyxFQUFFO1lBQ1QsT0FBTyxDQUFDLE1BQU0sR0FBRztnQkFDZixTQUFTLEVBQUUsS0FBSyxDQUFDLGVBQWU7Z0JBQ2hDLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUzthQUMzQixDQUFDO1lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzdDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLGNBQWMsQ0FBQztnQkFDckMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQztnQkFDekMsR0FBRyxLQUFLO2FBQ1QsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztTQUN6QztRQUVELE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDbkMsSUFBSSxNQUFNLEVBQUU7WUFDVixPQUFPLENBQUMsRUFBRSxHQUFHO2dCQUNYLFVBQVUsRUFBRSxNQUFNLENBQUMsVUFBVTthQUM5QixDQUFDO1lBRUYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxPQUFPLEVBQUUsU0FBUyxDQUFDLFdBQVcsQ0FBQztnQkFDbEMsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQztnQkFDckMsR0FBRyxNQUFNO2FBQ1YsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQztTQUNyQztRQUVELElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVwRCxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFakMsSUFBSSxPQUFPLENBQUMsTUFBTSxJQUFJLElBQUksRUFBRTtZQUMxQixJQUFJLHdCQUFVLENBQUMsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLE9BQU8sQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO1NBQzVEO1FBRUQsa0RBQWtEO1FBQ2xELElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUM7U0FDakM7UUFFRCw2REFBNkQ7UUFDN0QsK0RBQStEO1FBQy9ELGlEQUFpRDtRQUNqRCwrRUFBK0U7UUFDL0UsSUFBSSxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssa0JBQWtCLENBQUMsUUFBMEIsRUFBRSxRQUFjO1FBQ25FLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE9BQU87U0FDUjtRQUVELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFMUQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsV0FBVyxRQUFRLEVBQUUsRUFBRTtZQUN4RCxNQUFNLEVBQUUsQ0FBQyxlQUFlLENBQUM7WUFDekIsV0FBVyxFQUFFLEVBQUU7WUFDZixLQUFLLEVBQUU7Z0JBQ0wsSUFBSSxFQUFFLEVBQUUsT0FBTyxFQUFFLE1BQU0sRUFBRTtnQkFDekIsR0FBRyxNQUFNLENBQUMsWUFBWTthQUN2QjtZQUNELEtBQUssRUFBRSxNQUFNLENBQUMsZUFBZSxJQUFJLEVBQUU7U0FDcEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGdCQUFnQixDQUFDLFFBQTBCO1FBQ2pELE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsUUFBUSxFQUFFLEVBQUU7WUFDNUQsV0FBVyxFQUFFLFVBQVUsUUFBUSxvQkFBb0I7U0FDcEQsQ0FBQyxDQUFDO1FBQ0gsV0FBVyxDQUFDLElBQUksQ0FBQywyQkFBMkIsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN2QyxPQUFPLFdBQVcsQ0FBQztJQUNyQixDQUFDO0lBRU8saUJBQWlCLENBQ3ZCLE1BQXdCLEVBQ3hCLFFBQWM7UUFFZCx3RUFBd0U7UUFDeEUsK0ZBQStGO1FBQy9GLDhFQUE4RTtRQUU5RSxPQUFPO1lBQ0wsWUFBWSxFQUFFLHVCQUFjLENBQUMsTUFBTSxDQUFDO1lBQ3BDLGVBQWUsRUFBRTtnQkFDZjtvQkFDRSxJQUFJLEVBQUUsb0JBQW9CO29CQUMxQixHQUFHLEVBQUUsTUFBTSxJQUFJLENBQUMsa0JBQWtCLElBQUksbUJBQW1CLEVBQUU7aUJBQzVEO2dCQUNEO29CQUNFLElBQUksRUFBRSxzQkFBc0I7b0JBQzVCLEdBQUcsRUFBRSxNQUFNLG1CQUFtQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFO2lCQUNuRTtnQkFDRDtvQkFDRSxJQUFJLEVBQUUsVUFBVSxNQUFNLFdBQVc7b0JBQ2pDLEdBQUcsRUFBRSxNQUFNLG1CQUFtQixrQkFBa0IsUUFBUSxDQUFDLElBQUksRUFBRTtpQkFDaEU7Z0JBQ0Q7b0JBQ0UsSUFBSSxFQUFFLFdBQVcsTUFBTSxXQUFXO29CQUNsQyxHQUFHLEVBQUUsTUFBTSxtQkFBbUIsSUFBSSxJQUFJLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO2lCQUN2RjthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7O0FBblJILGtDQW9SQzs7O0FBRUQsU0FBUyxrQkFBa0IsQ0FBQyxPQUEyQjtJQUNyRCxJQUFJLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQ3RDLElBQUksU0FBUyxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDbEMsSUFBSSxPQUFPLENBQUMsYUFBYSxFQUFFO1FBQ3pCLElBQUksT0FBTyxDQUFDLFdBQVcsSUFBSSxPQUFPLENBQUMsV0FBVyxLQUFLLE9BQU8sQ0FBQyxhQUFhLEVBQUU7WUFDeEUsTUFBTSxJQUFJLEtBQUssQ0FDYixnRkFBZ0YsQ0FDakYsQ0FBQztTQUNIO1FBRUQsSUFBSSxPQUFPLENBQUMsU0FBUyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUNwRSxNQUFNLElBQUksS0FBSyxDQUNiLCtFQUErRSxDQUNoRixDQUFDO1NBQ0g7UUFFRCxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQzNDLFdBQVcsR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDO1NBQ3JDO2FBQU0sSUFBSSxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNoRCxTQUFTLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQztTQUNuQzthQUFNO1lBQ0wsTUFBTSxJQUFJLEtBQUssQ0FDYiwyREFBMkQsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUNuRixDQUFDO1NBQ0g7S0FDRjtJQUNELE9BQU8sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLENBQUM7QUFDcEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFRhc2sgfSBmcm9tIFwiLi5cIjtcbmltcG9ydCB7IEVzbGludCB9IGZyb20gXCIuLi9qYXZhc2NyaXB0XCI7XG5pbXBvcnQge1xuICBDb21tb25QdWJsaXNoT3B0aW9ucyxcbiAgR29QdWJsaXNoT3B0aW9ucyxcbiAgTWF2ZW5QdWJsaXNoT3B0aW9ucyxcbiAgTnVnZXRQdWJsaXNoT3B0aW9ucyxcbiAgUHlQaVB1Ymxpc2hPcHRpb25zLFxufSBmcm9tIFwiLi4vcmVsZWFzZVwiO1xuaW1wb3J0IHsgVHlwZVNjcmlwdFByb2plY3QsIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyB9IGZyb20gXCIuLi90eXBlc2NyaXB0XCI7XG5pbXBvcnQgeyBkZWVwTWVyZ2UgfSBmcm9tIFwiLi4vdXRpbFwiO1xuaW1wb3J0IHsgSnNpaVBhY21ha1RhcmdldCwgSlNJSV9UT09MQ0hBSU4gfSBmcm9tIFwiLi9jb25zdHNcIjtcbmltcG9ydCB7IEpzaWlEb2NnZW4gfSBmcm9tIFwiLi9qc2lpLWRvY2dlblwiO1xuXG5jb25zdCBFTUFJTF9SRUdFWCA9XG4gIC8oPzpbYS16MC05ISMkJSYnKisvPT9eX2B7fH1+LV0rKD86XFwuW2EtejAtOSEjJCUmJyorLz0/Xl9ge3x9fi1dKykqfFwiKD86W1xceDAxLVxceDA4XFx4MGJcXHgwY1xceDBlLVxceDFmXFx4MjFcXHgyMy1cXHg1YlxceDVkLVxceDdmXXxcXFxcW1xceDAxLVxceDA5XFx4MGJcXHgwY1xceDBlLVxceDdmXSkqXCIpQCg/Oig/OlthLXowLTldKD86W2EtejAtOS1dKlthLXowLTldKT9cXC4pK1thLXowLTldKD86W2EtejAtOS1dKlthLXowLTldKT98XFxbKD86KD86MjVbMC01XXwyWzAtNF1bMC05XXxbMDFdP1swLTldWzAtOV0/KVxcLil7M30oPzoyNVswLTVdfDJbMC00XVswLTldfFswMV0/WzAtOV1bMC05XT98W2EtejAtOS1dKlthLXowLTldOig/OltcXHgwMS1cXHgwOFxceDBiXFx4MGNcXHgwZS1cXHgxZlxceDIxLVxceDVhXFx4NTMtXFx4N2ZdfFxcXFxbXFx4MDEtXFx4MDlcXHgwYlxceDBjXFx4MGUtXFx4N2ZdKSspXFxdKS87XG5jb25zdCBVUkxfUkVHRVggPVxuICAvKCgoW0EtWmEtel17Myw5fTooPzpcXC9cXC8pPykoPzpbXFwtOzomPVxcK1xcJCxcXHddK0ApP1tBLVphLXowLTlcXC5cXC1dK3woPzp3d3dcXC58W1xcLTs6Jj1cXCtcXCQsXFx3XStAKVtBLVphLXowLTlcXC5cXC1dKykoKD86XFwvW1xcK34lXFwvXFwuXFx3XFwtX10qKT9cXD8/KD86W1xcLVxcKz0mOyVAXFwuXFx3X10qKSM/KD86W1xcLlxcIVxcL1xcXFxcXHddKikpPykvO1xuY29uc3QgUkVQT19URU1QX0RJUkVDVE9SWSA9IFwiLnJlcG9cIjtcblxuZXhwb3J0IGludGVyZmFjZSBKc2lpUHJvamVjdE9wdGlvbnMgZXh0ZW5kcyBUeXBlU2NyaXB0UHJvamVjdE9wdGlvbnMge1xuICAvKipcbiAgICogQGRlZmF1bHQgXCIuXCJcbiAgICovXG4gIHJlYWRvbmx5IHJvb3RkaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEdpdCByZXBvc2l0b3J5IFVSTC5cbiAgICogQGRlZmF1bHQgJEdJVF9SRU1PVEVcbiAgICovXG4gIHJlYWRvbmx5IHJlcG9zaXRvcnlVcmw6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGxpYnJhcnkgYXV0aG9yLlxuICAgKiBAZGVmYXVsdCAkR0lUX1VTRVJfTkFNRVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVtYWlsIG9yIFVSTCBvZiB0aGUgbGlicmFyeSBhdXRob3IuXG4gICAqIEBkZWZhdWx0ICRHSVRfVVNFUl9FTUFJTFxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yQWRkcmVzczogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQdWJsaXNoIHRvIG1hdmVuXG4gICAqIEBkZWZhdWx0IC0gbm8gcHVibGlzaGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaFRvTWF2ZW4/OiBKc2lpSmF2YVRhcmdldDtcblxuICAvKipcbiAgICogUHVibGlzaCB0byBweXBpXG4gICAqIEBkZWZhdWx0IC0gbm8gcHVibGlzaGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHVibGlzaFRvUHlwaT86IEpzaWlQeXRob25UYXJnZXQ7XG5cbiAgLyoqXG4gICAqIFB1Ymxpc2ggR28gYmluZGluZ3MgdG8gYSBnaXQgcmVwb3NpdG9yeS5cbiAgICogQGRlZmF1bHQgLSBubyBwdWJsaXNoaW5nXG4gICAqL1xuICByZWFkb25seSBwdWJsaXNoVG9Hbz86IEpzaWlHb1RhcmdldDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwdWJsaXNoVG9QeVBpYFxuICAgKi9cbiAgcmVhZG9ubHkgcHl0aG9uPzogSnNpaVB5dGhvblRhcmdldDtcblxuICAvKipcbiAgICogUHVibGlzaCB0byBOdUdldFxuICAgKiBAZGVmYXVsdCAtIG5vIHB1Ymxpc2hpbmdcbiAgICovXG4gIHJlYWRvbmx5IHB1Ymxpc2hUb051Z2V0PzogSnNpaURvdE5ldFRhcmdldDtcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBwdWJsaXNoVG9OdWdldGBcbiAgICovXG4gIHJlYWRvbmx5IGRvdG5ldD86IEpzaWlEb3ROZXRUYXJnZXQ7XG5cbiAgLyoqXG4gICAqIEF1dG9tYXRpY2FsbHkgcnVuIEFQSSBjb21wYXRpYmlsaXR5IHRlc3QgYWdhaW5zdCB0aGUgbGF0ZXN0IHZlcnNpb24gcHVibGlzaGVkIHRvIG5wbSBhZnRlciBjb21waWxhdGlvbi5cbiAgICpcbiAgICogLSBZb3UgY2FuIG1hbnVhbGx5IHJ1biBjb21wYXRpYmlsaXR5IHRlc3RzIHVzaW5nIGB5YXJuIGNvbXBhdGAgaWYgdGhpcyBmZWF0dXJlIGlzIGRpc2FibGVkLlxuICAgKiAtIFlvdSBjYW4gaWdub3JlIGNvbXBhdGliaWxpdHkgZmFpbHVyZXMgYnkgYWRkaW5nIGxpbmVzIHRvIGEgXCIuY29tcGF0aWdub3JlXCIgZmlsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIGlnbm9yZSBmaWxlIGZvciBBUEkgY29tcGF0aWJpbGl0eSB0ZXN0cy5cbiAgICpcbiAgICogQGRlZmF1bHQgXCIuY29tcGF0aWdub3JlXCJcbiAgICovXG4gIHJlYWRvbmx5IGNvbXBhdElnbm9yZT86IHN0cmluZztcblxuICAvKipcbiAgICogQWNjZXB0cyBhIGxpc3Qgb2YgZ2xvYiBwYXR0ZXJucy4gRmlsZXMgbWF0Y2hpbmcgYW55IG9mIHRob3NlIHBhdHRlcm5zIHdpbGwgYmUgZXhjbHVkZWQgZnJvbSB0aGUgVHlwZVNjcmlwdCBjb21waWxlciBpbnB1dC5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwganNpaSB3aWxsIGluY2x1ZGUgYWxsICoudHMgZmlsZXMgKGV4Y2VwdCAuZC50cyBmaWxlcykgaW4gdGhlIFR5cGVTY3JpcHQgY29tcGlsZXIgaW5wdXQuXG4gICAqIFRoaXMgY2FuIGJlIHByb2JsZW1hdGljIGZvciBleGFtcGxlIHdoZW4gdGhlIHBhY2thZ2UncyBidWlsZCBvciB0ZXN0IHByb2NlZHVyZSBnZW5lcmF0ZXMgLnRzIGZpbGVzXG4gICAqIHRoYXQgY2Fubm90IGJlIGNvbXBpbGVkIHdpdGgganNpaSdzIGNvbXBpbGVyIHNldHRpbmdzLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhjbHVkZVR5cGVzY3JpcHQ/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogRmlsZSBwYXRoIGZvciBnZW5lcmF0ZWQgZG9jcy5cbiAgICogQGRlZmF1bHQgXCJBUEkubWRcIlxuICAgKi9cbiAgcmVhZG9ubHkgZG9jZ2VuRmlsZVBhdGg/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBlbnVtIFN0YWJpbGl0eSB7XG4gIEVYUEVSSU1FTlRBTCA9IFwiZXhwZXJpbWVudGFsXCIsXG4gIFNUQUJMRSA9IFwic3RhYmxlXCIsXG4gIERFUFJFQ0FURUQgPSBcImRlcHJlY2F0ZWRcIixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBKc2lpSmF2YVRhcmdldCBleHRlbmRzIE1hdmVuUHVibGlzaE9wdGlvbnMge1xuICByZWFkb25seSBqYXZhUGFja2FnZTogc3RyaW5nO1xuICByZWFkb25seSBtYXZlbkdyb3VwSWQ6IHN0cmluZztcbiAgcmVhZG9ubHkgbWF2ZW5BcnRpZmFjdElkOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgSnNpaVB5dGhvblRhcmdldCBleHRlbmRzIFB5UGlQdWJsaXNoT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGRpc3ROYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG1vZHVsZTogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEpzaWlEb3ROZXRUYXJnZXQgZXh0ZW5kcyBOdWdldFB1Ymxpc2hPcHRpb25zIHtcbiAgcmVhZG9ubHkgZG90TmV0TmFtZXNwYWNlOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHBhY2thZ2VJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIEdvIHRhcmdldCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSnNpaUdvVGFyZ2V0IGV4dGVuZHMgR29QdWJsaXNoT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgdGFyZ2V0IGdvIG1vZHVsZS5cbiAgICpcbiAgICogQGV4YW1wbGUgZ2l0aHViLmNvbS9vd25lci9yZXBvXG4gICAqIEBleGFtcGxlIGdpdGh1Yi5jb20vb3duZXIvcmVwby9zdWJkaXJcbiAgICovXG4gIHJlYWRvbmx5IG1vZHVsZU5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBNdWx0aS1sYW5ndWFnZSBqc2lpIGxpYnJhcnkgcHJvamVjdFxuICpcbiAqIEBwamlkIGpzaWlcbiAqL1xuZXhwb3J0IGNsYXNzIEpzaWlQcm9qZWN0IGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgZXNsaW50PzogRXNsaW50O1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcGFja2FnZUFsbFRhc2s6IFRhc2s7XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogSnNpaVByb2plY3RPcHRpb25zKSB7XG4gICAgY29uc3QgeyBhdXRob3JFbWFpbCwgYXV0aG9yVXJsIH0gPSBwYXJzZUF1dGhvckFkZHJlc3Mob3B0aW9ucyk7XG5cbiAgICBjb25zdCBkZWZhdWx0T3B0aW9ucyA9IHtcbiAgICAgIHJlcG9zaXRvcnk6IG9wdGlvbnMucmVwb3NpdG9yeVVybCxcbiAgICAgIGF1dGhvck5hbWU6IG9wdGlvbnMuYXV0aG9yLFxuICAgICAgYXV0aG9yRW1haWwsXG4gICAgICBhdXRob3JVcmwsXG4gICAgICBqZXN0T3B0aW9uczoge1xuICAgICAgICBqZXN0VmVyc2lvbjogXCJeMjdcIixcbiAgICAgIH0sXG4gICAgfTtcblxuICAgIGNvbnN0IGZvcmNlZE9wdGlvbnMgPSB7XG4gICAgICByZWxlYXNlVG9OcG06IGZhbHNlLCAvLyB3ZSBoYXZlIGEganNpaSByZWxlYXNlIHdvcmtmbG93XG4gICAgICBkaXNhYmxlVHNjb25maWc6IHRydWUsIC8vIGpzaWkgZ2VuZXJhdGVzIGl0cyBvd24gdHNjb25maWcuanNvblxuICAgICAgZG9jZ2VuOiBmYWxzZSwgLy8gd2UgdXNlIGpzaWktZG9jZ2VuIGhlcmUgc28gZGlzYWJsZSB0eXBlc2NyaXB0IGRvY2dlblxuICAgIH07XG5cbiAgICBjb25zdCBtZXJnZWRPcHRpb25zID0gZGVlcE1lcmdlKFtcbiAgICAgIGRlZmF1bHRPcHRpb25zLFxuICAgICAgb3B0aW9ucyxcbiAgICAgIGZvcmNlZE9wdGlvbnMsXG4gICAgXSkgYXMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zO1xuXG4gICAgc3VwZXIobWVyZ2VkT3B0aW9ucyk7XG5cbiAgICBjb25zdCBzcmNkaXIgPSB0aGlzLnNyY2RpcjtcbiAgICBjb25zdCBsaWJkaXIgPSB0aGlzLmxpYmRpcjtcblxuICAgIHRoaXMuYWRkRmllbGRzKHsgdHlwZXM6IGAke2xpYmRpcn0vaW5kZXguZC50c2AgfSk7XG5cbiAgICAvLyB0aGlzIGlzIGFuIHVuaGVscGZ1bCB3YXJuaW5nXG4gICAgY29uc3QganNpaUZsYWdzID0gW1wiLS1zaWxlbmNlLXdhcm5pbmdzPXJlc2VydmVkLXdvcmRcIl0uam9pbihcIiBcIik7XG5cbiAgICBjb25zdCBjb21wYXRJZ25vcmUgPSBvcHRpb25zLmNvbXBhdElnbm9yZSA/PyBcIi5jb21wYXRpZ25vcmVcIjtcblxuICAgIHRoaXMuYWRkRmllbGRzKHsgc3RhYmlsaXR5OiBvcHRpb25zLnN0YWJpbGl0eSA/PyBTdGFiaWxpdHkuU1RBQkxFIH0pO1xuXG4gICAgaWYgKG9wdGlvbnMuc3RhYmlsaXR5ID09PSBTdGFiaWxpdHkuREVQUkVDQVRFRCkge1xuICAgICAgdGhpcy5hZGRGaWVsZHMoeyBkZXByZWNhdGVkOiB0cnVlIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IGNvbXBhdFRhc2sgPSB0aGlzLmFkZFRhc2soXCJjb21wYXRcIiwge1xuICAgICAgZGVzY3JpcHRpb246IFwiUGVyZm9ybSBBUEkgY29tcGF0aWJpbGl0eSBjaGVjayBhZ2FpbnN0IGxhdGVzdCB2ZXJzaW9uXCIsXG4gICAgICBleGVjOiBganNpaS1kaWZmIG5wbTokKG5vZGUgLXAgXCJyZXF1aXJlKFxcJy4vcGFja2FnZS5qc29uXFwnKS5uYW1lXCIpIC1rIC0taWdub3JlLWZpbGUgJHtjb21wYXRJZ25vcmV9IHx8IChlY2hvIFwiXFxuVU5FWFBFQ1RFRCBCUkVBS0lORyBDSEFOR0VTOiBhZGQga2V5cyBzdWNoIGFzIFxcJ3JlbW92ZWQ6Y29uc3RydWN0cy5Ob2RlLm9mXFwnIHRvICR7Y29tcGF0SWdub3JlfSB0byBza2lwLlxcblwiICYmIGV4aXQgMSlgLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY29tcGF0ID0gb3B0aW9ucy5jb21wYXQgPz8gZmFsc2U7XG4gICAgaWYgKGNvbXBhdCkge1xuICAgICAgdGhpcy5jb21waWxlVGFzay5zcGF3bihjb21wYXRUYXNrKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvbXBpbGVUYXNrLnJlc2V0KGBqc2lpICR7anNpaUZsYWdzfWApO1xuICAgIHRoaXMud2F0Y2hUYXNrLnJlc2V0KGBqc2lpIC13ICR7anNpaUZsYWdzfWApO1xuICAgIHRoaXMucGFja2FnZUFsbFRhc2sgPSB0aGlzLmFkZFRhc2soXCJwYWNrYWdlLWFsbFwiLCB7XG4gICAgICBkZXNjcmlwdGlvbjogXCJQYWNrYWdlcyBhcnRpZmFjdHMgZm9yIGFsbCB0YXJnZXQgbGFuZ3VhZ2VzXCIsXG4gICAgfSk7XG5cbiAgICAvLyBpbiBqc2lpIHdlIGNvbnNpZGVyIHRoZSBlbnRpcmUgcmVwbyAocG9zdCBidWlsZCkgYXMgdGhlIGJ1aWxkIGFydGlmYWN0XG4gICAgLy8gd2hpY2ggaXMgdGhlbiB1c2VkIHRvIGNyZWF0ZSB0aGUgbGFuZ3VhZ2UgYmluZGluZ3MgaW4gc2VwYXJhdGUgam9icy5cbiAgICBjb25zdCBwcmVwYXJlUmVwb0ZvckNJID0gW1xuICAgICAgYHJzeW5jIC1hIC4gLnJlcG8gLS1leGNsdWRlIC5naXQgLS1leGNsdWRlIG5vZGVfbW9kdWxlc2AsXG4gICAgICBgcm0gLXJmICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9YCxcbiAgICAgIGBtdiAucmVwbyAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fWAsXG4gICAgXS5qb2luKFwiICYmIFwiKTtcblxuICAgIC8vIHdoZW4gcnVubmluZyBpbnNpZGUgQ0kgd2UganVzdCBwcmVwYXJlIHRoZSByZXBvIGZvciBwYWNrYWdpbmcsIHdoaWNoXG4gICAgLy8gdGFrZXMgcGxhY2UgaW4gc2VwYXJhdGUgdGFza3MuXG4gICAgLy8gb3V0c2lkZSBvZiBDSSAoaS5lIGxvY2FsbHkpIHdlIHNpbXBseSBwYWNrYWdlIGFsbCB0YXJnZXRzLlxuICAgIHRoaXMucGFja2FnZVRhc2sucmVzZXQoXG4gICAgICBgaWYgWyAhIC16IFxcJHtDSX0gXTsgdGhlbiAke3ByZXBhcmVSZXBvRm9yQ0l9OyBlbHNlICR7dGhpcy5ydW5UYXNrQ29tbWFuZChcbiAgICAgICAgdGhpcy5wYWNrYWdlQWxsVGFza1xuICAgICAgKX07IGZpYFxuICAgICk7XG5cbiAgICBjb25zdCB0YXJnZXRzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG5cbiAgICBjb25zdCBqc2lpOiBhbnkgPSB7XG4gICAgICBvdXRkaXI6IHRoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5LFxuICAgICAgdGFyZ2V0cyxcbiAgICAgIHRzYzoge1xuICAgICAgICBvdXREaXI6IGxpYmRpcixcbiAgICAgICAgcm9vdERpcjogc3JjZGlyLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgaWYgKG9wdGlvbnMuZXhjbHVkZVR5cGVzY3JpcHQpIHtcbiAgICAgIGpzaWkuZXhjbHVkZVR5cGVzY3JpcHQgPSBvcHRpb25zLmV4Y2x1ZGVUeXBlc2NyaXB0O1xuICAgIH1cblxuICAgIHRoaXMuYWRkRmllbGRzKHsganNpaSB9KTtcblxuICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLmFkZEdpdEh1YlByZVB1Ymxpc2hpbmdTdGVwcyhcbiAgICAgIHtcbiAgICAgICAgbmFtZTogXCJQcmVwYXJlIFJlcG9zaXRvcnlcIixcbiAgICAgICAgcnVuOiBgbXYgJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX0gJHtSRVBPX1RFTVBfRElSRUNUT1JZfWAsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiBcIkNvbGxlY3QgR2l0SHViIE1ldGFkYXRhXCIsXG4gICAgICAgIHJ1bjogYG12ICR7UkVQT19URU1QX0RJUkVDVE9SWX0vJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX0gJHt0aGlzLmFydGlmYWN0c0RpcmVjdG9yeX1gLFxuICAgICAgfVxuICAgICk7XG5cbiAgICBpZiAob3B0aW9ucy5yZWxlYXNlVG9OcG0gIT0gZmFsc2UpIHtcbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJqc1wiKTtcbiAgICAgIHRoaXMucmVsZWFzZT8ucHVibGlzaGVyLnB1Ymxpc2hUb05wbSh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJqc1wiLCB0YXNrKSxcbiAgICAgICAgcmVnaXN0cnk6IHRoaXMucGFja2FnZS5ucG1SZWdpc3RyeSxcbiAgICAgICAgbnBtVG9rZW5TZWNyZXQ6IHRoaXMucGFja2FnZS5ucG1Ub2tlblNlY3JldCxcbiAgICAgICAgY29kZUFydGlmYWN0T3B0aW9uczogb3B0aW9ucy5jb2RlQXJ0aWZhY3RPcHRpb25zLFxuICAgICAgfSk7XG4gICAgICB0aGlzLmFkZFBhY2thZ2luZ1RhcmdldChcImpzXCIsIHRhc2spO1xuICAgIH1cblxuICAgIC8vIHdlIGNhbm5vdCBjYWxsIGFuIG9wdGlvbiBgamF2YWAgYmVjYXVzZSB0aGUgamF2YSBjb2RlIGdlbmVyYXRlZCBieSBqc2lpXG4gICAgLy8gZG9lcyBub3QgY29tcGlsZSBkdWUgdG8gYSBjb25mbGljdCBiZXR3ZWVuIHRoaXMgb3B0aW9uIG5hbWUgYW5kIHRoZSBgamF2YWBcbiAgICAvLyBwYWNrYWdlIChlLmcuIHdoZW4gYGphdmEudXRpbC5PYmplY3RzYCBpcyByZWZlcmVuY2VkKS5cbiAgICBpZiAoXCJqYXZhXCIgaW4gb3B0aW9ucykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCd0aGUgXCJqYXZhXCIgb3B0aW9uIGlzIG5vdyBjYWxsZWQgXCJwdWJsaXNoVG9NYXZlblwiJyk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucHVibGlzaFRvTWF2ZW4pIHtcbiAgICAgIHRhcmdldHMuamF2YSA9IHtcbiAgICAgICAgcGFja2FnZTogb3B0aW9ucy5wdWJsaXNoVG9NYXZlbi5qYXZhUGFja2FnZSxcbiAgICAgICAgbWF2ZW46IHtcbiAgICAgICAgICBncm91cElkOiBvcHRpb25zLnB1Ymxpc2hUb01hdmVuLm1hdmVuR3JvdXBJZCxcbiAgICAgICAgICBhcnRpZmFjdElkOiBvcHRpb25zLnB1Ymxpc2hUb01hdmVuLm1hdmVuQXJ0aWZhY3RJZCxcbiAgICAgICAgfSxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJqYXZhXCIpO1xuXG4gICAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5wdWJsaXNoVG9NYXZlbih7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJqYXZhXCIsIHRhc2spLFxuICAgICAgICAuLi5vcHRpb25zLnB1Ymxpc2hUb01hdmVuLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwiamF2YVwiLCB0YXNrKTtcbiAgICB9XG5cbiAgICBjb25zdCBweXBpID0gb3B0aW9ucy5wdWJsaXNoVG9QeXBpID8/IG9wdGlvbnMucHl0aG9uO1xuICAgIGlmIChweXBpKSB7XG4gICAgICB0YXJnZXRzLnB5dGhvbiA9IHtcbiAgICAgICAgZGlzdE5hbWU6IHB5cGkuZGlzdE5hbWUsXG4gICAgICAgIG1vZHVsZTogcHlwaS5tb2R1bGUsXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB0YXNrID0gdGhpcy5hZGRQYWNrYWdpbmdUYXNrKFwicHl0aG9uXCIpO1xuICAgICAgdGhpcy5yZWxlYXNlPy5wdWJsaXNoZXIucHVibGlzaFRvUHlQaSh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJweXRob25cIiwgdGFzayksXG4gICAgICAgIC4uLnB5cGksXG4gICAgICB9KTtcblxuICAgICAgdGhpcy5hZGRQYWNrYWdpbmdUYXJnZXQoXCJweXRob25cIiwgdGFzayk7XG4gICAgfVxuXG4gICAgY29uc3QgbnVnZXQgPSBvcHRpb25zLnB1Ymxpc2hUb051Z2V0ID8/IG9wdGlvbnMuZG90bmV0O1xuICAgIGlmIChudWdldCkge1xuICAgICAgdGFyZ2V0cy5kb3RuZXQgPSB7XG4gICAgICAgIG5hbWVzcGFjZTogbnVnZXQuZG90TmV0TmFtZXNwYWNlLFxuICAgICAgICBwYWNrYWdlSWQ6IG51Z2V0LnBhY2thZ2VJZCxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHRhc2sgPSB0aGlzLmFkZFBhY2thZ2luZ1Rhc2soXCJkb3RuZXRcIik7XG4gICAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5wdWJsaXNoVG9OdWdldCh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJkb3RuZXRcIiwgdGFzayksXG4gICAgICAgIC4uLm51Z2V0LFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwiZG90bmV0XCIsIHRhc2spO1xuICAgIH1cblxuICAgIGNvbnN0IGdvbGFuZyA9IG9wdGlvbnMucHVibGlzaFRvR287XG4gICAgaWYgKGdvbGFuZykge1xuICAgICAgdGFyZ2V0cy5nbyA9IHtcbiAgICAgICAgbW9kdWxlTmFtZTogZ29sYW5nLm1vZHVsZU5hbWUsXG4gICAgICB9O1xuXG4gICAgICBjb25zdCB0YXNrID0gdGhpcy5hZGRQYWNrYWdpbmdUYXNrKFwiZ29cIik7XG4gICAgICB0aGlzLnJlbGVhc2U/LnB1Ymxpc2hlci5wdWJsaXNoVG9Hbyh7XG4gICAgICAgIC4uLnRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UoXCJnb1wiLCB0YXNrKSxcbiAgICAgICAgLi4uZ29sYW5nLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMuYWRkUGFja2FnaW5nVGFyZ2V0KFwiZ29cIiwgdGFzayk7XG4gICAgfVxuXG4gICAgdGhpcy5hZGREZXZEZXBzKFwianNpaVwiLCBcImpzaWktZGlmZlwiLCBcImpzaWktcGFjbWFrXCIpO1xuXG4gICAgdGhpcy5naXRpZ25vcmUuZXhjbHVkZShcIi5qc2lpXCIsIFwidHNjb25maWcuanNvblwiKTtcbiAgICB0aGlzLm5wbWlnbm9yZT8uaW5jbHVkZShcIi5qc2lpXCIpO1xuXG4gICAgaWYgKG9wdGlvbnMuZG9jZ2VuID8/IHRydWUpIHtcbiAgICAgIG5ldyBKc2lpRG9jZ2VuKHRoaXMsIHsgZmlsZVBhdGg6IG9wdGlvbnMuZG9jZ2VuRmlsZVBhdGggfSk7XG4gICAgfVxuXG4gICAgLy8ganNpaSB1cGRhdGVzIC5ucG1pZ25vcmUsIHNvIHdlIG1ha2UgaXQgd3JpdGFibGVcbiAgICBpZiAodGhpcy5ucG1pZ25vcmUpIHtcbiAgICAgIHRoaXMubnBtaWdub3JlLnJlYWRvbmx5ID0gZmFsc2U7XG4gICAgfVxuXG4gICAgLy8ganNpaSByZWxpZXMgb24gdHlwZXNjcmlwdCA8IDQuMCwgd2hpY2ggY2F1c2VzIGJ1aWxkIGVycm9yc1xuICAgIC8vIHNpbmNlIEB0eXBlcy9wcmV0dGllckAyLjYuMSBvbmx5IHN1cHBvcnRzIHR5cGVzY3JpcHQgPj0gNC4yLlxuICAgIC8vIGFkZCBhIHBhY2thZ2UgcmVzb2x1dGlvbiBvdmVycmlkZSB0byBmaXggdGhpcy5cbiAgICAvLyB0aGlzIHNob3VsZCBoYXZlIG5vIGVmZmVjdCBpZiBAdHlwZXMvcHJldHRpZXIgaXMgbm90IGEgdHJhbnNpdGl2ZSBkZXBlbmRlbmN5XG4gICAgdGhpcy5wYWNrYWdlLmFkZFBhY2thZ2VSZXNvbHV0aW9ucyhcIkB0eXBlcy9wcmV0dGllckAyLjYuMFwiKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgdGFyZ2V0IGxhbmd1YWdlIHRvIHRoZSBidWlsZCB3b3JrZmxvdyBhbmQgY3JlYXRlcyBhIHBhY2thZ2UgdGFzay5cbiAgICogQHBhcmFtIGxhbmd1YWdlXG4gICAqIEByZXR1cm5zXG4gICAqL1xuICBwcml2YXRlIGFkZFBhY2thZ2luZ1RhcmdldChsYW5ndWFnZTogSnNpaVBhY21ha1RhcmdldCwgcGFja1Rhc2s6IFRhc2spIHtcbiAgICBpZiAoIXRoaXMuYnVpbGRXb3JrZmxvdykge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IHBhY21hayA9IHRoaXMucGFjbWFrRm9yTGFuZ3VhZ2UobGFuZ3VhZ2UsIHBhY2tUYXNrKTtcblxuICAgIHRoaXMuYnVpbGRXb3JrZmxvdy5hZGRQb3N0QnVpbGRKb2IoYHBhY2thZ2UtJHtsYW5ndWFnZX1gLCB7XG4gICAgICBydW5zT246IFtcInVidW50dS1sYXRlc3RcIl0sXG4gICAgICBwZXJtaXNzaW9uczoge30sXG4gICAgICB0b29sczoge1xuICAgICAgICBub2RlOiB7IHZlcnNpb246IFwiMTQueFwiIH0sXG4gICAgICAgIC4uLnBhY21hay5wdWJsaXNoVG9vbHMsXG4gICAgICB9LFxuICAgICAgc3RlcHM6IHBhY21hay5wcmVQdWJsaXNoU3RlcHMgPz8gW10sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGFkZFBhY2thZ2luZ1Rhc2sobGFuZ3VhZ2U6IEpzaWlQYWNtYWtUYXJnZXQpOiBUYXNrIHtcbiAgICBjb25zdCBwYWNrYWdlVGFzayA9IHRoaXMudGFza3MuYWRkVGFzayhgcGFja2FnZToke2xhbmd1YWdlfWAsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBgQ3JlYXRlICR7bGFuZ3VhZ2V9IGxhbmd1YWdlIGJpbmRpbmdzYCxcbiAgICB9KTtcbiAgICBwYWNrYWdlVGFzay5leGVjKGBqc2lpLXBhY21hayAtdiAtLXRhcmdldCAke2xhbmd1YWdlfWApO1xuICAgIHRoaXMucGFja2FnZUFsbFRhc2suc3Bhd24ocGFja2FnZVRhc2spO1xuICAgIHJldHVybiBwYWNrYWdlVGFzaztcbiAgfVxuXG4gIHByaXZhdGUgcGFjbWFrRm9yTGFuZ3VhZ2UoXG4gICAgdGFyZ2V0OiBKc2lpUGFjbWFrVGFyZ2V0LFxuICAgIHBhY2tUYXNrOiBUYXNrXG4gICk6IENvbW1vblB1Ymxpc2hPcHRpb25zIHtcbiAgICAvLyBhdCB0aGlzIHN0YWdlLCBgYXJ0aWZhY3RzRGlyZWN0b3J5YCBjb250YWlucyB0aGUgcHJlYnVpbHQgcmVwb3NpdG9yeS5cbiAgICAvLyBmb3IgdGhlIHB1Ymxpc2hpbmcgdG8gd29yayBzZWFtbGVzc2VseSwgdGhhdCBkaXJlY3RvcnkgbmVlZHMgdG8gY29udGFpbiB0aGUgYWN0dWFsIGFydGlmYWN0LlxuICAgIC8vIHNvIHdlIG1vdmUgdGhlIHJlcG8sIGNyZWF0ZSB0aGUgYXJ0aWZhY3QsIGFuZCBwdXQgaXQgaW4gdGhlIGV4cGVjdGVkIHBsYWNlLlxuXG4gICAgcmV0dXJuIHtcbiAgICAgIHB1Ymxpc2hUb29sczogSlNJSV9UT09MQ0hBSU5bdGFyZ2V0XSxcbiAgICAgIHByZVB1Ymxpc2hTdGVwczogW1xuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogXCJQcmVwYXJlIFJlcG9zaXRvcnlcIixcbiAgICAgICAgICBydW46IGBtdiAke3RoaXMuYXJ0aWZhY3RzRGlyZWN0b3J5fSAke1JFUE9fVEVNUF9ESVJFQ1RPUll9YCxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6IFwiSW5zdGFsbCBEZXBlbmRlbmNpZXNcIixcbiAgICAgICAgICBydW46IGBjZCAke1JFUE9fVEVNUF9ESVJFQ1RPUll9ICYmICR7dGhpcy5wYWNrYWdlLmluc3RhbGxDb21tYW5kfWAsXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBuYW1lOiBgQ3JlYXRlICR7dGFyZ2V0fSBhcnRpZmFjdGAsXG4gICAgICAgICAgcnVuOiBgY2QgJHtSRVBPX1RFTVBfRElSRUNUT1JZfSAmJiBucHggcHJvamVuICR7cGFja1Rhc2submFtZX1gLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgbmFtZTogYENvbGxlY3QgJHt0YXJnZXR9IEFydGlmYWN0YCxcbiAgICAgICAgICBydW46IGBtdiAke1JFUE9fVEVNUF9ESVJFQ1RPUll9LyR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9ICR7dGhpcy5hcnRpZmFjdHNEaXJlY3Rvcnl9YCxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfTtcbiAgfVxufVxuXG5mdW5jdGlvbiBwYXJzZUF1dGhvckFkZHJlc3Mob3B0aW9uczogSnNpaVByb2plY3RPcHRpb25zKSB7XG4gIGxldCBhdXRob3JFbWFpbCA9IG9wdGlvbnMuYXV0aG9yRW1haWw7XG4gIGxldCBhdXRob3JVcmwgPSBvcHRpb25zLmF1dGhvclVybDtcbiAgaWYgKG9wdGlvbnMuYXV0aG9yQWRkcmVzcykge1xuICAgIGlmIChvcHRpb25zLmF1dGhvckVtYWlsICYmIG9wdGlvbnMuYXV0aG9yRW1haWwgIT09IG9wdGlvbnMuYXV0aG9yQWRkcmVzcykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBcImF1dGhvckVtYWlsIGlzIGRlcHJlY2F0ZWQgYW5kIGNhbm5vdCBiZSB1c2VkIGluIGNvbmp1bmN0aW9uIHdpdGggYXV0aG9yQWRkcmVzc1wiXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLmF1dGhvclVybCAmJiBvcHRpb25zLmF1dGhvclVybCAhPT0gb3B0aW9ucy5hdXRob3JBZGRyZXNzKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIFwiYXV0aG9yVXJsIGlzIGRlcHJlY2F0ZWQgYW5kIGNhbm5vdCBiZSB1c2VkIGluIGNvbmp1bmN0aW9uIHdpdGggYXV0aG9yQWRkcmVzcy5cIlxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAoRU1BSUxfUkVHRVgudGVzdChvcHRpb25zLmF1dGhvckFkZHJlc3MpKSB7XG4gICAgICBhdXRob3JFbWFpbCA9IG9wdGlvbnMuYXV0aG9yQWRkcmVzcztcbiAgICB9IGVsc2UgaWYgKFVSTF9SRUdFWC50ZXN0KG9wdGlvbnMuYXV0aG9yQWRkcmVzcykpIHtcbiAgICAgIGF1dGhvclVybCA9IG9wdGlvbnMuYXV0aG9yQWRkcmVzcztcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgYXV0aG9yQWRkcmVzcyBtdXN0IGJlIGVpdGhlciBhbiBlbWFpbCBhZGRyZXNzIG9yIGEgVVJMOiAke29wdGlvbnMuYXV0aG9yQWRkcmVzc31gXG4gICAgICApO1xuICAgIH1cbiAgfVxuICByZXR1cm4geyBhdXRob3JFbWFpbCwgYXV0aG9yVXJsIH07XG59XG4iXX0=