"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const PULL_REQUEST_REF = "${{ github.event.pull_request.head.ref }}";
const PULL_REQUEST_REPOSITORY = "${{ github.event.pull_request.head.repo.full_name }}";
const BUILD_JOBID = "build";
const SELF_MUTATION_STEP = "self_mutation";
const SELF_MUTATION_HAPPENED_OUTPUT = "self_mutation_happened";
const IS_FORK = "github.event.pull_request.head.repo.full_name != github.repository";
const NOT_FORK = `!(${IS_FORK})`;
const SELF_MUTATION_CONDITION = `needs.${BUILD_JOBID}.outputs.${SELF_MUTATION_HAPPENED_OUTPUT}`;
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.defaultRunners = ["ubuntu-latest"];
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(project);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = options.preBuildSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory = options.artifactsDirectory;
        const mutableBuilds = options.mutableBuild ?? true;
        this.workflow = new github_1.GithubWorkflow(github, "build");
        this.workflow.on(options.workflowTriggers ?? {
            pullRequest: {},
            workflowDispatch: {},
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
    }
    addBuildJob(options) {
        this.workflow.addJob(BUILD_JOBID, {
            runsOn: options.runsOn ?? this.defaultRunners,
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            steps: (() => this.renderBuildSteps()),
            outputs: {
                [SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: SELF_MUTATION_STEP,
                    outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        });
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        if (this.artifactsDirectory) {
            steps.push({
                name: "Download build artifacts",
                uses: "actions/download-artifact@v3",
                with: {
                    name: constants_1.BUILD_ARTIFACT_NAME,
                    path: this.artifactsDirectory,
                },
            });
        }
        steps.push(...(job.steps ?? []));
        this.workflow.addJob(id, {
            needs: [BUILD_JOBID],
            // only run if build did not self-mutate
            if: `! ${SELF_MUTATION_CONDITION}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options = {}) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            runsOn: options.runsOn,
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        const steps = [];
        if (options?.checkoutRepo) {
            steps.push({
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            });
        }
        if (options?.checkoutRepo &&
            options?.installDeps &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options?.tools,
            runsOn: options?.runsOn ?? this.defaultRunners,
            steps,
        });
    }
    addSelfMutationJob(options) {
        this.workflow.addJob("self-mutation", {
            runsOn: options.runsOn ?? this.defaultRunners,
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [BUILD_JOBID],
            if: `always() && ${SELF_MUTATION_CONDITION} && ${NOT_FORK}`,
            steps: [
                ...this.workflow.projenCredentials.setupSteps,
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: this.workflow.projenCredentials.tokenRef,
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                }),
                ...workflow_actions_1.WorkflowActions.setGitIdentity(this.gitIdentity),
                {
                    name: "Push changes",
                    run: [
                        "  git add .",
                        '  git commit -s -m "chore: self mutation"',
                        `  git push origin HEAD:${PULL_REQUEST_REF}`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps() {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    ref: PULL_REQUEST_REF,
                    repository: PULL_REQUEST_REPOSITORY,
                },
            },
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.createUploadGitPatch({
                stepId: SELF_MUTATION_STEP,
                outputName: SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Upload artifact",
                        uses: "actions/upload-artifact@v2.1.1",
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.artifactsDirectory,
                        },
                    },
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.60.2" };
//# sourceMappingURL=data:application/json;base64,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