"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ["aws-sdk"],
            ...options.bundlingOptions,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.60.2" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.60.2" };
/**
 * Node.js 10.x
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10");
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12");
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14");
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16");
//# sourceMappingURL=data:application/json;base64,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